/*
 * Decompiled with CFR 0.152.
 */
package dorkbox.network.dns.utils;

import dorkbox.network.dns.Name;
import dorkbox.network.dns.exceptions.RelativeNameException;
import dorkbox.network.dns.exceptions.TextParseException;
import dorkbox.network.dns.records.TTL;
import dorkbox.network.dns.utils.Address;
import dorkbox.network.dns.utils.base16;
import dorkbox.network.dns.utils.base32;
import dorkbox.util.Base64Fast;
import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.PushbackInputStream;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Arrays;

public class Tokenizer {
    private static final char[] VALID = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/".toCharArray();
    private static final int[] INTERNAL = new int[256];
    private static String delim;
    private static String quotes;
    public static final int EOF = 0;
    public static final int EOL = 1;
    public static final int WHITESPACE = 2;
    public static final int IDENTIFIER = 3;
    public static final int QUOTED_STRING = 4;
    public static final int COMMENT = 5;
    private PushbackInputStream is;
    private boolean ungottenToken;
    private int multiline;
    private boolean quoting;
    private String delimiters;
    private Token current;
    private StringBuilder sb;
    private boolean wantClose;
    private String filename;
    private int line;

    public Tokenizer(String s) {
        this(new ByteArrayInputStream(s.getBytes()));
    }

    public Tokenizer(InputStream is) {
        if (!(is instanceof BufferedInputStream)) {
            is = new BufferedInputStream(is);
        }
        this.is = new PushbackInputStream(is, 2);
        this.ungottenToken = false;
        this.multiline = 0;
        this.quoting = false;
        this.delimiters = delim;
        this.current = new Token();
        this.sb = new StringBuilder();
        this.filename = "<none>";
        this.line = 1;
    }

    public Tokenizer(File f) throws FileNotFoundException {
        this(new FileInputStream(f));
        this.wantClose = true;
        this.filename = f.getName();
    }

    public String getString() throws IOException {
        Token next = this.get();
        if (!next.isString()) {
            throw this.exception("expected a string");
        }
        return next.value;
    }

    public Token get() throws IOException {
        return this.get(false, false);
    }

    public Token get(boolean wantWhitespace, boolean wantComment) throws IOException {
        int c;
        int skipped;
        if (this.ungottenToken) {
            this.ungottenToken = false;
            if (this.current.type == 2) {
                if (wantWhitespace) {
                    return this.current;
                }
            } else if (this.current.type == 5) {
                if (wantComment) {
                    return this.current;
                }
            } else {
                if (this.current.type == 1) {
                    ++this.line;
                }
                return this.current;
            }
        }
        if ((skipped = this.skipWhitespace()) > 0 && wantWhitespace) {
            return this.current.set(2, null);
        }
        int type = 3;
        this.sb.setLength(0);
        while (true) {
            if ((c = this.getChar()) == -1 || this.delimiters.indexOf(c) != -1) {
                if (c == -1) {
                    if (this.quoting) {
                        throw this.exception("EOF in quoted string");
                    }
                    if (this.sb.length() == 0) {
                        return this.current.set(0, null);
                    }
                    return this.current.set(type, this.sb);
                }
                if (this.sb.length() == 0 && type != 4) {
                    if (c == 40) {
                        ++this.multiline;
                        this.skipWhitespace();
                        continue;
                    }
                    if (c == 41) {
                        if (this.multiline <= 0) {
                            throw this.exception("invalid close parenthesis");
                        }
                        --this.multiline;
                        this.skipWhitespace();
                        continue;
                    }
                    if (c == 34) {
                        if (!this.quoting) {
                            this.quoting = true;
                            this.delimiters = quotes;
                            type = 4;
                            continue;
                        }
                        this.quoting = false;
                        this.delimiters = delim;
                        this.skipWhitespace();
                        continue;
                    }
                    if (c == 10) {
                        return this.current.set(1, null);
                    }
                    if (c == 59) {
                        while ((c = this.getChar()) != 10 && c != -1) {
                            this.sb.append((char)c);
                        }
                        if (wantComment) {
                            this.ungetChar(c);
                            return this.current.set(5, this.sb);
                        }
                        if (c == -1 && type != 4) {
                            this.checkUnbalancedParens();
                            return this.current.set(0, null);
                        }
                        if (this.multiline > 0) {
                            this.skipWhitespace();
                            this.sb.setLength(0);
                            continue;
                        }
                        return this.current.set(1, null);
                    }
                    throw new IllegalStateException();
                }
                break;
            }
            if (c == 92) {
                c = this.getChar();
                if (c == -1) {
                    throw this.exception("unterminated escape sequence");
                }
                this.sb.append('\\');
            } else if (this.quoting && c == 10) {
                throw this.exception("newline in quoted string");
            }
            this.sb.append((char)c);
        }
        this.ungetChar(c);
        if (this.sb.length() == 0 && type != 4) {
            this.checkUnbalancedParens();
            return this.current.set(0, null);
        }
        return this.current.set(type, this.sb);
    }

    private int skipWhitespace() throws IOException {
        int skipped = 0;
        while (true) {
            int c;
            if ((c = this.getChar()) != 32 && c != 9 && (c != 10 || this.multiline <= 0)) {
                this.ungetChar(c);
                return skipped;
            }
            ++skipped;
        }
    }

    private int getChar() throws IOException {
        int c = this.is.read();
        if (c == 13) {
            int next = this.is.read();
            if (next != 10) {
                this.is.unread(next);
            }
            c = 10;
        }
        if (c == 10) {
            ++this.line;
        }
        return c;
    }

    private void ungetChar(int c) throws IOException {
        if (c == -1) {
            return;
        }
        this.is.unread(c);
        if (c == 10) {
            --this.line;
        }
    }

    private void checkUnbalancedParens() throws TextParseException {
        if (this.multiline > 0) {
            throw this.exception("unbalanced parentheses");
        }
    }

    public TextParseException exception(String s) {
        return new TokenizerException(this.filename, this.line, s);
    }

    public String getIdentifier() throws IOException {
        return this._getIdentifier("an identifier");
    }

    private String _getIdentifier(String expected) throws IOException {
        Token next = this.get();
        if (next.type != 3) {
            throw this.exception("expected " + expected);
        }
        return next.value;
    }

    public long getUInt32() throws IOException {
        long l = this.getLong();
        if (l < 0L || l > 0xFFFFFFFFL) {
            throw this.exception("expected an 32 bit unsigned integer");
        }
        return l;
    }

    public long getLong() throws IOException {
        String next = this._getIdentifier("an integer");
        if (!Character.isDigit(next.charAt(0))) {
            throw this.exception("expected an integer");
        }
        try {
            return Long.parseLong(next);
        }
        catch (NumberFormatException e) {
            throw this.exception("expected an integer");
        }
    }

    public int getUInt16() throws IOException {
        long l = this.getLong();
        if (l < 0L || l > 65535L) {
            throw this.exception("expected an 16 bit unsigned integer");
        }
        return (int)l;
    }

    public int getUInt8() throws IOException {
        long l = this.getLong();
        if (l < 0L || l > 255L) {
            throw this.exception("expected an 8 bit unsigned integer");
        }
        return (int)l;
    }

    public long getTTL() throws IOException {
        String next = this._getIdentifier("a TTL value");
        try {
            return TTL.parseTTL(next);
        }
        catch (NumberFormatException e) {
            throw this.exception("expected a TTL value");
        }
    }

    public long getTTLLike() throws IOException {
        String next = this._getIdentifier("a TTL-like value");
        try {
            return TTL.parse(next, false);
        }
        catch (NumberFormatException e) {
            throw this.exception("expected a TTL-like value");
        }
    }

    public Name getName(Name origin) throws IOException {
        String next = this._getIdentifier("a name");
        try {
            Name name = Name.fromString(next, origin);
            if (!name.isAbsolute()) {
                throw new RelativeNameException(name);
            }
            return name;
        }
        catch (TextParseException e) {
            throw this.exception(e.getMessage());
        }
    }

    public byte[] getAddressBytes(int family) throws IOException {
        String next = this._getIdentifier("an address");
        byte[] bytes = Address.toByteArray(next, family);
        if (bytes == null) {
            throw this.exception("Invalid address: " + next);
        }
        return bytes;
    }

    public InetAddress getAddress(int family) throws IOException {
        String next = this._getIdentifier("an address");
        try {
            return Address.getByAddress(next, family);
        }
        catch (UnknownHostException e) {
            throw this.exception(e.getMessage());
        }
    }

    public void getEOL() throws IOException {
        Token next = this.get();
        if (next.type != 1 && next.type != 0) {
            throw this.exception("expected EOL or EOF");
        }
    }

    public byte[] getBase64() throws IOException {
        return this.getBase64(false);
    }

    public byte[] getBase64(boolean required) throws IOException {
        char[] chars;
        String s = this.remainingStrings();
        if (s == null) {
            if (required) {
                throw this.exception("expected base64 encoded string");
            }
            return null;
        }
        for (char aChar : chars = s.toCharArray()) {
            if (aChar <= '\u0100' && INTERNAL[aChar] == 1) continue;
            throw new TextParseException("Invalid base64 character!");
        }
        byte[] array = Base64Fast.decode2(s);
        if (array == null) {
            throw this.exception("invalid base64 encoding");
        }
        return array;
    }

    private String remainingStrings() throws IOException {
        Token t;
        StringBuilder buffer = null;
        while ((t = this.get()).isString()) {
            if (buffer == null) {
                buffer = new StringBuilder();
            }
            buffer.append(t.value);
        }
        this.unget();
        if (buffer == null) {
            return null;
        }
        return buffer.toString();
    }

    public void unget() {
        if (this.ungottenToken) {
            throw new IllegalStateException("Cannot unget multiple tokens");
        }
        if (this.current.type == 1) {
            --this.line;
        }
        this.ungottenToken = true;
    }

    public byte[] getHex() throws IOException {
        return this.getHex(false);
    }

    public byte[] getHex(boolean required) throws IOException {
        String s = this.remainingStrings();
        if (s == null) {
            if (required) {
                throw this.exception("expected hex encoded string");
            }
            return null;
        }
        byte[] array = base16.fromString(s);
        if (array == null) {
            throw this.exception("invalid hex encoding");
        }
        return array;
    }

    public byte[] getHexString() throws IOException {
        String next = this._getIdentifier("a hex string");
        byte[] array = base16.fromString(next);
        if (array == null) {
            throw this.exception("invalid hex encoding");
        }
        return array;
    }

    public byte[] getBase32String(base32 b32) throws IOException {
        String next = this._getIdentifier("a base32 string");
        byte[] array = b32.fromString(next);
        if (array == null) {
            throw this.exception("invalid base32 encoding");
        }
        return array;
    }

    public void close() {
        if (this.wantClose) {
            try {
                this.is.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }

    protected void finalize() {
        this.close();
    }

    static {
        Arrays.fill(INTERNAL, -1);
        int iS = VALID.length;
        for (int i = 0; i < iS; ++i) {
            Tokenizer.INTERNAL[Tokenizer.VALID[i]] = 1;
        }
        Tokenizer.INTERNAL[61] = 1;
        delim = " \t\n;()\"";
        quotes = "\"";
    }

    public static class Token {
        public int type = -1;
        public String value = null;

        private Token() {
        }

        private Token set(int type, StringBuilder value) {
            if (type < 0) {
                throw new IllegalArgumentException();
            }
            this.type = type;
            this.value = value == null ? null : value.toString();
            return this;
        }

        public String toString() {
            switch (this.type) {
                case 0: {
                    return "<eof>";
                }
                case 1: {
                    return "<eol>";
                }
                case 2: {
                    return "<whitespace>";
                }
                case 3: {
                    return "<identifier: " + this.value + ">";
                }
                case 4: {
                    return "<quoted_string: " + this.value + ">";
                }
                case 5: {
                    return "<comment: " + this.value + ">";
                }
            }
            return "<unknown>";
        }

        public boolean isString() {
            return this.type == 3 || this.type == 4;
        }

        public boolean isEOL() {
            return this.type == 1 || this.type == 0;
        }
    }

    public static class TokenizerException
    extends TextParseException {
        String message;

        public TokenizerException(String filename, int line, String message) {
            super(filename + ":" + line + ": " + message);
            this.message = message;
        }

        public String getBaseMessage() {
            return this.message;
        }
    }
}

