/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.hadoop.gcsio;

import com.google.api.client.auth.oauth2.Credential;
import com.google.api.client.util.Clock;
import com.google.cloud.hadoop.gcsio.CacheSupplementedGoogleCloudStorage;
import com.google.cloud.hadoop.gcsio.CreateFileOptions;
import com.google.cloud.hadoop.gcsio.CreateObjectOptions;
import com.google.cloud.hadoop.gcsio.DirectoryListCache;
import com.google.cloud.hadoop.gcsio.FileInfo;
import com.google.cloud.hadoop.gcsio.FileSystemBackedDirectoryListCache;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorage;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageFileSystemOptions;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageImpl;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageItemInfo;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageReadOptions;
import com.google.cloud.hadoop.gcsio.InMemoryDirectoryListCache;
import com.google.cloud.hadoop.gcsio.LegacyPathCodec;
import com.google.cloud.hadoop.gcsio.PathCodec;
import com.google.cloud.hadoop.gcsio.PerformanceCachingGoogleCloudStorage;
import com.google.cloud.hadoop.gcsio.StorageResourceId;
import com.google.cloud.hadoop.gcsio.UpdatableItemInfo;
import com.google.cloud.hadoop.gcsio.UriEncodingPathCodec;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URI;
import java.nio.channels.SeekableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.FileAlreadyExistsException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GoogleCloudStorageFileSystem {
    public static final String SCHEME = "gs";
    public static final URI GCS_ROOT = URI.create("gs:/");
    public static final Logger LOG = LoggerFactory.getLogger(GoogleCloudStorageFileSystem.class);
    private GoogleCloudStorage gcs;
    private final PathCodec pathCodec;
    private final GoogleCloudStorageFileSystemOptions options;
    private ExecutorService updateTimestampsExecutor = new ThreadPoolExecutor(2, 2, 2L, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(1000), new ThreadFactoryBuilder().setNameFormat("gcsfs-timestamp-updates-%d").setDaemon(true).build());
    @VisibleForTesting
    static Comparator<URI> pathComparator = new Comparator<URI>(){

        @Override
        public int compare(URI a, URI b) {
            String as = a.toString();
            String bs = b.toString();
            return as.length() == bs.length() ? as.compareTo(bs) : Integer.compare(as.length(), bs.length());
        }
    };
    @VisibleForTesting
    static Comparator<FileInfo> fileInfoPathComparator = new Comparator<FileInfo>(){

        @Override
        public int compare(FileInfo file1, FileInfo file2) {
            return pathComparator.compare(file1.getPath(), file2.getPath());
        }
    };
    public static final PathCodec LEGACY_PATH_CODEC = new LegacyPathCodec();
    public static final PathCodec URI_ENCODED_PATH_CODEC = new UriEncodingPathCodec();

    public GoogleCloudStorageFileSystem(Credential credential, GoogleCloudStorageFileSystemOptions options) throws IOException {
        LOG.debug("GCSFS({})", (Object)options.getCloudStorageOptions().getAppName());
        options.throwIfNotValid();
        Preconditions.checkArgument((credential != null ? 1 : 0) != 0, (Object)"credential must not be null");
        this.options = options;
        this.gcs = new GoogleCloudStorageImpl(options.getCloudStorageOptions(), credential);
        this.pathCodec = options.getPathCodec();
        if (options.isMetadataCacheEnabled()) {
            DirectoryListCache resourceCache = null;
            switch (options.getCacheType()) {
                case IN_MEMORY: {
                    resourceCache = InMemoryDirectoryListCache.getInstance();
                    break;
                }
                case FILESYSTEM_BACKED: {
                    Preconditions.checkArgument((!Strings.isNullOrEmpty((String)options.getCacheBasePath()) ? 1 : 0) != 0, (Object)"When using FILESYSTEM_BACKED DirectoryListCache, cacheBasePath must not be null.");
                    resourceCache = new FileSystemBackedDirectoryListCache(options.getCacheBasePath());
                    break;
                }
                default: {
                    throw new IllegalArgumentException(String.format("DirectoryListCache.Type '%s' not supported.", new Object[]{options.getCacheType()}));
                }
            }
            resourceCache.getMutableConfig().setMaxEntryAgeMillis(options.getCacheMaxEntryAgeMillis());
            resourceCache.getMutableConfig().setMaxInfoAgeMillis(options.getCacheMaxInfoAgeMillis());
            this.gcs = new CacheSupplementedGoogleCloudStorage(this.gcs, resourceCache);
        }
        if (options.isPerformanceCacheEnabled()) {
            this.gcs = new PerformanceCachingGoogleCloudStorage(this.gcs, options.getPerformanceCacheOptions());
        }
    }

    public GoogleCloudStorageFileSystem(GoogleCloudStorage gcs) throws IOException {
        this(gcs, GoogleCloudStorageFileSystemOptions.newBuilder().setImmutableCloudStorageOptions(gcs.getOptions()).build());
    }

    public GoogleCloudStorageFileSystem(GoogleCloudStorage gcs, GoogleCloudStorageFileSystemOptions options) throws IOException {
        this.gcs = gcs;
        this.options = options;
        this.pathCodec = options.getPathCodec();
    }

    @VisibleForTesting
    void setUpdateTimestampsExecutor(ExecutorService executor) {
        this.updateTimestampsExecutor = executor;
    }

    public GoogleCloudStorageFileSystemOptions getOptions() {
        return this.options;
    }

    public WritableByteChannel create(URI path) throws IOException {
        LOG.debug("create({})", (Object)path);
        return this.create(path, CreateFileOptions.DEFAULT);
    }

    public static CreateObjectOptions objectOptionsFromFileOptions(CreateFileOptions options) {
        return new CreateObjectOptions(options.overwriteExisting(), options.getContentType(), options.getAttributes());
    }

    public WritableByteChannel create(URI path, CreateFileOptions options) throws IOException {
        URI parentPath;
        URI dirPath;
        LOG.debug("create({})", (Object)path);
        Preconditions.checkNotNull((Object)path);
        if (FileInfo.isDirectoryPath(path)) {
            throw new IOException(String.format("Cannot create a file whose name looks like a directory. Got '%s'", path));
        }
        if (options.checkNoDirectoryConflict() && this.exists(dirPath = FileInfo.convertToDirectoryPath(this.pathCodec, path))) {
            throw new FileAlreadyExistsException("A directory with that name exists: " + path);
        }
        if (options.ensureParentDirectoriesExist() && (parentPath = this.getParentPath(path)) != null) {
            this.mkdirs(parentPath);
        }
        return this.createInternal(path, options);
    }

    WritableByteChannel createInternal(URI path, CreateFileOptions options) throws IOException {
        StorageResourceId resourceId = this.pathCodec.validatePathAndGetId(path, false);
        if (options.getExistingGenerationId() != -1L) {
            resourceId = new StorageResourceId(resourceId.getBucketName(), resourceId.getObjectName(), options.getExistingGenerationId());
        }
        WritableByteChannel channel = this.gcs.create(resourceId, GoogleCloudStorageFileSystem.objectOptionsFromFileOptions(options));
        this.tryUpdateTimestampsForParentDirectories((List<URI>)ImmutableList.of((Object)path), (List<URI>)ImmutableList.of());
        return channel;
    }

    public SeekableByteChannel open(URI path) throws IOException {
        return this.open(path, GoogleCloudStorageReadOptions.DEFAULT);
    }

    public SeekableByteChannel open(URI path, GoogleCloudStorageReadOptions readOptions) throws IOException {
        LOG.debug("open({}, {})", (Object)path, (Object)readOptions);
        Preconditions.checkNotNull((Object)path);
        Preconditions.checkArgument((!FileInfo.isDirectoryPath(path) ? 1 : 0) != 0, (Object)("Cannot open a directory for reading: " + path));
        StorageResourceId resourceId = this.pathCodec.validatePathAndGetId(path, false);
        return this.gcs.open(resourceId, readOptions);
    }

    public void delete(URI path, boolean recursive) throws IOException {
        LOG.debug("delete({}, {})", (Object)path, (Object)recursive);
        Preconditions.checkNotNull((Object)path);
        Preconditions.checkArgument((!path.equals(GCS_ROOT) ? 1 : 0) != 0, (Object)"Cannot delete root path.");
        FileInfo fileInfo = this.getFileInfo(path);
        if (!fileInfo.exists()) {
            throw GoogleCloudStorageFileSystem.getFileNotFoundException(path);
        }
        ArrayList<URI> itemsToDelete = new ArrayList<URI>();
        ArrayList<URI> bucketsToDelete = new ArrayList<URI>();
        if (fileInfo.isDirectory()) {
            List<URI> subpaths = this.listFileNames(fileInfo, recursive);
            if (recursive) {
                itemsToDelete.addAll(subpaths);
            } else if (subpaths.size() > 0) {
                throw new DirectoryNotEmptyException("Cannot delete a non-empty directory.");
            }
        }
        if (fileInfo.getItemInfo().isBucket()) {
            bucketsToDelete.add(fileInfo.getPath());
        } else {
            itemsToDelete.add(fileInfo.getPath());
        }
        this.deleteInternal(itemsToDelete, bucketsToDelete);
    }

    private void deleteInternal(List<URI> paths, List<URI> bucketPaths) throws IOException {
        StorageResourceId resourceId;
        Collections.sort(paths, pathComparator);
        Collections.reverse(paths);
        if (paths.size() > 0) {
            ArrayList<StorageResourceId> objectsToDelete = new ArrayList<StorageResourceId>();
            for (URI path : paths) {
                resourceId = this.pathCodec.validatePathAndGetId(path, false);
                objectsToDelete.add(resourceId);
            }
            this.gcs.deleteObjects(objectsToDelete);
            this.tryUpdateTimestampsForParentDirectories(paths, paths);
        }
        if (bucketPaths.size() > 0) {
            ArrayList<String> bucketsToDelete = new ArrayList<String>();
            for (URI path : bucketPaths) {
                resourceId = this.pathCodec.validatePathAndGetId(path, true);
                this.gcs.waitForBucketEmpty(resourceId.getBucketName());
                bucketsToDelete.add(resourceId.getBucketName());
            }
            if (this.options.enableBucketDelete()) {
                this.gcs.deleteBuckets(bucketsToDelete);
            } else {
                LOG.info("Skipping deletion of buckets because enableBucketDelete is false: {}", bucketsToDelete);
            }
        }
    }

    public boolean exists(URI path) throws IOException {
        LOG.debug("exists({})", (Object)path);
        return this.getFileInfo(path).exists();
    }

    public void repairDirs(List<URI> exactDirPaths) throws IOException {
        LOG.debug("repairDirs({})", exactDirPaths);
        ArrayList<StorageResourceId> dirsToCreate = new ArrayList<StorageResourceId>();
        for (URI dirUri : exactDirPaths) {
            StorageResourceId resourceId = this.pathCodec.validatePathAndGetId(dirUri, true);
            if (!resourceId.isStorageObject()) continue;
            resourceId = FileInfo.convertToDirectoryPath(resourceId);
            dirsToCreate.add(resourceId);
        }
        if (dirsToCreate.isEmpty()) {
            return;
        }
        if (dirsToCreate.size() == 1) {
            this.gcs.createEmptyObject((StorageResourceId)dirsToCreate.get(0));
        } else if (dirsToCreate.size() > 1) {
            this.gcs.createEmptyObjects(dirsToCreate);
        }
        LOG.warn("Successfully repaired {} directories.", (Object)dirsToCreate.size());
    }

    public void mkdirs(URI path) throws IOException {
        LOG.debug("mkdirs({})", (Object)path);
        Preconditions.checkNotNull((Object)path);
        if (path.equals(GCS_ROOT)) {
            return;
        }
        StorageResourceId resourceId = this.pathCodec.validatePathAndGetId(path, true);
        if (resourceId.isStorageObject()) {
            resourceId = FileInfo.convertToDirectoryPath(resourceId);
            path = this.pathCodec.getPath(resourceId.getBucketName(), resourceId.getObjectName(), false);
        }
        ArrayList<URI> subDirPaths = new ArrayList<URI>();
        List<String> subdirs = GoogleCloudStorageFileSystem.getSubDirs(resourceId.getObjectName());
        for (String subdir : subdirs) {
            URI subPath = this.pathCodec.getPath(resourceId.getBucketName(), subdir, true);
            subDirPaths.add(subPath);
            LOG.debug("mkdirs: sub-path: {}", (Object)subPath);
            if (Strings.isNullOrEmpty((String)subdir)) continue;
            URI uRI = this.pathCodec.getPath(resourceId.getBucketName(), subdir.substring(0, subdir.length() - 1), true);
            subDirPaths.add(uRI);
            LOG.debug("mkdirs: sub-path: {}", (Object)uRI);
        }
        URI bucketPath = this.pathCodec.getPath(resourceId.getBucketName(), null, true);
        subDirPaths.add(bucketPath);
        LOG.debug("mkdirs: sub-path: {}", (Object)bucketPath);
        List<FileInfo> subDirInfos = this.getFileInfos(subDirPaths);
        for (FileInfo fileInfo : subDirInfos) {
            if (!fileInfo.exists() || fileInfo.isDirectory()) continue;
            throw new FileAlreadyExistsException("Cannot create directories because of existing file: " + fileInfo.getPath());
        }
        Collections.sort(subDirInfos, new Comparator<FileInfo>(){

            @Override
            public int compare(FileInfo file1, FileInfo file2) {
                return Integer.valueOf(file1.getPath().toString().length()).compareTo(file2.getPath().toString().length());
            }
        });
        ArrayList<StorageResourceId> dirsToCreate = new ArrayList<StorageResourceId>();
        for (FileInfo fileInfo : subDirInfos) {
            if (!fileInfo.isDirectory() || fileInfo.exists()) continue;
            StorageResourceId dirId = fileInfo.getItemInfo().getResourceId();
            Preconditions.checkArgument((!dirId.isRoot() ? 1 : 0) != 0, (Object)"Cannot create root directory.");
            if (dirId.isBucket()) {
                this.gcs.create(dirId.getBucketName());
                continue;
            }
            dirId = FileInfo.convertToDirectoryPath(dirId);
            dirsToCreate.add(dirId);
        }
        if (dirsToCreate.size() == 1) {
            this.gcs.createEmptyObject((StorageResourceId)dirsToCreate.get(0));
        } else if (dirsToCreate.size() > 1) {
            this.gcs.createEmptyObjects(dirsToCreate);
        }
        List list = Lists.transform(dirsToCreate, (Function)new Function<StorageResourceId, URI>(){

            public URI apply(StorageResourceId resourceId) {
                return GoogleCloudStorageFileSystem.this.pathCodec.getPath(resourceId.getBucketName(), resourceId.getObjectName(), false);
            }
        });
        this.tryUpdateTimestampsForParentDirectories(list, list);
    }

    public void rename(URI src, URI dst) throws IOException {
        LOG.debug("rename({}, {})", (Object)src, (Object)dst);
        Preconditions.checkNotNull((Object)src);
        Preconditions.checkNotNull((Object)dst);
        Preconditions.checkArgument((!src.equals(GCS_ROOT) ? 1 : 0) != 0, (Object)"Root path cannot be renamed.");
        String srcItemName = this.getItemName(src);
        URI dstParent = this.getParentPath(dst);
        ArrayList<URI> paths = new ArrayList<URI>();
        paths.add(src);
        paths.add(dst);
        if (dstParent != null) {
            paths.add(dstParent);
        }
        List<FileInfo> fileInfo = this.getFileInfos(paths);
        FileInfo srcInfo = fileInfo.get(0);
        FileInfo dstInfo = fileInfo.get(1);
        FileInfo dstParentInfo = null;
        if (dstParent != null) {
            dstParentInfo = fileInfo.get(2);
        }
        src = srcInfo.getPath();
        dst = dstInfo.getPath();
        if (!srcInfo.exists()) {
            throw GoogleCloudStorageFileSystem.getFileNotFoundException(src);
        }
        if (!srcInfo.isDirectory() && dst.equals(GCS_ROOT)) {
            throw new IOException("A file cannot be created in root.");
        }
        if (dstInfo.exists() && !dstInfo.isDirectory()) {
            throw new IOException("Cannot overwrite existing file: " + dst);
        }
        if (dstParentInfo != null && !dstParentInfo.exists()) {
            throw new IOException("Cannot rename because path does not exist: " + dstParent);
        }
        if (srcInfo.isDirectory()) {
            if (!dstInfo.isDirectory()) {
                dst = FileInfo.convertToDirectoryPath(this.pathCodec, dst);
                dstInfo = this.getFileInfo(dst);
            }
            if (dstInfo.exists()) {
                dst = dst.equals(GCS_ROOT) ? this.pathCodec.getPath(srcItemName, null, true) : dst.resolve(srcItemName);
            }
        } else if (dstInfo.isDirectory()) {
            if (!dstInfo.exists()) {
                throw new IOException("Cannot rename because path does not exist: " + dstInfo.getPath());
            }
            dst = dst.resolve(srcItemName);
        } else {
            URI dstDir = FileInfo.convertToDirectoryPath(this.pathCodec, dst);
            FileInfo dstDirInfo = this.getFileInfo(dstDir);
            if (dstDirInfo.exists()) {
                dst = dstDir.resolve(srcItemName);
            }
        }
        this.renameInternal(srcInfo, dst);
    }

    public void compose(List<URI> sources, URI destination, String contentType) throws IOException {
        StorageResourceId destResource = StorageResourceId.fromObjectName(destination.toString());
        List sourceObjects = Lists.transform(sources, (Function)new Function<URI, String>(){

            public String apply(URI uri) {
                return StorageResourceId.fromObjectName(uri.toString()).getObjectName();
            }
        });
        this.gcs.compose(destResource.getBucketName(), sourceObjects, destResource.getObjectName(), contentType);
    }

    private void renameInternal(FileInfo srcInfo, URI dst) throws IOException {
        List<Object> srcItemNames = new ArrayList<URI>();
        HashMap<URI, URI> dstItemNames = new HashMap<URI, URI>();
        if (srcInfo.isDirectory()) {
            srcItemNames = this.listFileNames(srcInfo, true);
            Collections.sort(srcItemNames, pathComparator);
            dst = FileInfo.convertToDirectoryPath(this.pathCodec, dst);
            this.mkdir(dst);
            String prefix = srcInfo.getPath().toString();
            for (URI uRI : srcItemNames) {
                String relativeItemName = uRI.toString().substring(prefix.length());
                URI dstItemName = dst.resolve(relativeItemName);
                dstItemNames.put(uRI, dstItemName);
            }
        } else {
            srcItemNames.add(srcInfo.getPath());
            dstItemNames.put(srcInfo.getPath(), dst);
        }
        Preconditions.checkState((srcItemNames.size() == dstItemNames.size() ? 1 : 0) != 0, (String)"srcItemNames.size() != dstItemNames.size(), '%s' vs '%s'", (Object[])new Object[]{srcItemNames, dstItemNames});
        if (srcItemNames.size() > 0) {
            String srcBucketName = null;
            String dstBucketName = null;
            ArrayList<String> arrayList = new ArrayList<String>();
            ArrayList<String> dstObjectNames = new ArrayList<String>();
            for (URI uRI : srcItemNames) {
                StorageResourceId resourceId = this.pathCodec.validatePathAndGetId(uRI, true);
                srcBucketName = resourceId.getBucketName();
                String srcObjectName = resourceId.getObjectName();
                arrayList.add(srcObjectName);
                resourceId = this.pathCodec.validatePathAndGetId((URI)dstItemNames.get(uRI), true);
                dstBucketName = resourceId.getBucketName();
                String dstObjectName = resourceId.getObjectName();
                dstObjectNames.add(dstObjectName);
            }
            this.gcs.copy(srcBucketName, arrayList, dstBucketName, dstObjectNames);
            ArrayList<URI> destinationUris = new ArrayList<URI>(dstObjectNames.size());
            for (String dstObjectName : dstObjectNames) {
                destinationUris.add(this.pathCodec.getPath(dstBucketName, dstObjectName, false));
            }
            this.tryUpdateTimestampsForParentDirectories(destinationUris, destinationUris);
        }
        ArrayList<URI> bucketsToDelete = new ArrayList<URI>();
        if (srcInfo.isDirectory()) {
            if (srcInfo.getItemInfo().isBucket()) {
                bucketsToDelete.add(srcInfo.getPath());
            } else {
                srcItemNames.add(srcInfo.getPath());
            }
        }
        this.deleteInternal(srcItemNames, bucketsToDelete);
    }

    public List<URI> listFileNames(FileInfo fileInfo) throws IOException {
        return this.listFileNames(fileInfo, false);
    }

    public List<URI> listFileNames(FileInfo fileInfo, boolean recursive) throws IOException {
        Preconditions.checkNotNull((Object)fileInfo);
        URI path = fileInfo.getPath();
        LOG.debug("listFileNames({})", (Object)path);
        ArrayList<URI> paths = new ArrayList<URI>();
        if (fileInfo.isDirectory()) {
            if (fileInfo.exists()) {
                if (fileInfo.isGlobalRoot()) {
                    List<String> childNames = this.gcs.listBucketNames();
                    for (String childName : childNames) {
                        URI childPath = this.pathCodec.getPath(childName, null, true);
                        paths.add(childPath);
                        LOG.debug("listFileNames: added: {}", (Object)childPath);
                    }
                } else {
                    String delimiter = recursive ? null : "/";
                    List<String> childNames = this.gcs.listObjectNames(fileInfo.getItemInfo().getBucketName(), fileInfo.getItemInfo().getObjectName(), delimiter);
                    for (String childName : childNames) {
                        URI childPath = this.pathCodec.getPath(fileInfo.getItemInfo().getBucketName(), childName, false);
                        paths.add(childPath);
                        LOG.debug("listFileNames: added: {}", (Object)childPath);
                    }
                }
            }
        } else {
            paths.add(path);
            LOG.debug("listFileNames: added single original path since !isDirectory(): {}", (Object)path);
        }
        return paths;
    }

    public boolean repairPossibleImplicitDirectory(URI path) throws IOException {
        LOG.debug("repairPossibleImplicitDirectory({})", (Object)path);
        Preconditions.checkNotNull((Object)path);
        FileInfo pathInfo = this.getFileInfo(path);
        pathInfo = this.repairPossibleImplicitDirectory(pathInfo);
        if (pathInfo.exists()) {
            LOG.debug("Successfully repaired path '{}'", (Object)path);
            return true;
        }
        LOG.debug("Repair claimed to succeed, but somehow failed for path '{}'", (Object)path);
        return false;
    }

    private FileInfo repairPossibleImplicitDirectory(FileInfo pathInfo) throws IOException {
        if (pathInfo.exists()) {
            return pathInfo;
        }
        if (pathInfo.isGlobalRoot() || pathInfo.getItemInfo().isBucket() || pathInfo.getItemInfo().getObjectName().equals("/")) {
            return pathInfo;
        }
        try {
            this.gcs.listObjectInfo(pathInfo.getItemInfo().getBucketName(), FileInfo.convertToFilePath(pathInfo.getItemInfo().getObjectName()), "/");
        }
        catch (IOException ioe) {
            LOG.error("Got exception trying to listObjectInfo on " + pathInfo, (Throwable)ioe);
        }
        pathInfo = this.getFileInfo(pathInfo.getPath());
        return pathInfo;
    }

    public List<FileInfo> listAllFileInfoForPrefix(URI prefix) throws IOException {
        LOG.debug("listAllFileInfoForPrefix({})", (Object)prefix);
        Preconditions.checkNotNull((Object)prefix);
        StorageResourceId prefixId = this.pathCodec.validatePathAndGetId(prefix, true);
        Preconditions.checkState((!prefixId.isRoot() ? 1 : 0) != 0, (String)"Prefix must not be global root, got '%s'", (Object[])new Object[]{prefix});
        List<GoogleCloudStorageItemInfo> itemInfos = this.gcs.listObjectInfo(prefixId.getBucketName(), prefixId.getObjectName(), null);
        List<FileInfo> fileInfos = FileInfo.fromItemInfos(this.pathCodec, itemInfos);
        Collections.sort(fileInfos, fileInfoPathComparator);
        return fileInfos;
    }

    public List<FileInfo> listFileInfo(URI path, boolean enableAutoRepair) throws IOException {
        LOG.debug("listFileInfo({}, {})", (Object)path, (Object)enableAutoRepair);
        Preconditions.checkNotNull((Object)path);
        URI dirPath = FileInfo.convertToDirectoryPath(this.pathCodec, path);
        List<FileInfo> baseAndDirInfos = this.getFileInfosRaw((List<URI>)ImmutableList.of((Object)path, (Object)dirPath));
        Preconditions.checkState((baseAndDirInfos.size() == 2 ? 1 : 0) != 0, (String)"Expected baseAndDirInfos.size() == 2, got %s", (Object[])new Object[]{baseAndDirInfos.size()});
        if (!baseAndDirInfos.get(0).isDirectory() && baseAndDirInfos.get(0).exists()) {
            ArrayList<FileInfo> listedInfo = new ArrayList<FileInfo>();
            listedInfo.add(baseAndDirInfos.get(0));
            return listedInfo;
        }
        FileInfo dirInfo = baseAndDirInfos.get(1);
        if (!dirInfo.exists()) {
            if (enableAutoRepair) {
                dirInfo = this.repairPossibleImplicitDirectory(dirInfo);
            } else if (this.options.getCloudStorageOptions().isInferImplicitDirectoriesEnabled()) {
                StorageResourceId dirId = dirInfo.getItemInfo().getResourceId();
                if (!dirInfo.isDirectory()) {
                    dirId = FileInfo.convertToDirectoryPath(dirId);
                }
                dirInfo = FileInfo.fromItemInfo(this.pathCodec, this.getInferredItemInfo(dirId));
            }
        }
        if (!dirInfo.exists()) {
            throw GoogleCloudStorageFileSystem.getFileNotFoundException(path);
        }
        List<GoogleCloudStorageItemInfo> itemInfos = dirInfo.isGlobalRoot() ? this.gcs.listBucketInfo() : this.gcs.listObjectInfo(dirInfo.getItemInfo().getBucketName(), dirInfo.getItemInfo().getObjectName(), "/");
        List<FileInfo> fileInfos = FileInfo.fromItemInfos(this.pathCodec, itemInfos);
        Collections.sort(fileInfos, fileInfoPathComparator);
        return fileInfos;
    }

    public FileInfo getFileInfo(URI path) throws IOException {
        GoogleCloudStorageItemInfo newItemInfo;
        StorageResourceId newResourceId;
        LOG.debug("getFileInfo({})", (Object)path);
        Preconditions.checkArgument((path != null ? 1 : 0) != 0, (Object)"path must not be null");
        StorageResourceId resourceId = this.pathCodec.validatePathAndGetId(path, true);
        GoogleCloudStorageItemInfo itemInfo = this.gcs.getItemInfo(resourceId);
        if (!itemInfo.exists() && !FileInfo.isDirectory(itemInfo)) {
            newResourceId = FileInfo.convertToDirectoryPath(resourceId);
            LOG.debug("getFileInfo({}) : not found. trying: {}", (Object)path, (Object)newResourceId.toString());
            newItemInfo = this.gcs.getItemInfo(newResourceId);
            if (newItemInfo.exists()) {
                LOG.debug("getFileInfo: swapping not-found info: %s for converted info: %s", (Object)itemInfo, (Object)newItemInfo);
                itemInfo = newItemInfo;
                resourceId = newResourceId;
            }
        }
        if (!itemInfo.exists() && this.options.getCloudStorageOptions().isInferImplicitDirectoriesEnabled() && !itemInfo.isRoot() && !itemInfo.isBucket()) {
            newResourceId = resourceId;
            if (!FileInfo.isDirectory(itemInfo)) {
                newResourceId = FileInfo.convertToDirectoryPath(resourceId);
            }
            LOG.debug("getFileInfo({}) : still not found, trying inferred: {}", (Object)path, (Object)newResourceId.toString());
            newItemInfo = this.getInferredItemInfo(resourceId);
            if (newItemInfo.exists()) {
                LOG.debug("getFileInfo: swapping not-found info: %s for inferred info: %s", (Object)itemInfo, (Object)newItemInfo);
                itemInfo = newItemInfo;
                resourceId = newResourceId;
            }
        }
        FileInfo fileInfo = FileInfo.fromItemInfo(this.pathCodec, itemInfo);
        LOG.debug("getFileInfo: {}", (Object)fileInfo);
        return fileInfo;
    }

    public List<FileInfo> getFileInfos(List<URI> paths) throws IOException {
        LOG.debug("getFileInfos(list)");
        Preconditions.checkArgument((paths != null ? 1 : 0) != 0, (Object)"paths must not be null");
        ArrayList<StorageResourceId> resourceIdsForPaths = new ArrayList<StorageResourceId>();
        for (URI path : paths) {
            resourceIdsForPaths.add(this.pathCodec.validatePathAndGetId(path, true));
        }
        List<GoogleCloudStorageItemInfo> itemInfos = this.gcs.getItemInfos(resourceIdsForPaths);
        HashMap<StorageResourceId, Integer> convertedIdsToIndex = new HashMap<StorageResourceId, Integer>();
        for (int i = 0; i < itemInfos.size(); ++i) {
            if (itemInfos.get(i).exists() || FileInfo.isDirectory(itemInfos.get(i))) continue;
            StorageResourceId convertedId = FileInfo.convertToDirectoryPath(itemInfos.get(i).getResourceId());
            LOG.debug("getFileInfos({}) : not found. trying: {}", (Object)itemInfos.get(i).getResourceId(), (Object)convertedId);
            convertedIdsToIndex.put(convertedId, i);
        }
        if (!convertedIdsToIndex.isEmpty()) {
            ArrayList<StorageResourceId> convertedResourceIds = new ArrayList<StorageResourceId>(convertedIdsToIndex.keySet());
            List<GoogleCloudStorageItemInfo> convertedInfos = this.gcs.getItemInfos(convertedResourceIds);
            for (int i = 0; i < convertedResourceIds.size(); ++i) {
                if (!convertedInfos.get(i).exists()) continue;
                int replaceIndex = (Integer)convertedIdsToIndex.get(convertedResourceIds.get(i));
                LOG.debug("getFileInfos: swapping not-found info: {} for converted info: {}", (Object)itemInfos.get(replaceIndex), (Object)convertedInfos.get(i));
                itemInfos.set(replaceIndex, convertedInfos.get(i));
            }
        }
        if (this.options.getCloudStorageOptions().isInferImplicitDirectoriesEnabled()) {
            HashMap<StorageResourceId, Integer> inferredIdsToIndex = new HashMap<StorageResourceId, Integer>();
            for (int i = 0; i < itemInfos.size(); ++i) {
                if (itemInfos.get(i).exists()) continue;
                StorageResourceId inferredId = itemInfos.get(i).getResourceId();
                if (!FileInfo.isDirectory(itemInfos.get(i))) {
                    inferredId = FileInfo.convertToDirectoryPath(inferredId);
                }
                LOG.debug("getFileInfos({}) : still not found, trying inferred: {}", (Object)itemInfos.get(i).getResourceId(), (Object)inferredId);
                inferredIdsToIndex.put(inferredId, i);
            }
            if (!inferredIdsToIndex.isEmpty()) {
                ArrayList<StorageResourceId> inferredResourceIds = new ArrayList<StorageResourceId>(inferredIdsToIndex.keySet());
                List<GoogleCloudStorageItemInfo> inferredInfos = this.getInferredItemInfos(inferredResourceIds);
                for (int i = 0; i < inferredResourceIds.size(); ++i) {
                    if (!inferredInfos.get(i).exists()) continue;
                    int replaceIndex = (Integer)inferredIdsToIndex.get(inferredResourceIds.get(i));
                    LOG.debug("getFileInfos: swapping not-found info: %s for inferred info: %s", (Object)itemInfos.get(replaceIndex), (Object)inferredInfos.get(i));
                    itemInfos.set(replaceIndex, inferredInfos.get(i));
                }
            }
        }
        return FileInfo.fromItemInfos(this.pathCodec, itemInfos);
    }

    private List<FileInfo> getFileInfosRaw(List<URI> paths) throws IOException {
        LOG.debug("getFileInfosRaw({})", (Object)paths.toString());
        Preconditions.checkArgument((paths != null ? 1 : 0) != 0, (Object)"paths must not be null");
        ArrayList<StorageResourceId> resourceIdsForPaths = new ArrayList<StorageResourceId>();
        for (URI path : paths) {
            resourceIdsForPaths.add(this.pathCodec.validatePathAndGetId(path, true));
        }
        List<GoogleCloudStorageItemInfo> itemInfos = this.gcs.getItemInfos(resourceIdsForPaths);
        return FileInfo.fromItemInfos(this.pathCodec, itemInfos);
    }

    private GoogleCloudStorageItemInfo getInferredItemInfo(StorageResourceId dirId) throws IOException {
        String delimiter;
        String objectNamePrefix;
        if (dirId.isRoot() || dirId.isBucket()) {
            return GoogleCloudStorageImpl.createItemInfoForNotFound(dirId);
        }
        StorageResourceId bucketId = new StorageResourceId(dirId.getBucketName());
        if (!this.gcs.getItemInfo(bucketId).exists()) {
            return GoogleCloudStorageImpl.createItemInfoForNotFound(dirId);
        }
        String bucketName = (dirId = FileInfo.convertToDirectoryPath(dirId)).getBucketName();
        List<String> objectNames = this.gcs.listObjectNames(bucketName, objectNamePrefix = dirId.getObjectName(), delimiter = "/", 1L);
        if (objectNames.size() > 0) {
            return GoogleCloudStorageImpl.createItemInfoForInferredDirectory(dirId);
        }
        return GoogleCloudStorageImpl.createItemInfoForNotFound(dirId);
    }

    private List<GoogleCloudStorageItemInfo> getInferredItemInfos(List<StorageResourceId> resourceIds) throws IOException {
        ArrayList<GoogleCloudStorageItemInfo> itemInfos = new ArrayList<GoogleCloudStorageItemInfo>();
        for (int i = 0; i < resourceIds.size(); ++i) {
            itemInfos.add(this.getInferredItemInfo(resourceIds.get(i)));
        }
        return itemInfos;
    }

    public void close() {
        if (this.gcs != null) {
            LOG.debug("close()");
            try {
                this.gcs.close();
            }
            finally {
                this.gcs = null;
            }
        }
        if (this.updateTimestampsExecutor != null) {
            this.updateTimestampsExecutor.shutdown();
            try {
                if (!this.updateTimestampsExecutor.awaitTermination(10L, TimeUnit.SECONDS)) {
                    LOG.warn("Forcibly shutting down timestamp update threadpool.");
                    this.updateTimestampsExecutor.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                LOG.warn("Interrupted awaiting timestamp update threadpool.");
            }
            this.updateTimestampsExecutor = null;
        }
    }

    @VisibleForTesting
    public void mkdir(URI path) throws IOException {
        LOG.debug("mkdir({})", (Object)path);
        Preconditions.checkNotNull((Object)path);
        Preconditions.checkArgument((!path.equals(GCS_ROOT) ? 1 : 0) != 0, (Object)"Cannot create root directory.");
        StorageResourceId resourceId = this.pathCodec.validatePathAndGetId(path, true);
        if (resourceId.isBucket()) {
            this.gcs.create(resourceId.getBucketName());
            return;
        }
        resourceId = FileInfo.convertToDirectoryPath(resourceId);
        this.gcs.createEmptyObject(resourceId);
        this.tryUpdateTimestampsForParentDirectories((List<URI>)ImmutableList.of((Object)path), (List<URI>)ImmutableList.of());
    }

    protected void updateTimestampsForParentDirectories(List<URI> modifiedObjects, List<URI> excludedParents) throws IOException {
        LOG.debug("updateTimestampsForParentDirectories({}, {})", modifiedObjects, excludedParents);
        GoogleCloudStorageFileSystemOptions.TimestampUpdatePredicate updatePredicate = this.options.getShouldIncludeInTimestampUpdatesPredicate();
        HashSet<URI> excludedParentPathsSet = new HashSet<URI>(excludedParents);
        HashSet<URI> parentUrisToUpdate = new HashSet<URI>(modifiedObjects.size());
        for (URI modifiedObjectUri : modifiedObjects) {
            URI parentPathUri = this.getParentPath(modifiedObjectUri);
            if (excludedParentPathsSet.contains(parentPathUri) || !updatePredicate.shouldUpdateTimestamp(parentPathUri)) continue;
            parentUrisToUpdate.add(parentPathUri);
        }
        HashMap<String, byte[]> modificationAttributes = new HashMap<String, byte[]>();
        FileInfo.addModificationTimeToAttributes(modificationAttributes, Clock.SYSTEM);
        ArrayList<UpdatableItemInfo> itemUpdates = new ArrayList<UpdatableItemInfo>(parentUrisToUpdate.size());
        for (URI parentUri : parentUrisToUpdate) {
            StorageResourceId resourceId = this.pathCodec.validatePathAndGetId(parentUri, true);
            if (resourceId.isBucket() || resourceId.isRoot()) continue;
            itemUpdates.add(new UpdatableItemInfo(resourceId, modificationAttributes));
        }
        if (!itemUpdates.isEmpty()) {
            this.gcs.updateItems(itemUpdates);
        } else {
            LOG.debug("All paths were excluded from directory timestamp updating.");
        }
    }

    protected void tryUpdateTimestampsForParentDirectories(final List<URI> modifiedObjects, final List<URI> excludedParents) {
        LOG.debug("tryUpdateTimestampsForParentDirectories({}, {})", modifiedObjects, excludedParents);
        try {
            this.updateTimestampsExecutor.submit(new Runnable(){

                @Override
                public void run() {
                    try {
                        GoogleCloudStorageFileSystem.this.updateTimestampsForParentDirectories(modifiedObjects, excludedParents);
                    }
                    catch (IOException ioe) {
                        LOG.debug("Exception caught when trying to update parent directory timestamps.", (Throwable)ioe);
                    }
                }
            });
        }
        catch (RejectedExecutionException ree) {
            LOG.debug("Exhausted threadpool and queue space while updating parent timestamps", (Throwable)ree);
        }
    }

    static List<String> getSubDirs(String objectName) {
        ArrayList<String> subdirs = new ArrayList<String>();
        if (!Strings.isNullOrEmpty((String)objectName)) {
            int index;
            int currentIndex = 0;
            while (currentIndex < objectName.length() && (index = objectName.indexOf(47, currentIndex)) >= 0) {
                subdirs.add(objectName.substring(0, index + 1));
                currentIndex = index + 1;
            }
        }
        return subdirs;
    }

    static String validateBucketName(String bucketName) {
        if (Strings.isNullOrEmpty((String)(bucketName = FileInfo.convertToFilePath(bucketName)))) {
            throw new IllegalArgumentException("Google Cloud Storage bucket name cannot be empty.");
        }
        if (bucketName.indexOf(47) >= 0) {
            throw new IllegalArgumentException("Google Cloud Storage bucket name must not contain '/' character.");
        }
        return bucketName;
    }

    static String validateObjectName(String objectName, boolean allowEmptyObjectName) {
        LOG.debug("validateObjectName('{}', {})", (Object)objectName, (Object)allowEmptyObjectName);
        String objectNameMessage = "Google Cloud Storage path must include non-empty object name.";
        if (objectName == null) {
            if (allowEmptyObjectName) {
                objectName = "";
            } else {
                throw new IllegalArgumentException(objectNameMessage);
            }
        }
        for (int i = 0; i < objectName.length() - 1; ++i) {
            if (objectName.charAt(i) != '/' || objectName.charAt(i + 1) != '/') continue;
            throw new IllegalArgumentException(String.format("Google Cloud Storage path must not have consecutive '/' characters, got '%s'", objectName));
        }
        if (objectName.startsWith("/")) {
            objectName = objectName.substring(1);
        }
        if (objectName.length() == 0 && !allowEmptyObjectName) {
            throw new IllegalArgumentException(objectNameMessage);
        }
        LOG.debug("validateObjectName -> '{}'", (Object)objectName);
        return objectName;
    }

    String getItemName(URI path) {
        Preconditions.checkNotNull((Object)path);
        if (path.equals(GCS_ROOT)) {
            return null;
        }
        StorageResourceId resourceId = this.pathCodec.validatePathAndGetId(path, true);
        if (resourceId.isBucket()) {
            return resourceId.getBucketName();
        }
        int index = FileInfo.objectHasDirectoryPath(resourceId.getObjectName()) ? resourceId.getObjectName().lastIndexOf("/", resourceId.getObjectName().length() - 2) : resourceId.getObjectName().lastIndexOf("/");
        if (index < 0) {
            return resourceId.getObjectName();
        }
        return resourceId.getObjectName().substring(index + 1);
    }

    public URI getParentPath(URI path) {
        return GoogleCloudStorageFileSystem.getParentPath(this.getPathCodec(), path);
    }

    static FileNotFoundException getFileNotFoundException(URI path) {
        return new FileNotFoundException(String.format("Item not found: %s", path));
    }

    public GoogleCloudStorage getGcs() {
        return this.gcs;
    }

    public PathCodec getPathCodec() {
        return this.pathCodec;
    }

    @Deprecated
    public static StorageResourceId validatePathAndGetId(URI uri, boolean allowEmptyObjectNames) {
        return LEGACY_PATH_CODEC.validatePathAndGetId(uri, allowEmptyObjectNames);
    }

    @Deprecated
    public static URI getPath(String bucketName, String objectName, boolean allowEmptyObjectName) {
        return LEGACY_PATH_CODEC.getPath(bucketName, objectName, allowEmptyObjectName);
    }

    @Deprecated
    public static URI getPath(String bucketName) {
        return LEGACY_PATH_CODEC.getPath(bucketName, null, true);
    }

    @Deprecated
    public static URI getPath(String bucketName, String objectName) {
        return LEGACY_PATH_CODEC.getPath(bucketName, objectName, false);
    }

    @Deprecated
    public static URI getParentPath(PathCodec pathCodec, URI path) {
        Preconditions.checkNotNull((Object)path);
        if (path.equals(GCS_ROOT)) {
            return null;
        }
        StorageResourceId resourceId = pathCodec.validatePathAndGetId(path, true);
        if (resourceId.isBucket()) {
            return GCS_ROOT;
        }
        int index = FileInfo.objectHasDirectoryPath(resourceId.getObjectName()) ? resourceId.getObjectName().lastIndexOf("/", resourceId.getObjectName().length() - 2) : resourceId.getObjectName().lastIndexOf("/");
        if (index < 0) {
            return pathCodec.getPath(resourceId.getBucketName(), null, true);
        }
        return pathCodec.getPath(resourceId.getBucketName(), resourceId.getObjectName().substring(0, index + 1), false);
    }
}

