/*
 * Decompiled with CFR 0.152.
 */
package org.wildfly.security.audit;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.time.temporal.TemporalAdjusters;
import java.time.temporal.WeekFields;
import java.util.Locale;
import org.wildfly.common.Assert;
import org.wildfly.security.audit.AuditEndpoint;
import org.wildfly.security.audit.ElytronMessages;
import org.wildfly.security.audit.FileAuditEndpoint;

public class PeriodicRotatingFileAuditEndpoint
extends FileAuditEndpoint {
    private final DateTimeFormatter format;
    private final Period period;
    private final ZoneId timeZone;
    private long nextRollover = Long.MAX_VALUE;
    private String nextSuffix;

    PeriodicRotatingFileAuditEndpoint(Builder builder) throws IOException {
        super(builder);
        this.format = builder.format;
        this.period = builder.period;
        this.timeZone = builder.timeZone;
        File file = this.getFile();
        this.calcNextRollover(file != null && file.lastModified() > 0L ? file.lastModified() : this.clock.millis());
    }

    @Override
    protected void preWrite(Instant instant) {
        long recordMillis = instant.toEpochMilli();
        if (recordMillis >= this.nextRollover) {
            try {
                File file = this.getFile();
                if (file == null) {
                    return;
                }
                this.closeStreams();
                Path target = Paths.get(file.toPath() + this.nextSuffix, new String[0]);
                Files.move(file.toPath(), target, StandardCopyOption.REPLACE_EXISTING);
                this.setFile(file);
            }
            catch (IOException e) {
                ElytronMessages.audit.unableToRotateLogFile(e);
            }
            this.calcNextRollover(recordMillis);
        }
    }

    private void calcNextRollover(long fromTime) {
        if (this.period == Period.NEVER || this.format == null) {
            this.nextRollover = Long.MAX_VALUE;
            return;
        }
        ZonedDateTime zonedDateTime = ZonedDateTime.ofInstant(Instant.ofEpochMilli(fromTime), this.timeZone);
        this.nextSuffix = this.format.format(zonedDateTime);
        switch (this.period) {
            case YEAR: {
                zonedDateTime = zonedDateTime.truncatedTo(ChronoUnit.DAYS).withDayOfYear(1).plus(1L, ChronoUnit.YEARS);
                break;
            }
            case MONTH: {
                zonedDateTime = zonedDateTime.truncatedTo(ChronoUnit.DAYS).withDayOfMonth(1).plus(1L, ChronoUnit.MONTHS);
                break;
            }
            case WEEK: {
                zonedDateTime = zonedDateTime.truncatedTo(ChronoUnit.DAYS).with(TemporalAdjusters.next(WeekFields.of(Locale.getDefault()).getFirstDayOfWeek()));
                break;
            }
            case DAY: {
                zonedDateTime = zonedDateTime.truncatedTo(ChronoUnit.DAYS).plus(1L, ChronoUnit.DAYS);
                break;
            }
            case HALF_DAY: {
                ZonedDateTime halfDay = ZonedDateTime.from(zonedDateTime).truncatedTo(ChronoUnit.DAYS).plus(1L, ChronoUnit.HALF_DAYS);
                if (zonedDateTime.isBefore(halfDay)) {
                    zonedDateTime = halfDay;
                    break;
                }
                zonedDateTime = halfDay.plus(1L, ChronoUnit.HALF_DAYS);
                break;
            }
            case HOUR: {
                zonedDateTime = zonedDateTime.truncatedTo(ChronoUnit.HOURS).plus(1L, ChronoUnit.HOURS);
                break;
            }
            case MINUTE: {
                zonedDateTime = zonedDateTime.truncatedTo(ChronoUnit.MINUTES).plus(1L, ChronoUnit.MINUTES);
            }
        }
        this.nextRollover = zonedDateTime.toInstant().toEpochMilli();
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder
    extends FileAuditEndpoint.Builder {
        DateTimeFormatter format;
        Period period = Period.NEVER;
        ZoneId timeZone = ZoneId.systemDefault();

        Builder() {
        }

        public Builder setTimeZone(ZoneId timeZone) {
            this.timeZone = Assert.checkNotNullParam("timeZone", timeZone);
            return this;
        }

        public Builder setSuffix(String suffix) throws IllegalArgumentException {
            this.format = DateTimeFormatter.ofPattern(suffix).withZone(this.timeZone);
            int len = suffix.length();
            this.period = Period.NEVER;
            block11: for (int i = 0; i < len; ++i) {
                switch (suffix.charAt(i)) {
                    case 'y': {
                        this.period = Builder.min(this.period, Period.YEAR);
                        continue block11;
                    }
                    case 'M': {
                        this.period = Builder.min(this.period, Period.MONTH);
                        continue block11;
                    }
                    case 'W': 
                    case 'w': {
                        this.period = Builder.min(this.period, Period.WEEK);
                        continue block11;
                    }
                    case 'D': 
                    case 'E': 
                    case 'F': 
                    case 'd': {
                        this.period = Builder.min(this.period, Period.DAY);
                        continue block11;
                    }
                    case 'a': {
                        this.period = Builder.min(this.period, Period.HALF_DAY);
                        continue block11;
                    }
                    case 'H': 
                    case 'K': 
                    case 'h': 
                    case 'k': {
                        this.period = Builder.min(this.period, Period.HOUR);
                        continue block11;
                    }
                    case 'm': {
                        this.period = Builder.min(this.period, Period.MINUTE);
                        continue block11;
                    }
                    case '\'': {
                        while (suffix.charAt(++i) != '\'') {
                        }
                        continue block11;
                    }
                    case 'S': 
                    case 's': {
                        throw ElytronMessages.audit.rotatingBySecondUnsupported(suffix);
                    }
                }
            }
            return this;
        }

        @Override
        public AuditEndpoint build() throws IOException {
            return new PeriodicRotatingFileAuditEndpoint(this);
        }

        private static <T extends Comparable<? super T>> T min(T a, T b) {
            return a.compareTo(b) <= 0 ? a : b;
        }
    }

    static enum Period {
        MINUTE,
        HOUR,
        HALF_DAY,
        DAY,
        WEEK,
        MONTH,
        YEAR,
        NEVER;

    }
}

