/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.component.etcd3.policy;

import io.etcd.jetcd.ByteSequence;
import io.etcd.jetcd.Client;
import io.etcd.jetcd.KV;
import io.etcd.jetcd.Lease;
import io.etcd.jetcd.Txn;
import io.etcd.jetcd.common.exception.ErrorCode;
import io.etcd.jetcd.common.exception.EtcdException;
import io.etcd.jetcd.kv.TxnResponse;
import io.etcd.jetcd.lease.LeaseKeepAliveResponse;
import io.etcd.jetcd.op.Cmp;
import io.etcd.jetcd.op.CmpTarget;
import io.etcd.jetcd.op.Op;
import io.etcd.jetcd.options.PutOption;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.camel.CamelContext;
import org.apache.camel.CamelContextAware;
import org.apache.camel.Route;
import org.apache.camel.api.management.ManagedAttribute;
import org.apache.camel.api.management.ManagedResource;
import org.apache.camel.component.etcd3.Etcd3Configuration;
import org.apache.camel.support.RoutePolicySupport;
import org.apache.camel.util.ObjectHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ManagedResource(description="Route policy using Etcd as clustered lock")
public class Etcd3RoutePolicy
extends RoutePolicySupport
implements CamelContextAware {
    private static final Logger LOGGER = LoggerFactory.getLogger(Etcd3RoutePolicy.class);
    private final AtomicBoolean leader = new AtomicBoolean();
    private final Set<Route> suspendedRoutes = new HashSet<Route>();
    private int ttl = 60;
    private int timeout = 10;
    private volatile Route route;
    private String serviceName;
    private String servicePath;
    private CamelContext camelContext;
    private String[] endpoints;
    private volatile ScheduledExecutorService executorService;
    private final AtomicBoolean shouldStopConsumer = new AtomicBoolean(true);
    private final AtomicLong leaseId = new AtomicLong();
    private final AtomicReference<Client> client = new AtomicReference();
    private final AtomicReference<KV> kv = new AtomicReference();
    private final AtomicReference<Lease> lease = new AtomicReference();
    private final boolean managedClient;

    public Etcd3RoutePolicy() {
        this("http://localhost:2379");
    }

    public Etcd3RoutePolicy(Etcd3Configuration configuration) {
        this(configuration.createClient(), true);
    }

    public Etcd3RoutePolicy(Client client) {
        this(client, false);
    }

    private Etcd3RoutePolicy(Client client, boolean managedClient) {
        this.client.set(ObjectHelper.notNull(client, "client"));
        this.managedClient = managedClient;
    }

    public Etcd3RoutePolicy(String ... endpoints) {
        this.endpoints = endpoints;
        this.managedClient = true;
    }

    @Override
    public void onInit(Route route) {
        super.onInit(route);
        this.route = route;
        if (this.executorService == null) {
            this.executorService = ObjectHelper.notNull(this.camelContext, "camelContext", this).getExecutorServiceManager().newSingleThreadScheduledExecutor(this, "Etcd3RoutePolicy[" + route.getRouteId() + "]");
        }
    }

    @Override
    public void onStart(Route route) {
        if (!this.leader.get() && this.shouldStopConsumer.get()) {
            this.stopConsumer(route);
        }
    }

    @Override
    public void onStop(Route route) {
        this.lock.lock();
        try {
            this.suspendedRoutes.remove(route);
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public void onSuspend(Route route) {
        this.lock.lock();
        try {
            this.suspendedRoutes.remove(route);
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    protected void doStart() throws Exception {
        Client c = this.client.get();
        if (c == null) {
            c = Client.builder().endpoints(ObjectHelper.notNull(this.endpoints, "endpoints")).build();
            this.client.set(c);
        }
        this.lease.set(c.getLeaseClient());
        this.kv.set(c.getKVClient());
        this.evaluateLeadershipAndSchedule();
        super.doStart();
    }

    @Override
    protected void doStop() throws Exception {
        if (this.executorService != null) {
            this.camelContext.getExecutorServiceManager().shutdownNow(this.executorService);
            this.executorService = null;
        }
        try {
            Client c = this.client.get();
            if (this.managedClient && c != null) {
                c.close();
            }
        }
        finally {
            super.doStop();
        }
    }

    @Override
    public CamelContext getCamelContext() {
        return this.camelContext;
    }

    @Override
    public void setCamelContext(CamelContext camelContext) {
        this.camelContext = camelContext;
    }

    protected void setLeader(boolean isLeader) {
        if (isLeader) {
            if (this.leader.compareAndSet(false, true)) {
                LOGGER.info("Leadership taken (path={}, name={})", (Object)this.servicePath, (Object)this.serviceName);
                this.startAllStoppedConsumers();
            }
        } else if (this.leader.compareAndSet(true, false)) {
            LOGGER.info("Leadership lost (path={}, name={})", (Object)this.servicePath, (Object)this.serviceName);
        }
    }

    private void stopConsumer(Route route) {
        this.lock.lock();
        try {
            if (!this.suspendedRoutes.contains(route)) {
                LOGGER.debug("Stopping consumer for {} ({})", (Object)route.getId(), (Object)route.getConsumer());
                this.stopConsumer(route.getConsumer());
                this.suspendedRoutes.add(route);
            }
        }
        catch (Exception e) {
            this.handleException(e);
        }
        finally {
            this.lock.unlock();
        }
    }

    private void startAllStoppedConsumers() {
        this.lock.lock();
        try {
            for (Route suspendedRoute : this.suspendedRoutes) {
                LOGGER.debug("Starting consumer for {} ({})", (Object)suspendedRoute.getId(), (Object)suspendedRoute.getConsumer());
                this.startConsumer(suspendedRoute.getConsumer());
            }
            this.suspendedRoutes.clear();
        }
        catch (Exception e) {
            this.handleException(e);
        }
        finally {
            this.lock.unlock();
        }
    }

    public Client getClient() {
        return this.client.get();
    }

    @ManagedAttribute(description="The route id")
    public String getRouteId() {
        if (this.route != null) {
            return this.route.getId();
        }
        return null;
    }

    @ManagedAttribute(description="The consumer endpoint", mask=true)
    public String getEndpointUrl() {
        if (this.route != null && this.route.getConsumer() != null && this.route.getConsumer().getEndpoint() != null) {
            return this.route.getConsumer().getEndpoint().toString();
        }
        return null;
    }

    public String getServiceName() {
        return this.serviceName;
    }

    @ManagedAttribute(description="The etcd service name")
    public void setServiceName(String serviceName) {
        this.serviceName = serviceName;
    }

    @ManagedAttribute(description="The etcd service path")
    public String getServicePath() {
        return this.servicePath;
    }

    public void setServicePath(String servicePath) {
        this.servicePath = servicePath;
    }

    @ManagedAttribute(description="The time to live (seconds)")
    public int getTtl() {
        return this.ttl;
    }

    public void setTtl(int ttl) {
        this.ttl = ttl;
    }

    @ManagedAttribute(description="The request timeout (seconds)")
    public int getTimeout() {
        return this.timeout;
    }

    public void setTimeout(int timeout) {
        this.timeout = timeout;
    }

    @ManagedAttribute(description="Whether to stop consumer when starting up and failed to become master")
    public boolean isShouldStopConsumer() {
        return this.shouldStopConsumer.get();
    }

    public void setShouldStopConsumer(boolean shouldStopConsumer) {
        this.shouldStopConsumer.set(shouldStopConsumer);
    }

    @ManagedAttribute(description="Is this route the master or a slave")
    public boolean isLeader() {
        return this.leader.get();
    }

    @ManagedAttribute(description="Etcd endpoints")
    public String getEndpoints() {
        return this.endpoints == null ? "" : String.join((CharSequence)",", this.endpoints);
    }

    public void setEndpoints(String[] endpoints) {
        this.endpoints = endpoints;
    }

    public void setEndpoints(String endpoints) {
        this.endpoints = endpoints.split(",");
    }

    long getLeaseId() {
        return this.leaseId.get();
    }

    private void evaluateLeadershipAndSchedule() {
        this.evaluateLeadership();
        this.executorService.schedule(this::evaluateLeadershipAndSchedule, (long)Math.max(2 * this.ttl / 3, 1), TimeUnit.SECONDS);
    }

    private void evaluateLeadership() {
        if (this.isLeader() && this.renewLease()) {
            return;
        }
        this.setLeader(this.tryTakeLeadership());
    }

    private boolean renewLease() {
        long id = this.leaseId.get();
        if (id == 0L) {
            return false;
        }
        try {
            LeaseKeepAliveResponse keepAliveResponse = this.lease.get().keepAliveOnce(id).get(this.timeout, TimeUnit.SECONDS);
            LOGGER.debug("New TTL of the lease {} is {} seconds", (Object)id, (Object)keepAliveResponse.getTTL());
            return true;
        }
        catch (ExecutionException e) {
            boolean notFound = false;
            if (e.getCause() instanceof EtcdException) {
                EtcdException ex = (EtcdException)e.getCause();
                boolean bl = notFound = ex.getErrorCode() == ErrorCode.NOT_FOUND;
            }
            if (notFound) {
                LOGGER.debug("The lease {} doesn't exist anymore", (Object)id);
                this.leaseId.set(0L);
            } else {
                LOGGER.debug("Could not renew the lease {}", (Object)id, (Object)e);
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        catch (TimeoutException e) {
            LOGGER.debug("Timeout while trying to renew the lease {}", (Object)id);
        }
        return false;
    }

    private boolean tryTakeLeadership() {
        try {
            long id = this.lease.get().grant(this.ttl, this.timeout, TimeUnit.SECONDS).get(this.timeout, TimeUnit.SECONDS).getID();
            Txn transaction = this.kv.get().txn();
            ByteSequence key = ByteSequence.from(this.servicePath.getBytes());
            TxnResponse response = transaction.If(new Cmp(key, Cmp.Op.EQUAL, CmpTarget.version(0L))).Then(Op.put(key, ByteSequence.from(this.serviceName.getBytes()), PutOption.newBuilder().withLeaseId(id).build())).commit().get(this.timeout, TimeUnit.SECONDS);
            boolean succeeded = response.isSucceeded();
            if (succeeded) {
                this.leaseId.set(id);
            } else {
                this.lease.get().revoke(id);
            }
            return succeeded;
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        catch (ExecutionException e) {
            LOGGER.debug("Could not try to take the leadership", (Throwable)e);
        }
        catch (TimeoutException e) {
            LOGGER.debug("Timeout while trying to take the leadership");
        }
        return false;
    }
}

