/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.component.nats;

import io.nats.client.Connection;
import io.nats.client.Message;
import io.nats.client.impl.Headers;
import io.nats.client.impl.NatsMessage;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.apache.camel.AsyncCallback;
import org.apache.camel.Exchange;
import org.apache.camel.ExchangeTimedOutException;
import org.apache.camel.InvalidPayloadException;
import org.apache.camel.component.nats.NatsConfiguration;
import org.apache.camel.component.nats.NatsEndpoint;
import org.apache.camel.spi.ExecutorServiceManager;
import org.apache.camel.spi.HeaderFilterStrategy;
import org.apache.camel.spi.ThreadPoolProfile;
import org.apache.camel.support.DefaultAsyncProducer;
import org.apache.camel.util.ObjectHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NatsProducer
extends DefaultAsyncProducer {
    private static final Logger LOG = LoggerFactory.getLogger(NatsProducer.class);
    private final ExecutorServiceManager executorServiceManager;
    private ScheduledExecutorService scheduler;
    private Connection connection;

    public NatsProducer(NatsEndpoint endpoint) {
        super(endpoint);
        this.executorServiceManager = endpoint.getCamelContext().getExecutorServiceManager();
    }

    @Override
    public NatsEndpoint getEndpoint() {
        return (NatsEndpoint)super.getEndpoint();
    }

    @Override
    public boolean process(Exchange exchange, AsyncCallback callback) {
        NatsConfiguration config = this.getEndpoint().getConfiguration();
        byte[] body = exchange.getIn().getBody(byte[].class);
        if (body == null) {
            try {
                body = exchange.getIn().getMandatoryBody(String.class).getBytes();
            }
            catch (InvalidPayloadException e2) {
                exchange.setException(e2);
                callback.done(true);
                return true;
            }
        }
        if (exchange.getPattern().isOutCapable()) {
            LOG.debug("Requesting to topic: {}", (Object)config.getTopic());
            NatsMessage.Builder builder = NatsMessage.builder().data(body).subject(config.getTopic()).headers(this.buildHeaders(exchange));
            CompletableFuture<Message> requestFuture = this.connection.request(builder.build());
            CompletableFuture timeoutFuture = this.failAfter(exchange, Duration.ofMillis(config.getRequestTimeout()));
            CompletableFuture.anyOf(requestFuture, timeoutFuture).whenComplete((message, e) -> {
                if (e == null) {
                    Message msg = (Message)message;
                    exchange.getMessage().setBody(msg.getData());
                    exchange.getMessage().setHeader("CamelNatsReplyTo", msg.getReplyTo());
                    exchange.getMessage().setHeader("CamelNatsSID", msg.getSID());
                    exchange.getMessage().setHeader("CamelNatsSubject", msg.getSubject());
                    exchange.getMessage().setHeader("CamelNatsQueueName", msg.getSubscription().getQueueName());
                    exchange.getMessage().setHeader("CamelNatsMessageTimestamp", System.currentTimeMillis());
                } else {
                    exchange.setException(e.getCause());
                }
                callback.done(false);
                if (!requestFuture.isDone()) {
                    requestFuture.cancel(true);
                }
                if (!timeoutFuture.isDone()) {
                    timeoutFuture.cancel(true);
                }
            });
            return false;
        }
        LOG.debug("Publishing to topic: {}", (Object)config.getTopic());
        NatsMessage.Builder builder = NatsMessage.builder().data(body).subject(config.getTopic()).headers(this.buildHeaders(exchange));
        if (ObjectHelper.isNotEmpty(config.getReplySubject())) {
            String replySubject = config.getReplySubject();
            builder.replyTo(replySubject);
        }
        this.connection.publish(builder.build());
        callback.done(true);
        return true;
    }

    private Headers buildHeaders(Exchange exchange) {
        Headers headers = new Headers();
        HeaderFilterStrategy filteringStrategy = this.getEndpoint().getConfiguration().getHeaderFilterStrategy();
        exchange.getIn().getHeaders().forEach((key, value) -> {
            if (!filteringStrategy.applyFilterToCamelHeaders((String)key, value, exchange)) {
                String headerValue = value instanceof byte[] ? new String((byte[])value, StandardCharsets.UTF_8) : String.valueOf(value);
                if (headers.get((String)key) != null) {
                    headers.get((String)key).add(headerValue);
                } else {
                    headers.add((String)key, headerValue);
                }
            } else {
                LOG.debug("Excluding header {} as per strategy", key);
            }
        });
        return headers;
    }

    private <T> CompletableFuture<T> failAfter(Exchange exchange, Duration duration) {
        CompletableFuture future = new CompletableFuture();
        this.scheduler.schedule(() -> {
            ExchangeTimedOutException ex = new ExchangeTimedOutException(exchange, duration.toMillis());
            return future.completeExceptionally(ex);
        }, duration.toNanos(), TimeUnit.NANOSECONDS);
        return future;
    }

    @Override
    protected void doStart() throws Exception {
        ThreadPoolProfile profile = this.executorServiceManager.getThreadPoolProfile("CamelNatsRequestTimeoutExecutor");
        if (profile == null) {
            profile = this.executorServiceManager.getDefaultThreadPoolProfile();
        }
        this.scheduler = this.executorServiceManager.newScheduledThreadPool((Object)this, "CamelNatsRequestTimeoutExecutor", profile);
        super.doStart();
        LOG.debug("Starting Nats Producer");
        LOG.debug("Getting Nats Connection");
        this.connection = this.getEndpoint().getConfiguration().getConnection() != null ? this.getEndpoint().getConfiguration().getConnection() : this.getEndpoint().getConnection();
    }

    @Override
    protected void doStop() throws Exception {
        if (this.scheduler != null) {
            this.executorServiceManager.shutdownNow(this.scheduler);
        }
        LOG.debug("Stopping Nats Producer");
        if (ObjectHelper.isEmpty(this.getEndpoint().getConfiguration().getConnection())) {
            LOG.debug("Closing Nats Connection");
            if (this.connection != null && !this.connection.getStatus().equals((Object)Connection.Status.CLOSED)) {
                if (this.getEndpoint().getConfiguration().isFlushConnection()) {
                    LOG.debug("Flushing Nats Connection");
                    this.connection.flush(Duration.ofMillis(this.getEndpoint().getConfiguration().getFlushTimeout()));
                }
                this.connection.close();
            }
        }
        super.doStop();
    }
}

