/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.component.platform.http.vertx;

import io.netty.handler.codec.http.HttpHeaderValues;
import io.vertx.core.Context;
import io.vertx.core.Future;
import io.vertx.core.MultiMap;
import io.vertx.core.Promise;
import io.vertx.core.Vertx;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.http.HttpServerRequest;
import io.vertx.core.http.HttpServerResponse;
import io.vertx.core.net.SocketAddress;
import io.vertx.core.streams.Pump;
import io.vertx.ext.web.RoutingContext;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.apache.camel.Exchange;
import org.apache.camel.Message;
import org.apache.camel.NoTypeConversionAvailableException;
import org.apache.camel.TypeConverter;
import org.apache.camel.component.platform.http.vertx.AsyncInputStream;
import org.apache.camel.spi.HeaderFilterStrategy;
import org.apache.camel.support.ExceptionHelper;
import org.apache.camel.support.ExchangeHelper;
import org.apache.camel.support.MessageHelper;
import org.apache.camel.support.http.HttpUtil;
import org.apache.camel.util.CollectionHelper;
import org.apache.camel.util.ObjectHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class VertxPlatformHttpSupport {
    private static final Logger LOGGER = LoggerFactory.getLogger(VertxPlatformHttpSupport.class);
    public static final String DEFAULT_CONTENT_TYPE_ON_EXCEPTION = "text/plain; charset=utf-8";

    private VertxPlatformHttpSupport() {
    }

    static Object toHttpResponse(RoutingContext ctx, Message message, HeaderFilterStrategy headerFilterStrategy, boolean muteExceptions) {
        Object body;
        Integer length;
        Exchange exchange = message.getExchange();
        HttpServerResponse response = ctx.response();
        int code = HttpUtil.determineResponseCode(exchange, message.getBody());
        response.setStatusCode(code);
        if (headerFilterStrategy != null) {
            VertxPlatformHttpSupport.copyMessageHeadersToResponse(response, ctx.pathParams(), message, headerFilterStrategy, exchange);
        }
        if ((length = VertxPlatformHttpSupport.determineContentLength(body = VertxPlatformHttpSupport.getBody(message, muteExceptions, exchange))) != null) {
            response.putHeader("Content-Length", String.valueOf(length));
        } else {
            response.setChunked(true);
        }
        String contentType = MessageHelper.getContentType(message);
        if (contentType != null) {
            response.putHeader("Content-Type", contentType);
        }
        return body;
    }

    private static Object getBody(Message message, boolean muteExceptions, Exchange exchange) {
        Exception exception = exchange.getException();
        if (exception != null) {
            return VertxPlatformHttpSupport.handleExceptions(message, muteExceptions, exception, exchange);
        }
        return message.getBody();
    }

    private static Object handleExceptions(Message message, boolean muteExceptions, Exception exception, Exchange exchange) {
        Object body;
        if (muteExceptions) {
            body = "";
            message.setHeader("Content-Type", DEFAULT_CONTENT_TYPE_ON_EXCEPTION);
        } else {
            String stackTrace = ExceptionHelper.stackTraceToString(exception);
            body = ByteBuffer.wrap(stackTrace.getBytes(StandardCharsets.UTF_8));
            message.setHeader("Content-Type", DEFAULT_CONTENT_TYPE_ON_EXCEPTION);
        }
        ExchangeHelper.setFailureHandled(exchange);
        return body;
    }

    private static void copyMessageHeadersToResponse(HttpServerResponse response, Map<String, String> pathParams, Message message, HeaderFilterStrategy headerFilterStrategy, Exchange exchange) {
        TypeConverter tc = exchange.getContext().getTypeConverter();
        for (Map.Entry<String, Object> entry : message.getHeaders().entrySet()) {
            String key = entry.getKey();
            if (pathParams.containsKey(key)) continue;
            Object value = entry.getValue();
            Iterator<?> it = org.apache.camel.support.ObjectHelper.createIterator(value, null, true);
            VertxPlatformHttpSupport.putHeader(response, headerFilterStrategy, exchange, it, tc, key);
        }
    }

    private static void putHeader(HttpServerResponse response, HeaderFilterStrategy headerFilterStrategy, Exchange exchange, Iterator<?> it, TypeConverter tc, String key) {
        HttpUtil.applyHeader(headerFilterStrategy, exchange, it, tc, key, (values, firstValue) -> VertxPlatformHttpSupport.applyHeader(response, key, values, firstValue));
    }

    private static void applyHeader(HttpServerResponse response, String key, List<String> values, String firstValue) {
        if (values != null) {
            response.putHeader(key, values);
        } else if (firstValue != null) {
            response.putHeader(key, firstValue);
        }
    }

    static Integer determineContentLength(Object body) {
        if (body instanceof byte[]) {
            return ((byte[])body).length;
        }
        if (body instanceof ByteBuffer) {
            return ((ByteBuffer)body).remaining();
        }
        return null;
    }

    static Future<Void> writeResponse(RoutingContext ctx, Exchange camelExchange, HeaderFilterStrategy headerFilterStrategy, boolean muteExceptions) {
        Promise<Void> promise = Promise.promise();
        try {
            Object body = VertxPlatformHttpSupport.toHttpResponse(ctx, camelExchange.getMessage(), headerFilterStrategy, muteExceptions);
            if (body == null) {
                LOGGER.trace("No payload to send as reply for exchange: {}", (Object)camelExchange);
                ctx.end();
                promise.complete();
            } else if (body instanceof String) {
                ctx.end((String)body);
                promise.complete();
            } else if (body instanceof InputStream) {
                VertxPlatformHttpSupport.writeResponseAs(promise, ctx, (InputStream)body);
            } else if (body instanceof Buffer) {
                ctx.end((Buffer)body);
                promise.complete();
            } else {
                VertxPlatformHttpSupport.writeResponseAsFallback(promise, camelExchange, body, ctx);
            }
        }
        catch (Exception e) {
            promise.fail(e);
        }
        return promise.future();
    }

    private static void writeResponseAsFallback(Promise<Void> promise, Exchange camelExchange, Object body, RoutingContext ctx) throws NoTypeConversionAvailableException {
        TypeConverter tc = camelExchange.getContext().getTypeConverter();
        ByteBuffer bb = tc.tryConvertTo(ByteBuffer.class, camelExchange, body);
        if (bb != null) {
            VertxPlatformHttpSupport.writeResponseAs(promise, ctx, bb);
        } else {
            InputStream is = tc.mandatoryConvertTo(InputStream.class, camelExchange, body);
            VertxPlatformHttpSupport.writeResponseAs(promise, ctx, is);
        }
    }

    private static void writeResponseAs(Promise<Void> promise, RoutingContext ctx, ByteBuffer bb) {
        Buffer b = Buffer.buffer(bb.capacity());
        b.setBytes(0, bb);
        ctx.end(b);
        promise.complete();
    }

    private static void writeResponseAs(Promise<Void> promise, RoutingContext ctx, InputStream is) {
        HttpServerResponse response = ctx.response();
        Vertx vertx = ctx.vertx();
        Context context = vertx.getOrCreateContext();
        AsyncInputStream asyncInputStream = new AsyncInputStream(vertx, context, is);
        asyncInputStream.exceptionHandler(promise::fail);
        asyncInputStream.endHandler(event -> VertxPlatformHttpSupport.endHandler(promise, response, asyncInputStream));
        Pump pump = Pump.pump(asyncInputStream, response);
        context.runOnContext(event -> pump.start());
    }

    private static void endHandler(Promise<Void> promise, HttpServerResponse response, AsyncInputStream asyncInputStream) {
        response.end().onComplete(result -> VertxPlatformHttpSupport.onComplete(promise, asyncInputStream));
    }

    private static void onComplete(Promise<Void> promise, AsyncInputStream asyncInputStream) {
        asyncInputStream.close(closeResult -> promise.complete());
    }

    static void populateCamelHeaders(RoutingContext ctx, Map<String, Object> headersMap, Exchange exchange, HeaderFilterStrategy headerFilterStrategy) {
        SocketAddress remoteAddress;
        HttpServerRequest request = ctx.request();
        headersMap.put("CamelHttpPath", ctx.normalizedPath());
        if (headerFilterStrategy != null) {
            VertxPlatformHttpSupport.applyHeaderFilterStrategy(ctx, headersMap, exchange, headerFilterStrategy, request);
        }
        for (Map.Entry<String, String> en : ctx.pathParams().entrySet()) {
            CollectionHelper.appendEntry(headersMap, en.getKey(), en.getValue());
        }
        SocketAddress localAddress = request.localAddress();
        if (localAddress != null) {
            headersMap.put("CamelVertxPlatformHttpLocalAddress", localAddress);
        }
        if ((remoteAddress = request.remoteAddress()) != null) {
            headersMap.put("CamelVertxPlatformHttpRemoteAddress", remoteAddress);
        }
        headersMap.put("CamelHttpMethod", request.method().toString());
        headersMap.put("CamelHttpUrl", request.absoluteURI());
        headersMap.put("CamelHttpUri", request.uri());
        headersMap.put("CamelHttpQuery", request.query());
        headersMap.put("CamelHttpRawQuery", request.query());
    }

    private static void applyHeaderFilterStrategy(RoutingContext ctx, Map<String, Object> headersMap, Exchange exchange, HeaderFilterStrategy headerFilterStrategy, HttpServerRequest request) {
        MultiMap requestHeaders = request.headers();
        VertxPlatformHttpSupport.applyAuthHeaders(headersMap, exchange, headerFilterStrategy, requestHeaders);
        VertxPlatformHttpSupport.applyHeaders(headersMap, exchange, headerFilterStrategy, requestHeaders);
        MultiMap pathParameters = ctx.queryParams();
        if (!pathParameters.isEmpty()) {
            VertxPlatformHttpSupport.applyHeaders(headersMap, exchange, headerFilterStrategy, pathParameters);
        }
    }

    private static void applyHeaders(Map<String, Object> headersMap, Exchange exchange, HeaderFilterStrategy headerFilterStrategy, MultiMap requestHeaders) {
        List<Map.Entry<String, String>> entries = requestHeaders.entries();
        for (Map.Entry<String, String> entry : entries) {
            String value;
            String name = entry.getKey();
            if (headerFilterStrategy.applyFilterToExternalHeaders(name, value = entry.getValue(), exchange)) continue;
            CollectionHelper.appendEntry(headersMap, name, value);
        }
    }

    private static void applyAuthHeaders(Map<String, Object> headersMap, Exchange exchange, HeaderFilterStrategy headerFilterStrategy, MultiMap requestHeaders) {
        String authorization = requestHeaders.get("authorization");
        if (authorization != null && authorization.trim().startsWith("Basic") && !headerFilterStrategy.applyFilterToExternalHeaders("CamelAuthentication", "Basic", exchange)) {
            CollectionHelper.appendEntry(headersMap, "CamelAuthentication", "Basic");
        }
    }

    static boolean isMultiPartFormData(RoutingContext ctx) {
        return VertxPlatformHttpSupport.isContentTypeMatching(ctx, HttpHeaderValues.MULTIPART_FORM_DATA.toString());
    }

    static boolean isFormUrlEncoded(RoutingContext ctx) {
        return VertxPlatformHttpSupport.isContentTypeMatching(ctx, HttpHeaderValues.APPLICATION_X_WWW_FORM_URLENCODED.toString());
    }

    private static boolean isContentTypeMatching(RoutingContext ctx, String expectedContentType) {
        String contentType = ctx.parsedHeaders().contentType().value();
        boolean match = false;
        if (ObjectHelper.isNotEmpty(contentType)) {
            String lowerCaseContentType = contentType.toLowerCase();
            match = lowerCaseContentType.startsWith(expectedContentType);
        }
        return match;
    }
}

