/*
 * Decompiled with CFR 0.152.
 */
package org.redisson.connection;

import io.netty.resolver.AddressResolver;
import io.netty.resolver.AddressResolverGroup;
import io.netty.util.NetUtil;
import io.netty.util.Timeout;
import io.netty.util.concurrent.Future;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import org.redisson.client.RedisClient;
import org.redisson.connection.ConnectionManager;
import org.redisson.connection.MasterSlaveEntry;
import org.redisson.misc.RedisURI;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DNSMonitor {
    private static final Logger log = LoggerFactory.getLogger(DNSMonitor.class);
    private final AddressResolver<InetSocketAddress> resolver;
    private final ConnectionManager connectionManager;
    private final Map<RedisURI, InetSocketAddress> masters = new HashMap<RedisURI, InetSocketAddress>();
    private final Map<RedisURI, InetSocketAddress> slaves = new HashMap<RedisURI, InetSocketAddress>();
    private volatile Timeout dnsMonitorFuture;
    private final long dnsMonitoringInterval;
    private boolean printed;

    public DNSMonitor(ConnectionManager connectionManager, RedisClient masterHost, Collection<RedisURI> slaveHosts, long dnsMonitoringInterval, AddressResolverGroup<InetSocketAddress> resolverGroup) {
        this.resolver = resolverGroup.getResolver(connectionManager.getServiceManager().getGroup().next());
        masterHost.resolveAddr().join();
        this.masters.put(masterHost.getConfig().getAddress(), masterHost.getAddr());
        for (RedisURI host : slaveHosts) {
            Future<InetSocketAddress> resolveFuture = this.resolver.resolve(InetSocketAddress.createUnresolved(host.getHost(), host.getPort()));
            resolveFuture.syncUninterruptibly();
            this.slaves.put(host, resolveFuture.getNow());
        }
        this.connectionManager = connectionManager;
        this.dnsMonitoringInterval = dnsMonitoringInterval;
    }

    public void start() {
        this.monitorDnsChange();
        log.debug("DNS monitoring enabled; Current masters: {}, slaves: {}", this.masters, this.slaves);
    }

    public void stop() {
        if (this.dnsMonitorFuture != null) {
            this.dnsMonitorFuture.cancel();
        }
    }

    private void monitorDnsChange() {
        this.dnsMonitorFuture = this.connectionManager.getServiceManager().newTimeout(t -> {
            if (this.connectionManager.getServiceManager().isShuttingDown()) {
                return;
            }
            CompletableFuture<Void> mf = this.monitorMasters();
            CompletableFuture<Void> sf = this.monitorSlaves();
            CompletableFuture.allOf(mf, sf).whenComplete((r, e) -> this.monitorDnsChange());
        }, this.dnsMonitoringInterval, TimeUnit.MILLISECONDS);
    }

    private CompletableFuture<Void> monitorMasters() {
        ArrayList futures = new ArrayList();
        for (Map.Entry<RedisURI, InetSocketAddress> entry : this.masters.entrySet()) {
            CompletableFuture promise = new CompletableFuture();
            futures.add(promise);
            CompletableFuture<List<RedisURI>> ipsFuture = this.connectionManager.getServiceManager().resolveAll(entry.getKey());
            ipsFuture.whenComplete((addresses, ex) -> {
                RedisURI address2;
                if (ex != null) {
                    log.error("Unable to resolve {}", (Object)((RedisURI)entry.getKey()).getHost(), ex);
                    promise.complete(null);
                    return;
                }
                if (addresses.size() > 1 && !this.printed) {
                    log.info("Try Redisson PRO with Proxy mode to use all ip addresses: {}", addresses);
                    this.printed = true;
                }
                for (RedisURI address2 : addresses) {
                    if (!address2.equals((InetSocketAddress)entry.getValue())) continue;
                    log.debug("{} resolved to {}", (Object)((RedisURI)entry.getKey()).getHost(), addresses);
                    promise.complete(null);
                    return;
                }
                int index = 0;
                if (addresses.size() > 1) {
                    addresses.sort(Comparator.comparing(RedisURI::getHost));
                }
                address2 = (RedisURI)addresses.get(index);
                log.debug("{} resolved to {} and {} selected", new Object[]{((RedisURI)entry.getKey()).getHost(), addresses, address2});
                try {
                    InetSocketAddress currentMasterAddr = (InetSocketAddress)entry.getValue();
                    byte[] addr = NetUtil.createByteArrayFromIpAddressString(address2.getHost());
                    InetSocketAddress newMasterAddr = new InetSocketAddress(InetAddress.getByAddress(((RedisURI)entry.getKey()).getHost(), addr), address2.getPort());
                    if (!address2.equals(currentMasterAddr)) {
                        log.info("Detected DNS change. Master {} has changed ip from {} to {}", new Object[]{entry.getKey(), currentMasterAddr.getAddress().getHostAddress(), newMasterAddr.getAddress().getHostAddress()});
                        MasterSlaveEntry masterSlaveEntry = this.connectionManager.getEntry(currentMasterAddr);
                        if (masterSlaveEntry == null) {
                            log.error("Unable to find entry for current master {}", (Object)currentMasterAddr);
                            promise.complete(null);
                            return;
                        }
                        CompletableFuture<RedisClient> changeFuture = masterSlaveEntry.changeMaster(newMasterAddr, (RedisURI)entry.getKey());
                        changeFuture.whenComplete((r, e) -> {
                            promise.complete(null);
                            if (e == null) {
                                this.masters.put((RedisURI)entry.getKey(), newMasterAddr);
                            }
                        });
                    } else {
                        promise.complete(null);
                    }
                }
                catch (UnknownHostException e2) {
                    log.error(e2.getMessage(), (Throwable)e2);
                    promise.complete(null);
                }
            });
        }
        return CompletableFuture.allOf(futures.toArray(new CompletableFuture[0]));
    }

    private CompletableFuture<Void> monitorSlaves() {
        ArrayList futures = new ArrayList();
        for (Map.Entry<RedisURI, InetSocketAddress> entry : this.slaves.entrySet()) {
            CompletableFuture promise = new CompletableFuture();
            futures.add(promise);
            log.debug("Request sent to resolve ip address for slave host: {}", (Object)entry.getKey().getHost());
            Future<InetSocketAddress> resolveFuture = this.resolver.resolve(InetSocketAddress.createUnresolved(entry.getKey().getHost(), entry.getKey().getPort()));
            resolveFuture.addListener(future -> {
                if (!future.isSuccess()) {
                    log.error("Unable to resolve {}", (Object)((RedisURI)entry.getKey()).getHost(), (Object)future.cause());
                    promise.complete(null);
                    return;
                }
                log.debug("Resolved ip: {} for slave host: {}", (Object)((InetSocketAddress)future.getNow()).getAddress(), (Object)((RedisURI)entry.getKey()).getHost());
                InetSocketAddress currentSlaveAddr = (InetSocketAddress)entry.getValue();
                InetSocketAddress newSlaveAddr = (InetSocketAddress)future.getNow();
                if (!newSlaveAddr.getAddress().equals(currentSlaveAddr.getAddress())) {
                    log.info("Detected DNS change. Slave {} has changed ip from {} to {}", new Object[]{((RedisURI)entry.getKey()).getHost(), currentSlaveAddr.getAddress().getHostAddress(), newSlaveAddr.getAddress().getHostAddress()});
                    boolean slaveFound = false;
                    for (MasterSlaveEntry masterSlaveEntry : this.connectionManager.getEntrySet()) {
                        if (!masterSlaveEntry.hasSlave(currentSlaveAddr)) continue;
                        slaveFound = true;
                        if (masterSlaveEntry.hasSlave(newSlaveAddr)) {
                            CompletableFuture<Boolean> slaveUpFuture = masterSlaveEntry.slaveUpAsync(newSlaveAddr);
                            slaveUpFuture.whenComplete((r, e) -> {
                                if (e != null) {
                                    promise.complete(null);
                                    return;
                                }
                                if (r.booleanValue()) {
                                    this.slaves.put((RedisURI)entry.getKey(), newSlaveAddr);
                                    masterSlaveEntry.slaveDown(currentSlaveAddr);
                                }
                                promise.complete(null);
                            });
                            break;
                        }
                        CompletableFuture<Void> addFuture = masterSlaveEntry.addSlave(newSlaveAddr, (RedisURI)entry.getKey());
                        addFuture.whenComplete((res, e) -> {
                            if (e != null) {
                                log.error("Can't add slave: {}", (Object)newSlaveAddr, e);
                                promise.complete(null);
                                return;
                            }
                            this.slaves.put((RedisURI)entry.getKey(), newSlaveAddr);
                            masterSlaveEntry.slaveDown(currentSlaveAddr);
                            promise.complete(null);
                        });
                        break;
                    }
                    if (!slaveFound) {
                        promise.complete(null);
                    }
                } else {
                    promise.complete(null);
                }
            });
        }
        return CompletableFuture.allOf(futures.toArray(new CompletableFuture[0]));
    }
}

