/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.client.transport;

import java.net.URI;
import java.nio.ByteBuffer;
import java.util.concurrent.Executor;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.atomic.AtomicReference;
import org.eclipse.jetty.client.Request;
import org.eclipse.jetty.client.Result;
import org.eclipse.jetty.client.transport.HttpChannel;
import org.eclipse.jetty.client.transport.HttpDestination;
import org.eclipse.jetty.client.transport.HttpExchange;
import org.eclipse.jetty.client.transport.HttpRequest;
import org.eclipse.jetty.http.HttpHeader;
import org.eclipse.jetty.http.HttpHeaderValue;
import org.eclipse.jetty.io.Content;
import org.eclipse.jetty.util.BufferUtil;
import org.eclipse.jetty.util.Callback;
import org.eclipse.jetty.util.IteratingCallback;
import org.eclipse.jetty.util.Promise;
import org.eclipse.jetty.util.thread.Invocable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class HttpSender {
    private static final Logger LOG = LoggerFactory.getLogger(HttpSender.class);
    private final ContentSender contentSender = new ContentSender();
    private final AtomicReference<RequestState> requestState = new AtomicReference<RequestState>(RequestState.QUEUED);
    private final AtomicReference<Throwable> failure = new AtomicReference();
    private final HttpChannel channel;

    protected HttpSender(HttpChannel channel) {
        this.channel = channel;
    }

    protected HttpChannel getHttpChannel() {
        return this.channel;
    }

    protected HttpExchange getHttpExchange() {
        return this.channel.getHttpExchange();
    }

    public boolean isFailed() {
        return this.requestState.get() == RequestState.FAILURE;
    }

    public void send(HttpExchange exchange) {
        if (!this.queuedToBegin(exchange)) {
            return;
        }
        if (!this.beginToHeaders(exchange)) {
            return;
        }
        this.contentSender.iterate();
    }

    protected boolean expects100Continue(Request request) {
        return request.getHeaders().contains(HttpHeader.EXPECT, HttpHeaderValue.CONTINUE.asString());
    }

    protected boolean queuedToBegin(HttpExchange exchange) {
        if (!this.updateRequestState(RequestState.QUEUED, RequestState.TRANSIENT)) {
            return false;
        }
        HttpRequest request = exchange.getRequest();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Request begin {}", (Object)request);
        }
        request.notifyBegin();
        this.contentSender.exchange = exchange;
        this.contentSender.expect100 = this.expects100Continue(request);
        if (this.updateRequestState(RequestState.TRANSIENT, RequestState.BEGIN)) {
            return true;
        }
        this.abortRequest(exchange);
        return false;
    }

    protected boolean beginToHeaders(HttpExchange exchange) {
        if (!this.updateRequestState(RequestState.BEGIN, RequestState.TRANSIENT)) {
            return false;
        }
        HttpRequest request = exchange.getRequest();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Request headers {}{}{}", new Object[]{request, System.lineSeparator(), request.getHeaders().toString().trim()});
        }
        request.notifyHeaders();
        if (this.updateRequestState(RequestState.TRANSIENT, RequestState.HEADERS)) {
            return true;
        }
        this.abortRequest(exchange);
        return false;
    }

    protected boolean headersToCommit(HttpExchange exchange) {
        if (!this.updateRequestState(RequestState.HEADERS, RequestState.TRANSIENT)) {
            return false;
        }
        HttpRequest request = exchange.getRequest();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Request committed {}", (Object)request);
        }
        request.notifyCommit();
        if (this.updateRequestState(RequestState.TRANSIENT, RequestState.COMMIT)) {
            return true;
        }
        this.abortRequest(exchange);
        return false;
    }

    protected boolean someToContent(HttpExchange exchange, ByteBuffer content) {
        RequestState current = this.requestState.get();
        return switch (current.ordinal()) {
            case 4, 5 -> {
                if (!this.updateRequestState(current, RequestState.TRANSIENT)) {
                    yield false;
                }
                HttpRequest request = exchange.getRequest();
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Request content {}{}{}", new Object[]{request, System.lineSeparator(), BufferUtil.toDetailString(content)});
                }
                request.notifyContent(content);
                if (this.updateRequestState(RequestState.TRANSIENT, RequestState.CONTENT)) {
                    yield true;
                }
                this.abortRequest(exchange);
                yield false;
            }
            default -> false;
        };
    }

    protected boolean someToSuccess(HttpExchange exchange) {
        RequestState current = this.requestState.get();
        return switch (current.ordinal()) {
            case 4, 5 -> {
                if (!exchange.requestComplete(null)) {
                    yield false;
                }
                this.requestState.set(RequestState.QUEUED);
                this.reset();
                HttpRequest request = exchange.getRequest();
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Request success {}", (Object)request);
                }
                request.notifySuccess();
                Result result = exchange.terminateRequest();
                this.terminateRequest(exchange, null, result);
                yield true;
            }
            default -> false;
        };
    }

    private boolean failRequest(Throwable failure) {
        HttpExchange exchange = this.getHttpExchange();
        if (exchange == null) {
            return false;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Request failure {}, response {}", new Object[]{exchange.getRequest(), exchange.getResponse(), failure});
        }
        return exchange.requestComplete(failure);
    }

    private void executeAbort(HttpExchange exchange, Throwable failure) {
        try {
            Executor executor = this.getHttpChannel().getHttpDestination().getHttpClient().getExecutor();
            executor.execute(() -> this.abort(exchange, failure, Promise.noop()));
        }
        catch (RejectedExecutionException x) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Exchange aborted {}", (Object)exchange, (Object)x);
            }
            this.abort(exchange, failure, Promise.noop());
        }
    }

    private void abortRequest(HttpExchange exchange) {
        Throwable failure = this.failure.get();
        HttpRequest request = exchange.getRequest();
        Request.Content content = request.getBody();
        if (content != null) {
            content.fail(failure);
        }
        this.dispose();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Request abort {} {} on {}", new Object[]{request, exchange, this.getHttpChannel(), failure});
        }
        request.notifyFailure(failure);
        Result result = exchange.terminateRequest();
        this.terminateRequest(exchange, failure, result);
    }

    private void terminateRequest(HttpExchange exchange, Throwable failure, Result result) {
        HttpRequest request = exchange.getRequest();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Terminating request {}", (Object)request);
        }
        if (result == null) {
            if (failure != null && exchange.responseComplete(failure)) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Response failure from request {} {}", (Object)request, (Object)exchange);
                }
                this.getHttpChannel().abortResponse(exchange, failure, Promise.noop());
            }
        } else {
            result = this.channel.exchangeTerminating(exchange, result);
            HttpDestination destination = this.getHttpChannel().getHttpDestination();
            boolean ordered = destination.getHttpClient().isStrictEventOrdering();
            if (!ordered) {
                this.channel.exchangeTerminated(exchange, result);
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("Request/Response {}: {}", (Object)(failure == null ? "succeeded" : "failed"), (Object)result);
            }
            exchange.getConversation().getResponseListeners().notifyComplete(result);
            if (ordered) {
                this.channel.exchangeTerminated(exchange, result);
            }
        }
    }

    protected abstract void sendHeaders(HttpExchange var1, ByteBuffer var2, boolean var3, Callback var4);

    protected abstract void sendContent(HttpExchange var1, ByteBuffer var2, boolean var3, Callback var4);

    protected void reset() {
        this.contentSender.reset();
    }

    protected void dispose() {
    }

    public void proceed(HttpExchange exchange, Throwable failure) {
        if (!this.contentSender.expect100) {
            return;
        }
        this.contentSender.expect100 = false;
        if (failure == null) {
            this.contentSender.iterate();
        } else if (this.failRequest(failure)) {
            this.executeAbort(exchange, failure);
        }
    }

    public void abort(HttpExchange exchange, Throwable failure, Promise<Boolean> promise) {
        this.externalAbort(failure, promise);
    }

    private boolean anyToFailure(Throwable failure) {
        boolean abort;
        block1: {
            RequestState current;
            this.failure.compareAndSet(null, failure);
            do {
                if ((current = this.requestState.get()) != RequestState.FAILURE) continue;
                abort = false;
                break block1;
            } while (!this.updateRequestState(current, RequestState.FAILURE));
            abort = current != RequestState.TRANSIENT;
        }
        return abort;
    }

    private void externalAbort(Throwable failure, Promise<Boolean> promise) {
        boolean abort = this.anyToFailure(failure);
        if (abort) {
            this.contentSender.abort = promise;
            this.contentSender.abort(this.failure.get());
        } else {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Concurrent failure: request termination skipped, performed by helpers");
            }
            promise.succeeded(false);
        }
    }

    private void internalAbort(HttpExchange exchange, Throwable failure) {
        this.anyToFailure(failure);
        this.abortRequest(exchange);
    }

    private boolean updateRequestState(RequestState from, RequestState to) {
        boolean updated = this.requestState.compareAndSet(from, to);
        if (!updated && LOG.isDebugEnabled()) {
            LOG.debug("RequestState update failed: {} -> {}: {}", new Object[]{from, to, this.requestState.get()});
        }
        return updated;
    }

    protected String relativize(String path) {
        try {
            String result = path;
            URI uri = URI.create(result);
            if (uri.isAbsolute()) {
                result = uri.getPath();
            }
            return result.isEmpty() ? "/" : result;
        }
        catch (Throwable x) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Could not relativize {}", (Object)path);
            }
            return path;
        }
    }

    public String toString() {
        return String.format("%s@%x(req=%s,failure=%s)", this.getClass().getSimpleName(), this.hashCode(), this.requestState, this.failure);
    }

    private class ContentSender
    extends IteratingCallback {
        private HttpExchange exchange;
        private Content.Chunk chunk;
        private ByteBuffer contentBuffer;
        private boolean expect100;
        private boolean committed;
        private boolean success;
        private boolean complete;
        private Promise<Boolean> abort;

        private ContentSender() {
        }

        @Override
        public boolean reset() {
            this.exchange = null;
            this.chunk = null;
            this.contentBuffer = null;
            this.expect100 = false;
            this.committed = false;
            this.success = false;
            this.complete = false;
            this.abort = null;
            return super.reset();
        }

        @Override
        protected IteratingCallback.Action process() throws Throwable {
            if (this.complete) {
                if (this.success) {
                    HttpSender.this.someToSuccess(this.exchange);
                }
                return IteratingCallback.Action.IDLE;
            }
            HttpRequest request = this.exchange.getRequest();
            Request.Content content = request.getBody();
            if (this.expect100) {
                if (this.committed) {
                    return IteratingCallback.Action.IDLE;
                }
                this.chunk = null;
            } else {
                this.chunk = content.read();
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("Content {} for {}", (Object)this.chunk, (Object)request);
            }
            if (this.chunk == null) {
                if (this.committed) {
                    content.demand(this::iterate);
                    return IteratingCallback.Action.IDLE;
                }
                this.chunk = Content.Chunk.EMPTY;
            }
            if (Content.Chunk.isFailure(this.chunk)) {
                Content.Chunk failure = this.chunk;
                this.chunk = Content.Chunk.next(failure);
                throw failure.getFailure();
            }
            ByteBuffer buffer = this.chunk.getByteBuffer();
            this.contentBuffer = buffer.asReadOnlyBuffer();
            boolean last = this.chunk.isLast();
            if (this.committed) {
                HttpSender.this.sendContent(this.exchange, buffer, last, this);
            } else {
                HttpSender.this.sendHeaders(this.exchange, buffer, last, this);
            }
            return IteratingCallback.Action.SCHEDULED;
        }

        @Override
        protected void onSuccess() {
            boolean proceed = true;
            if (this.committed) {
                if (this.contentBuffer.hasRemaining()) {
                    proceed = HttpSender.this.someToContent(this.exchange, this.contentBuffer);
                }
            } else {
                this.committed = true;
                if (HttpSender.this.headersToCommit(this.exchange)) {
                    if (this.contentBuffer.hasRemaining()) {
                        proceed = HttpSender.this.someToContent(this.exchange, this.contentBuffer);
                    }
                } else {
                    proceed = false;
                }
            }
            boolean last = this.chunk.isLast();
            this.chunk.release();
            this.chunk = null;
            if (proceed) {
                if (last) {
                    this.success = true;
                    this.complete = true;
                } else if (this.expect100 && LOG.isDebugEnabled()) {
                    LOG.debug("Expecting 100 Continue for {}", (Object)this.exchange.getRequest());
                }
            } else {
                this.complete = true;
            }
        }

        @Override
        protected void onCompleteFailure(Throwable x) {
            if (this.chunk != null) {
                this.chunk.release();
                this.chunk = Content.Chunk.next(this.chunk);
            }
            HttpSender.this.failRequest(x);
            HttpSender.this.internalAbort(this.exchange, x);
            Promise<Boolean> promise = this.abort;
            if (promise != null) {
                promise.succeeded(true);
            }
        }

        @Override
        public Invocable.InvocationType getInvocationType() {
            return Invocable.InvocationType.NON_BLOCKING;
        }
    }

    private static enum RequestState {
        TRANSIENT,
        QUEUED,
        BEGIN,
        HEADERS,
        COMMIT,
        CONTENT,
        FAILURE;

    }
}

