/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.data.redis.connection;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Random;
import java.util.TreeMap;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.core.task.AsyncTaskExecutor;
import org.springframework.core.task.SimpleAsyncTaskExecutor;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.ClusterRedirectException;
import org.springframework.data.redis.ClusterStateFailureException;
import org.springframework.data.redis.ExceptionTranslationStrategy;
import org.springframework.data.redis.TooManyClusterRedirectionsException;
import org.springframework.data.redis.connection.ClusterCommandExecutionFailureException;
import org.springframework.data.redis.connection.ClusterNodeResourceProvider;
import org.springframework.data.redis.connection.ClusterTopology;
import org.springframework.data.redis.connection.ClusterTopologyProvider;
import org.springframework.data.redis.connection.RedisClusterNode;
import org.springframework.data.redis.connection.util.ByteArraySet;
import org.springframework.data.redis.connection.util.ByteArrayWrapper;
import org.springframework.lang.Nullable;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

public class ClusterCommandExecutor
implements DisposableBean {
    private int maxRedirects = 5;
    private final AsyncTaskExecutor executor;
    private final ClusterNodeResourceProvider resourceProvider;
    private final ClusterTopologyProvider topologyProvider;
    private final ExceptionTranslationStrategy exceptionTranslationStrategy;

    public ClusterCommandExecutor(ClusterTopologyProvider topologyProvider, ClusterNodeResourceProvider resourceProvider, ExceptionTranslationStrategy exceptionTranslation) {
        this(topologyProvider, resourceProvider, exceptionTranslation, new SimpleAsyncTaskExecutor());
    }

    public ClusterCommandExecutor(ClusterTopologyProvider topologyProvider, ClusterNodeResourceProvider resourceProvider, ExceptionTranslationStrategy exceptionTranslation, @Nullable AsyncTaskExecutor executor) {
        Assert.notNull((Object)topologyProvider, "ClusterTopologyProvider must not be null");
        Assert.notNull((Object)resourceProvider, "ClusterNodeResourceProvider must not be null");
        Assert.notNull((Object)exceptionTranslation, "ExceptionTranslationStrategy must not be null");
        this.topologyProvider = topologyProvider;
        this.resourceProvider = resourceProvider;
        this.exceptionTranslationStrategy = exceptionTranslation;
        this.executor = executor != null ? executor : new SimpleAsyncTaskExecutor();
    }

    public <T> NodeResult<T> executeCommandOnArbitraryNode(ClusterCommandCallback<?, T> commandCallback) {
        Assert.notNull(commandCallback, "ClusterCommandCallback must not be null");
        ArrayList<RedisClusterNode> nodes = new ArrayList<RedisClusterNode>(this.getClusterTopology().getActiveNodes());
        RedisClusterNode arbitraryNode = (RedisClusterNode)nodes.get(new Random().nextInt(nodes.size()));
        return this.executeCommandOnSingleNode(commandCallback, arbitraryNode);
    }

    public <S, T> NodeResult<T> executeCommandOnSingleNode(ClusterCommandCallback<S, T> commandCallback, RedisClusterNode node) {
        return this.executeCommandOnSingleNode(commandCallback, node, 0);
    }

    private <S, T> NodeResult<T> executeCommandOnSingleNode(ClusterCommandCallback<S, T> commandCallback, RedisClusterNode node, int redirectCount) {
        Assert.notNull(commandCallback, "ClusterCommandCallback must not be null");
        Assert.notNull((Object)node, "RedisClusterNode must not be null");
        if (redirectCount > this.maxRedirects) {
            String message = String.format("Cannot follow Cluster Redirects over more than %s legs; Consider increasing the number of redirects to follow; Current value is: %s.", redirectCount, this.maxRedirects);
            throw new TooManyClusterRedirectionsException(message);
        }
        RedisClusterNode nodeToUse = this.lookupNode(node);
        Object client = this.resourceProvider.getResourceForSpecificNode(nodeToUse);
        Assert.notNull(client, "Could not acquire resource for node; Is your cluster info up to date");
        try {
            NodeResult<T> nodeResult = new NodeResult<T>(node, commandCallback.doInCluster(client));
            return nodeResult;
        }
        catch (RuntimeException ex) {
            DataAccessException translatedException = this.convertToDataAccessException(ex);
            if (translatedException instanceof ClusterRedirectException) {
                ClusterRedirectException clusterRedirectException = (ClusterRedirectException)translatedException;
                String targetHost = clusterRedirectException.getTargetHost();
                int targetPort = clusterRedirectException.getTargetPort();
                RedisClusterNode clusterNode = this.topologyProvider.getTopology().lookup(targetHost, targetPort);
                NodeResult<T> nodeResult = this.executeCommandOnSingleNode(commandCallback, clusterNode, redirectCount + 1);
                return nodeResult;
            }
            throw translatedException != null ? translatedException : ex;
        }
        finally {
            this.resourceProvider.returnResourceForSpecificNode(nodeToUse, client);
        }
    }

    private RedisClusterNode lookupNode(RedisClusterNode node) {
        try {
            return this.topologyProvider.getTopology().lookup(node);
        }
        catch (ClusterStateFailureException ex) {
            throw new IllegalArgumentException(String.format("Node %s is unknown to cluster", node), ex);
        }
    }

    public <S, T> MultiNodeResult<T> executeCommandOnAllNodes(ClusterCommandCallback<S, T> commandCallback) {
        return this.executeCommandAsyncOnNodes(commandCallback, this.getClusterTopology().getActiveMasterNodes());
    }

    public <S, T> MultiNodeResult<T> executeCommandAsyncOnNodes(ClusterCommandCallback<S, T> commandCallback, Iterable<RedisClusterNode> nodes) {
        Assert.notNull(commandCallback, "Callback must not be null");
        Assert.notNull(nodes, "Nodes must not be null");
        ClusterTopology topology = this.topologyProvider.getTopology();
        ArrayList<RedisClusterNode> resolvedRedisClusterNodes = new ArrayList<RedisClusterNode>();
        for (RedisClusterNode node : nodes) {
            try {
                resolvedRedisClusterNodes.add(topology.lookup(node));
            }
            catch (ClusterStateFailureException ex) {
                throw new IllegalArgumentException(String.format("Node %s is unknown to cluster", node), ex);
            }
        }
        LinkedHashMap<NodeExecution, Future<NodeResult<T>>> futures = new LinkedHashMap<NodeExecution, Future<NodeResult<T>>>();
        for (RedisClusterNode node : resolvedRedisClusterNodes) {
            Callable<NodeResult> nodeCommandExecution = () -> this.executeCommandOnSingleNode(commandCallback, node);
            futures.put(new NodeExecution(node), this.executor.submit(nodeCommandExecution));
        }
        return this.collectResults(futures);
    }

    <T> MultiNodeResult<T> collectResults(Map<NodeExecution, Future<NodeResult<T>>> futures) {
        MultiNodeResult<T> result = new MultiNodeResult<T>();
        NodeExceptionCollector exceptionCollector = new NodeExceptionCollector();
        block4: while (!futures.isEmpty()) {
            Iterator<Map.Entry<NodeExecution, Future<NodeResult<T>>>> entryIterator = futures.entrySet().iterator();
            while (entryIterator.hasNext()) {
                Map.Entry<NodeExecution, Future<NodeResult<T>>> entry = entryIterator.next();
                NodeExecution nodeExecution = entry.getKey();
                Future<NodeResult<T>> futureNodeResult = entry.getValue();
                try {
                    NodeResult<T> nodeResult = futureNodeResult.get(10L, TimeUnit.MICROSECONDS);
                    if (nodeExecution.isPositional()) {
                        result.add(nodeExecution.getPositionalKey(), nodeResult);
                    } else {
                        result.add(nodeResult);
                    }
                    entryIterator.remove();
                }
                catch (ExecutionException ex) {
                    entryIterator.remove();
                    exceptionCollector.addException(nodeExecution, ex.getCause());
                }
                catch (TimeoutException ex) {
                }
                catch (InterruptedException ex) {
                    Thread.currentThread().interrupt();
                    exceptionCollector.addException(nodeExecution, ex);
                    break block4;
                }
            }
        }
        if (exceptionCollector.hasExceptions()) {
            throw new ClusterCommandExecutionFailureException(exceptionCollector.getExceptions());
        }
        return result;
    }

    public <S, T> MultiNodeResult<T> executeMultiKeyCommand(MultiKeyClusterCommandCallback<S, T> commandCallback, Iterable<byte[]> keys) {
        HashMap<RedisClusterNode, PositionalKeys> nodeKeyMap = new HashMap<RedisClusterNode, PositionalKeys>();
        int index = 0;
        for (byte[] key : keys) {
            for (RedisClusterNode node : this.getClusterTopology().getKeyServingNodes(key)) {
                nodeKeyMap.computeIfAbsent(node, val -> PositionalKeys.empty()).append(PositionalKey.of(key, index++));
            }
        }
        LinkedHashMap<NodeExecution, Future<NodeResult<T>>> futures = new LinkedHashMap<NodeExecution, Future<NodeResult<T>>>();
        for (Map.Entry entry : nodeKeyMap.entrySet()) {
            if (!((RedisClusterNode)entry.getKey()).isMaster()) continue;
            for (PositionalKey key : (PositionalKeys)entry.getValue()) {
                futures.put(new NodeExecution((RedisClusterNode)entry.getKey(), key), this.executor.submit(() -> this.executeMultiKeyCommandOnSingleNode(commandCallback, (RedisClusterNode)entry.getKey(), key.getBytes())));
            }
        }
        return this.collectResults(futures);
    }

    private <S, T> NodeResult<T> executeMultiKeyCommandOnSingleNode(MultiKeyClusterCommandCallback<S, T> commandCallback, RedisClusterNode node, byte[] key) {
        Assert.notNull(commandCallback, "MultiKeyCommandCallback must not be null");
        Assert.notNull((Object)node, "RedisClusterNode must not be null");
        Assert.notNull((Object)key, "Keys for execution must not be null");
        Object client = this.resourceProvider.getResourceForSpecificNode(node);
        Assert.notNull(client, "Could not acquire resource for node; Is your cluster info up to date");
        try {
            NodeResult<T> nodeResult = new NodeResult<T>(node, commandCallback.doInCluster(client, key), key);
            return nodeResult;
        }
        catch (RuntimeException ex) {
            DataAccessException translatedException = this.convertToDataAccessException(ex);
            throw translatedException != null ? translatedException : ex;
        }
        finally {
            this.resourceProvider.returnResourceForSpecificNode(node, client);
        }
    }

    private ClusterTopology getClusterTopology() {
        return this.topologyProvider.getTopology();
    }

    @Nullable
    private DataAccessException convertToDataAccessException(Exception cause) {
        return this.exceptionTranslationStrategy.translate(cause);
    }

    public void setMaxRedirects(int maxRedirects) {
        this.maxRedirects = maxRedirects;
    }

    @Override
    public void destroy() throws Exception {
        ClusterNodeResourceProvider clusterNodeResourceProvider = this.resourceProvider;
        if (clusterNodeResourceProvider instanceof DisposableBean) {
            DisposableBean disposableBean = (DisposableBean)((Object)clusterNodeResourceProvider);
            disposableBean.destroy();
        }
    }

    public static interface ClusterCommandCallback<T, S> {
        public S doInCluster(T var1);
    }

    public static class NodeResult<T> {
        private final RedisClusterNode node;
        private final ByteArrayWrapper key;
        @Nullable
        private final T value;

        public NodeResult(RedisClusterNode node, @Nullable T value) {
            this(node, value, new byte[0]);
        }

        public NodeResult(RedisClusterNode node, @Nullable T value, byte[] key) {
            this.node = node;
            this.key = new ByteArrayWrapper(key);
            this.value = value;
        }

        public RedisClusterNode getNode() {
            return this.node;
        }

        public byte[] getKey() {
            return this.key.getArray();
        }

        @Nullable
        public T getValue() {
            return this.value;
        }

        @Nullable
        public <U> U mapValue(Function<? super T, ? extends U> mapper) {
            Assert.notNull(mapper, "Mapper function must not be null");
            return mapper.apply(this.getValue());
        }

        public boolean equals(@Nullable Object obj) {
            if (obj == this) {
                return true;
            }
            if (!(obj instanceof NodeResult)) {
                return false;
            }
            NodeResult that = (NodeResult)obj;
            return ObjectUtils.nullSafeEquals(this.getNode(), that.getNode()) && Objects.equals(this.key, that.key) && Objects.equals(this.getValue(), that.getValue());
        }

        public int hashCode() {
            int hashValue = 17;
            hashValue = 37 * hashValue + ObjectUtils.nullSafeHashCode(this.getNode());
            hashValue = 37 * hashValue + ObjectUtils.nullSafeHashCode(this.key);
            hashValue = 37 * hashValue + ObjectUtils.nullSafeHashCode(this.getValue());
            return hashValue;
        }
    }

    public static class MultiNodeResult<T> {
        List<NodeResult<T>> nodeResults = new ArrayList<NodeResult<T>>();
        Map<PositionalKey, NodeResult<T>> positionalResults = new LinkedHashMap<PositionalKey, NodeResult<T>>();

        private void add(NodeResult<T> result) {
            this.nodeResults.add(result);
        }

        private void add(PositionalKey key, NodeResult<T> result) {
            this.positionalResults.put(key, result);
            this.add(result);
        }

        public List<NodeResult<T>> getResults() {
            return Collections.unmodifiableList(this.nodeResults);
        }

        public List<T> resultsAsList() {
            return this.toList(this.nodeResults);
        }

        public List<T> resultsAsListSortBy(byte[] ... keys) {
            if (this.positionalResults.isEmpty()) {
                ArrayList<NodeResult<T>> clone = new ArrayList<NodeResult<T>>(this.nodeResults);
                clone.sort(new ResultByReferenceKeyPositionComparator(keys));
                return this.toList(clone);
            }
            TreeMap<PositionalKey, NodeResult<T>> result = new TreeMap<PositionalKey, NodeResult<T>>(new ResultByKeyPositionComparator(keys));
            result.putAll(this.positionalResults);
            return result.values().stream().map(tNodeResult -> tNodeResult.value).collect(Collectors.toList());
        }

        @Nullable
        public T getFirstNonNullNotEmptyOrDefault(@Nullable T returnValue) {
            for (NodeResult<T> nodeResult : this.nodeResults) {
                if (nodeResult.getValue() == null) continue;
                if (nodeResult.getValue() instanceof Map) {
                    if (!CollectionUtils.isEmpty((Map)nodeResult.getValue())) continue;
                    return nodeResult.getValue();
                }
                if (nodeResult.getValue() instanceof Collection && CollectionUtils.isEmpty((Collection)nodeResult.getValue())) {
                    return nodeResult.getValue();
                }
                return nodeResult.getValue();
            }
            return returnValue;
        }

        private List<T> toList(Collection<NodeResult<T>> source) {
            ArrayList<T> result = new ArrayList<T>();
            for (NodeResult<T> nodeResult : source) {
                result.add(nodeResult.getValue());
            }
            return result;
        }

        private static class ResultByReferenceKeyPositionComparator
        implements Comparator<NodeResult<?>> {
            private final List<ByteArrayWrapper> reference;

            ResultByReferenceKeyPositionComparator(byte[] ... keys) {
                this.reference = new ArrayList<ByteArrayWrapper>(new ByteArraySet(Arrays.asList(keys)));
            }

            @Override
            public int compare(NodeResult<?> o1, NodeResult<?> o2) {
                return Integer.compare(this.reference.indexOf(o1.key), this.reference.indexOf(o2.key));
            }
        }

        private static class ResultByKeyPositionComparator
        implements Comparator<PositionalKey> {
            private final PositionalKeys reference;

            ResultByKeyPositionComparator(byte[] ... keys) {
                this.reference = PositionalKeys.of(keys);
            }

            @Override
            public int compare(PositionalKey o1, PositionalKey o2) {
                return Integer.compare(this.reference.indexOf(o1), this.reference.indexOf(o2));
            }
        }
    }

    static class NodeExecution {
        private final RedisClusterNode node;
        @Nullable
        private final PositionalKey positionalKey;

        NodeExecution(RedisClusterNode node) {
            this(node, null);
        }

        NodeExecution(RedisClusterNode node, @Nullable PositionalKey positionalKey) {
            this.node = node;
            this.positionalKey = positionalKey;
        }

        RedisClusterNode getNode() {
            return this.node;
        }

        PositionalKey getPositionalKey() {
            return this.positionalKey;
        }

        boolean isPositional() {
            return this.positionalKey != null;
        }
    }

    private class NodeExceptionCollector {
        private final Map<RedisClusterNode, Throwable> exceptions = new HashMap<RedisClusterNode, Throwable>();

        private NodeExceptionCollector() {
        }

        public boolean hasExceptions() {
            return !this.exceptions.isEmpty();
        }

        public void addException(NodeExecution execution, Throwable throwable) {
            Throwable throwable2;
            if (throwable instanceof Exception) {
                Exception e = (Exception)throwable;
                throwable2 = ClusterCommandExecutor.this.convertToDataAccessException(e);
            } else {
                throwable2 = throwable;
            }
            Throwable translated = throwable2;
            Throwable resolvedException = translated != null ? translated : throwable;
            this.exceptions.putIfAbsent(execution.getNode(), resolvedException);
        }

        public List<? extends Throwable> getExceptions() {
            return new ArrayList<Throwable>(this.exceptions.values());
        }
    }

    private static class PositionalKey {
        private final ByteArrayWrapper key;
        private final int position;

        private PositionalKey(ByteArrayWrapper key, int position) {
            this.key = key;
            this.position = position;
        }

        static PositionalKey of(byte[] key, int index) {
            return new PositionalKey(new ByteArrayWrapper(key), index);
        }

        byte[] getBytes() {
            return this.getKey().getArray();
        }

        public ByteArrayWrapper getKey() {
            return this.key;
        }

        public int getPosition() {
            return this.position;
        }

        public boolean equals(@Nullable Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof PositionalKey)) {
                return false;
            }
            PositionalKey that = (PositionalKey)obj;
            return this.getPosition() == that.getPosition() && ObjectUtils.nullSafeEquals(this.getKey(), that.getKey());
        }

        public int hashCode() {
            int result = ObjectUtils.nullSafeHashCode(this.getKey());
            result = 31 * result + ObjectUtils.nullSafeHashCode(this.getPosition());
            return result;
        }
    }

    private static class PositionalKeys
    implements Iterable<PositionalKey> {
        private final List<PositionalKey> keys;

        private PositionalKeys(List<PositionalKey> keys) {
            this.keys = keys;
        }

        static PositionalKeys empty() {
            return new PositionalKeys(new ArrayList<PositionalKey>());
        }

        static PositionalKeys of(byte[] ... keys) {
            ArrayList<PositionalKey> result = new ArrayList<PositionalKey>(keys.length);
            for (int i = 0; i < keys.length; ++i) {
                result.add(PositionalKey.of(keys[i], i));
            }
            return new PositionalKeys(result);
        }

        static PositionalKeys of(PositionalKey ... keys) {
            PositionalKeys result = PositionalKeys.empty();
            result.append(keys);
            return result;
        }

        void append(PositionalKey ... keys) {
            this.keys.addAll(Arrays.asList(keys));
        }

        int indexOf(PositionalKey key) {
            return this.keys.indexOf(key);
        }

        @Override
        public Iterator<PositionalKey> iterator() {
            return this.keys.iterator();
        }
    }

    public static interface MultiKeyClusterCommandCallback<T, S> {
        public S doInCluster(T var1, byte[] var2);
    }
}

