/*
 * Copyright (C) 2006-2024 Talend Inc. - www.talend.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 */
package com.talend.excel.xssf.event;

import java.util.LinkedList;

public class DataBufferCache {

    // private static DataBufferCache instance = new DataBufferCache();

    private long bufferSize = 5000;

    private boolean isEnd = false;

    private int bufferNum = 10;

    private final LinkedList<Buffer> buffers = new LinkedList<>();

    private Buffer currentReadBuff = null;

    private Buffer currentWriteBuff = null;

    private DataBufferCache() {
    }

    public static DataBufferCache getInstance() {
        return new DataBufferCache();
    }

    public void setIsEnd() {
        synchronized (buffers) {
            try {
                if (!isEnd && currentWriteBuff != null) {
                    buffers.add(currentWriteBuff);
                }
                this.isEnd = true;
            } finally {
                this.buffers.notifyAll();
            }
        }
    }

    public void setBufferNum(int num) {
        if (num <= 0) {
            return;
        }
        this.bufferNum = num;
    }

    public void setBufferSize(long size) {
        if (size <= 0) {
            return;
        }
        this.bufferSize = size;
    }

    public boolean hasData() {
        getCurrReadBuffer();
        if (currentReadBuff == null) {
            return false;
        }
        return this.currentReadBuff.hasNext();
    }

    public SheetRow readData() {
        return this.currentReadBuff.Next();
    }

    private void getCurrReadBuffer() {
        if (currentReadBuff != null && currentReadBuff.hasNext()) {
            return;
        }
        synchronized (this.buffers) {
            try {
                while (!this.isEnd && buffers.isEmpty()) {
                    try {
                        buffers.wait();
                    } catch (InterruptedException ex) {
                        System.err.println(ex.getMessage());
                    }
                }
                if (!buffers.isEmpty()) {
                    currentReadBuff = buffers.remove();
                }
            } finally {
                this.buffers.notifyAll();
            }
        }
    }

    public void writeData(SheetRow record) {
        if (currentWriteBuff == null) {
            currentWriteBuff = new Buffer();
        }
        if (currentWriteBuff.size() < this.bufferSize) {
            currentWriteBuff.add(record);
        } else {
            currentWriteBuff.add(record);
            synchronized (buffers) {
                try {
                    while (buffers.size() > bufferNum) {
                        try {
                            buffers.wait();
                        } catch (InterruptedException ex) {
                            System.err.println(ex.getMessage());
                        }
                    }
                    if (currentWriteBuff.size() > 0) {
                        this.buffers.add(currentWriteBuff);
                    }
                    currentWriteBuff = null;
                } finally {
                    this.buffers.notifyAll();
                }
            }

        }

    }

    public void notifyErrorOccurred() {
        setIsEnd();
    }

    static class Buffer {

        private LinkedList<SheetRow> buffer = null;

        private boolean isForRead = false; // false for write, true for read

        public Buffer() {
            buffer = createNewBuffer();
        }

        public void setIsForRead(boolean isForRead) {
            this.isForRead = isForRead;
        }

        public boolean isForRead() {
            return this.isForRead;
        }

        public boolean hasNext() {
            return !buffer.isEmpty();
        }

        public SheetRow Next() {
            return buffer.remove();
        }

        public void add(SheetRow o) {
            this.buffer.add(o);
        }

        public int size() {
            return this.buffer.size();
        }

        public void clear() {
            this.buffer.clear();
            buffer = null;
        }

        private LinkedList<SheetRow> createNewBuffer() {
            return new LinkedList<>();
        }
    }
}
