/*
 * Copyright (C) 2006-2024 Talend Inc. - www.talend.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 */
package com.talend.excel.xssf.event;

import org.apache.poi.xssf.usermodel.XSSFComment;

import lombok.Setter;
import lombok.extern.log4j.Log4j;

/**
 * created by wwang on 2012-9-27 Detailled comment
 *
 */
@Log4j
public class DefaultTalendSheetContentsHandler implements TalendXSSFSheetXMLHandler.TalendSheetContentsHandler {

    private DataBufferCache cache = null;

    private SheetRow row = null;

    private boolean stop = false;

    private int currentColumnIndex = -1;

    private int lastColumnIndex = -1;

    @Setter
    private String currentSheetName;

    public DefaultTalendSheetContentsHandler(DataBufferCache cache) {
        this.cache = cache;
    }

    @Override
    public void startRow(int rowNum) {
        if (currentSheetName == null) {
            log.warn("Current sheet name is null.");
        }
        row = new SheetRow(currentSheetName);
    }

    @Override
    public void endRow(int rowNum) {
        cache.writeData(row);
        row = null;
        // when each row end ,reset lastColumnIndex
        lastColumnIndex = -1;
        if (stop) {
            throw new EnoughDataException("Get enough data,now stop the xml parse action");
        }
    }

    @Override
    public void cell(String cellReference, String formattedValue, XSSFComment comment) {
        checkHasNullValue(cellReference);
        row.add(formattedValue);
    }

    @Override
    public void headerFooter(String text, boolean isHeader, String tagName) {
        /* NOP */
    }

    @Override
    public void endSheet() {
        /* NOP */
    }

    public void stop() {
        this.stop = true;
    }

    /**
     * when currentCellReferecnce is "A1" ,and next currentCellReferecnce is "C1",we need add a null value for "B1"
     */
    private void checkHasNullValue(String cellReference) {
        /*
         * Avoid the NPE for old format.
         * When the row is like that:
         * <row><c t="inlineStr"><is><t>Cell text</t></is></c></row>
         * the cellReference (comes from XSSFSheetXMLHandler::startElement) is null
         * because it's taken from the <c> attribute 'r', for new format the same row is looks like this
         * <row r="1" collapsed="false" outlineLevel="0" customHeight="false" hidden="false" ht="12.8"
         * customFormat="false">
         * <c r="A1" t="s" s="0"><v>0</v></c>
         * </row>
         * and the cellReference is "A1"
         */
        if (cellReference == null)
            return;
        currentColumnIndex = ColumnUtil.calculateIndexOfColumn(cellReference);
        // Might be the empty string.
        for (int i = lastColumnIndex; i < currentColumnIndex - 1; i++) {
            row.add(null);
        }
        // Update column
        if (currentColumnIndex > -1) {
            lastColumnIndex = currentColumnIndex;
        }
    }

}
