/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.gateway;

import com.carrotsearch.hppc.cursors.ObjectCursor;
import java.util.concurrent.atomic.AtomicBoolean;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterService;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.ClusterStateUpdateTask;
import org.elasticsearch.cluster.block.ClusterBlock;
import org.elasticsearch.cluster.block.ClusterBlockLevel;
import org.elasticsearch.cluster.block.ClusterBlocks;
import org.elasticsearch.cluster.metadata.IndexMetaData;
import org.elasticsearch.cluster.metadata.MetaData;
import org.elasticsearch.cluster.node.DiscoveryNodes;
import org.elasticsearch.cluster.routing.RoutingTable;
import org.elasticsearch.cluster.routing.allocation.AllocationService;
import org.elasticsearch.cluster.routing.allocation.RoutingAllocation;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.discovery.DiscoveryService;
import org.elasticsearch.gateway.Gateway;
import org.elasticsearch.rest.RestStatus;
import org.elasticsearch.threadpool.ThreadPool;

public class GatewayService
extends AbstractLifecycleComponent<GatewayService>
implements ClusterStateListener {
    public static final ClusterBlock STATE_NOT_RECOVERED_BLOCK = new ClusterBlock(1, "state not recovered / initialized", true, true, RestStatus.SERVICE_UNAVAILABLE, ClusterBlockLevel.ALL);
    public static final TimeValue DEFAULT_RECOVER_AFTER_TIME_IF_EXPECTED_NODES_IS_SET = TimeValue.timeValueMinutes(5L);
    private final Gateway gateway;
    private final ThreadPool threadPool;
    private final AllocationService allocationService;
    private final ClusterService clusterService;
    private final DiscoveryService discoveryService;
    private final TimeValue recoverAfterTime;
    private final int recoverAfterNodes;
    private final int expectedNodes;
    private final int recoverAfterDataNodes;
    private final int expectedDataNodes;
    private final int recoverAfterMasterNodes;
    private final int expectedMasterNodes;
    private final AtomicBoolean recovered = new AtomicBoolean();
    private final AtomicBoolean scheduledRecovery = new AtomicBoolean();

    @Inject
    public GatewayService(Settings settings, Gateway gateway, AllocationService allocationService, ClusterService clusterService, DiscoveryService discoveryService, ThreadPool threadPool) {
        super(settings);
        this.gateway = gateway;
        this.allocationService = allocationService;
        this.clusterService = clusterService;
        this.discoveryService = discoveryService;
        this.threadPool = threadPool;
        this.expectedNodes = this.settings.getAsInt("gateway.expected_nodes", (Integer)-1);
        this.expectedDataNodes = this.settings.getAsInt("gateway.expected_data_nodes", (Integer)-1);
        this.expectedMasterNodes = this.settings.getAsInt("gateway.expected_master_nodes", (Integer)-1);
        TimeValue defaultRecoverAfterTime = null;
        if (this.expectedNodes >= 0 || this.expectedDataNodes >= 0 || this.expectedMasterNodes >= 0) {
            defaultRecoverAfterTime = DEFAULT_RECOVER_AFTER_TIME_IF_EXPECTED_NODES_IS_SET;
        }
        this.recoverAfterTime = this.settings.getAsTime("gateway.recover_after_time", defaultRecoverAfterTime);
        this.recoverAfterNodes = this.settings.getAsInt("gateway.recover_after_nodes", (Integer)-1);
        this.recoverAfterDataNodes = this.settings.getAsInt("gateway.recover_after_data_nodes", (Integer)-1);
        this.recoverAfterMasterNodes = this.settings.getAsInt("gateway.recover_after_master_nodes", settings.getAsInt("discovery.zen.minimum_master_nodes", (Integer)-1));
        this.clusterService.addInitialStateBlock(STATE_NOT_RECOVERED_BLOCK);
    }

    @Override
    protected void doStart() {
        this.clusterService.addLast(this);
        this.clusterService.submitStateUpdateTask("gateway_initial_state_recovery", new ClusterStateUpdateTask(){

            @Override
            public ClusterState execute(ClusterState currentState) throws Exception {
                GatewayService.this.checkStateMeetsSettingsAndMaybeRecover(currentState);
                return currentState;
            }

            @Override
            public boolean runOnlyOnMaster() {
                return false;
            }

            @Override
            public void onFailure(String source, Throwable t) {
                GatewayService.this.logger.warn("unexpected failure while checking if state can be recovered. another attempt will be made with the next cluster state change", t, new Object[0]);
            }
        });
    }

    @Override
    protected void doStop() {
        this.clusterService.remove(this);
    }

    @Override
    protected void doClose() {
    }

    @Override
    public void clusterChanged(ClusterChangedEvent event) {
        if (this.lifecycle.stoppedOrClosed()) {
            return;
        }
        this.checkStateMeetsSettingsAndMaybeRecover(event.state());
    }

    protected void checkStateMeetsSettingsAndMaybeRecover(ClusterState state) {
        if (!state.nodes().localNodeMaster()) {
            return;
        }
        if (!state.blocks().hasGlobalBlock(STATE_NOT_RECOVERED_BLOCK)) {
            return;
        }
        DiscoveryNodes nodes = state.nodes();
        if (state.blocks().hasGlobalBlock(this.discoveryService.getNoMasterBlock())) {
            this.logger.debug("not recovering from gateway, no master elected yet", new Object[0]);
        } else if (this.recoverAfterNodes != -1 && nodes.masterAndDataNodes().size() < this.recoverAfterNodes) {
            this.logger.debug("not recovering from gateway, nodes_size (data+master) [" + nodes.masterAndDataNodes().size() + "] < recover_after_nodes [" + this.recoverAfterNodes + "]", new Object[0]);
        } else if (this.recoverAfterDataNodes != -1 && nodes.dataNodes().size() < this.recoverAfterDataNodes) {
            this.logger.debug("not recovering from gateway, nodes_size (data) [" + nodes.dataNodes().size() + "] < recover_after_data_nodes [" + this.recoverAfterDataNodes + "]", new Object[0]);
        } else if (this.recoverAfterMasterNodes != -1 && nodes.masterNodes().size() < this.recoverAfterMasterNodes) {
            this.logger.debug("not recovering from gateway, nodes_size (master) [" + nodes.masterNodes().size() + "] < recover_after_master_nodes [" + this.recoverAfterMasterNodes + "]", new Object[0]);
        } else {
            String reason;
            boolean enforceRecoverAfterTime;
            if (this.expectedNodes == -1 && this.expectedMasterNodes == -1 && this.expectedDataNodes == -1) {
                enforceRecoverAfterTime = true;
                reason = "recover_after_time was set to [" + this.recoverAfterTime + "]";
            } else {
                enforceRecoverAfterTime = false;
                reason = "";
                if (this.expectedNodes != -1 && nodes.masterAndDataNodes().size() < this.expectedNodes) {
                    enforceRecoverAfterTime = true;
                    reason = "expecting [" + this.expectedNodes + "] nodes, but only have [" + nodes.masterAndDataNodes().size() + "]";
                } else if (this.expectedDataNodes != -1 && nodes.dataNodes().size() < this.expectedDataNodes) {
                    enforceRecoverAfterTime = true;
                    reason = "expecting [" + this.expectedDataNodes + "] data nodes, but only have [" + nodes.dataNodes().size() + "]";
                } else if (this.expectedMasterNodes != -1 && nodes.masterNodes().size() < this.expectedMasterNodes) {
                    enforceRecoverAfterTime = true;
                    reason = "expecting [" + this.expectedMasterNodes + "] master nodes, but only have [" + nodes.masterNodes().size() + "]";
                }
            }
            this.performStateRecovery(enforceRecoverAfterTime, reason);
        }
    }

    private void performStateRecovery(boolean enforceRecoverAfterTime, String reason) {
        final GatewayRecoveryListener recoveryListener = new GatewayRecoveryListener();
        if (enforceRecoverAfterTime && this.recoverAfterTime != null) {
            if (this.scheduledRecovery.compareAndSet(false, true)) {
                this.logger.info("delaying initial state recovery for [{}]. {}", this.recoverAfterTime, reason);
                this.threadPool.schedule(this.recoverAfterTime, "generic", new Runnable(){

                    @Override
                    public void run() {
                        if (GatewayService.this.recovered.compareAndSet(false, true)) {
                            GatewayService.this.logger.info("recover_after_time [{}] elapsed. performing state recovery...", GatewayService.this.recoverAfterTime);
                            GatewayService.this.gateway.performStateRecovery(recoveryListener);
                        }
                    }
                });
            }
        } else if (this.recovered.compareAndSet(false, true)) {
            this.threadPool.generic().execute(new Runnable(){

                @Override
                public void run() {
                    GatewayService.this.gateway.performStateRecovery(recoveryListener);
                }
            });
        }
    }

    public TimeValue recoverAfterTime() {
        return this.recoverAfterTime;
    }

    class GatewayRecoveryListener
    implements Gateway.GatewayStateRecoveredListener {
        GatewayRecoveryListener() {
        }

        @Override
        public void onSuccess(final ClusterState recoveredState) {
            GatewayService.this.logger.trace("successful state recovery, importing cluster state...", new Object[0]);
            GatewayService.this.clusterService.submitStateUpdateTask("local-gateway-elected-state", new ClusterStateUpdateTask(){

                @Override
                public ClusterState execute(ClusterState currentState) {
                    assert (currentState.metaData().indices().isEmpty());
                    ClusterBlocks.Builder blocks = ClusterBlocks.builder().blocks(currentState.blocks()).blocks(recoveredState.blocks()).removeGlobalBlock(STATE_NOT_RECOVERED_BLOCK);
                    MetaData.Builder metaDataBuilder = MetaData.builder(recoveredState.metaData());
                    metaDataBuilder.generateClusterUuidIfNeeded();
                    if (recoveredState.metaData().settings().getAsBoolean("cluster.blocks.read_only", (Boolean)false).booleanValue() || currentState.metaData().settings().getAsBoolean("cluster.blocks.read_only", (Boolean)false).booleanValue()) {
                        blocks.addGlobalBlock(MetaData.CLUSTER_READ_ONLY_BLOCK);
                    }
                    for (IndexMetaData indexMetaData : recoveredState.metaData()) {
                        metaDataBuilder.put(indexMetaData, false);
                        blocks.addBlocks(indexMetaData);
                    }
                    ClusterState updatedState = ClusterState.builder(currentState).blocks(blocks).metaData(metaDataBuilder).build();
                    RoutingTable.Builder routingTableBuilder = RoutingTable.builder(updatedState.routingTable());
                    for (ObjectCursor<IndexMetaData> objectCursor : updatedState.metaData().indices().values()) {
                        routingTableBuilder.addAsRecovery((IndexMetaData)objectCursor.value);
                    }
                    routingTableBuilder.version(0L);
                    RoutingAllocation.Result routingResult = GatewayService.this.allocationService.reroute(ClusterState.builder(updatedState).routingTable(routingTableBuilder.build()).build(), "state recovered");
                    return ClusterState.builder(updatedState).routingResult(routingResult).build();
                }

                @Override
                public void onFailure(String source, Throwable t) {
                    GatewayService.this.logger.error("unexpected failure during [{}]", t, source);
                    GatewayRecoveryListener.this.onFailure("failed to updated cluster state");
                }

                @Override
                public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
                    GatewayService.this.logger.info("recovered [{}] indices into cluster_state", newState.metaData().indices().size());
                }
            });
        }

        @Override
        public void onFailure(String message) {
            GatewayService.this.recovered.set(false);
            GatewayService.this.scheduledRecovery.set(false);
            GatewayService.this.logger.info("metadata state not restored, reason: {}", message);
        }
    }
}

