/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.blobstore;

import com.google.common.io.ByteStreams;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.NoSuchFileException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import org.apache.lucene.store.RateLimiter;
import org.elasticsearch.ElasticsearchParseException;
import org.elasticsearch.Version;
import org.elasticsearch.cluster.metadata.IndexMetaData;
import org.elasticsearch.cluster.metadata.MetaData;
import org.elasticsearch.cluster.metadata.SnapshotId;
import org.elasticsearch.common.ParseFieldMatcher;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.blobstore.BlobContainer;
import org.elasticsearch.common.blobstore.BlobMetaData;
import org.elasticsearch.common.blobstore.BlobPath;
import org.elasticsearch.common.blobstore.BlobStore;
import org.elasticsearch.common.bytes.BytesArray;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.compress.NotXContentException;
import org.elasticsearch.common.io.stream.BytesStreamOutput;
import org.elasticsearch.common.io.stream.OutputStreamStreamOutput;
import org.elasticsearch.common.metrics.CounterMetric;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.common.xcontent.XContentFactory;
import org.elasticsearch.common.xcontent.XContentHelper;
import org.elasticsearch.common.xcontent.XContentParser;
import org.elasticsearch.common.xcontent.XContentType;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.snapshots.IndexShardRepository;
import org.elasticsearch.index.snapshots.blobstore.BlobStoreIndexShardRepository;
import org.elasticsearch.repositories.Repository;
import org.elasticsearch.repositories.RepositoryException;
import org.elasticsearch.repositories.RepositorySettings;
import org.elasticsearch.repositories.RepositoryVerificationException;
import org.elasticsearch.repositories.blobstore.BlobStoreFormat;
import org.elasticsearch.repositories.blobstore.ChecksumBlobStoreFormat;
import org.elasticsearch.repositories.blobstore.LegacyBlobStoreFormat;
import org.elasticsearch.snapshots.InvalidSnapshotNameException;
import org.elasticsearch.snapshots.Snapshot;
import org.elasticsearch.snapshots.SnapshotCreationException;
import org.elasticsearch.snapshots.SnapshotException;
import org.elasticsearch.snapshots.SnapshotMissingException;
import org.elasticsearch.snapshots.SnapshotShardFailure;

public abstract class BlobStoreRepository
extends AbstractLifecycleComponent<Repository>
implements Repository,
BlobStoreIndexShardRepository.RateLimiterListener {
    private BlobContainer snapshotsBlobContainer;
    protected final String repositoryName;
    private static final String LEGACY_SNAPSHOT_PREFIX = "snapshot-";
    private static final String SNAPSHOT_PREFIX = "snap-";
    private static final String SNAPSHOT_SUFFIX = ".dat";
    private static final String COMMON_SNAPSHOT_PREFIX = "snap";
    private static final String SNAPSHOT_CODEC = "snapshot";
    private static final String SNAPSHOTS_FILE = "index";
    private static final String TESTS_FILE = "tests-";
    private static final String METADATA_NAME_FORMAT = "meta-%s.dat";
    private static final String LEGACY_METADATA_NAME_FORMAT = "metadata-%s";
    private static final String METADATA_CODEC = "metadata";
    private static final String INDEX_METADATA_CODEC = "index-metadata";
    private static final String SNAPSHOT_NAME_FORMAT = "snap-%s.dat";
    private static final String LEGACY_SNAPSHOT_NAME_FORMAT = "snapshot-%s";
    private final BlobStoreIndexShardRepository indexShardRepository;
    private final RateLimiter snapshotRateLimiter;
    private final RateLimiter restoreRateLimiter;
    private final CounterMetric snapshotRateLimitingTimeInNanos = new CounterMetric();
    private final CounterMetric restoreRateLimitingTimeInNanos = new CounterMetric();
    private ChecksumBlobStoreFormat<MetaData> globalMetaDataFormat;
    private LegacyBlobStoreFormat<MetaData> globalMetaDataLegacyFormat;
    private ChecksumBlobStoreFormat<IndexMetaData> indexMetaDataFormat;
    private LegacyBlobStoreFormat<IndexMetaData> indexMetaDataLegacyFormat;
    private ChecksumBlobStoreFormat<Snapshot> snapshotFormat;
    private LegacyBlobStoreFormat<Snapshot> snapshotLegacyFormat;
    private final boolean readOnly;

    protected BlobStoreRepository(String repositoryName, RepositorySettings repositorySettings, IndexShardRepository indexShardRepository) {
        super(repositorySettings.globalSettings());
        this.repositoryName = repositoryName;
        this.indexShardRepository = (BlobStoreIndexShardRepository)indexShardRepository;
        this.snapshotRateLimiter = this.getRateLimiter(repositorySettings, "max_snapshot_bytes_per_sec", new ByteSizeValue(40L, ByteSizeUnit.MB));
        this.restoreRateLimiter = this.getRateLimiter(repositorySettings, "max_restore_bytes_per_sec", new ByteSizeValue(40L, ByteSizeUnit.MB));
        this.readOnly = repositorySettings.settings().getAsBoolean("readonly", (Boolean)false);
    }

    @Override
    protected void doStart() {
        this.snapshotsBlobContainer = this.blobStore().blobContainer(this.basePath());
        this.indexShardRepository.initialize(this.blobStore(), this.basePath(), this.chunkSize(), this.snapshotRateLimiter, this.restoreRateLimiter, this, this.isCompress());
        ParseFieldMatcher parseFieldMatcher = new ParseFieldMatcher(this.settings);
        this.globalMetaDataFormat = new ChecksumBlobStoreFormat<MetaData>(METADATA_CODEC, METADATA_NAME_FORMAT, MetaData.PROTO, parseFieldMatcher, this.isCompress());
        this.globalMetaDataLegacyFormat = new LegacyBlobStoreFormat<MetaData>(LEGACY_METADATA_NAME_FORMAT, MetaData.PROTO, parseFieldMatcher);
        this.indexMetaDataFormat = new ChecksumBlobStoreFormat<IndexMetaData>(INDEX_METADATA_CODEC, METADATA_NAME_FORMAT, IndexMetaData.PROTO, parseFieldMatcher, this.isCompress());
        this.indexMetaDataLegacyFormat = new LegacyBlobStoreFormat<IndexMetaData>(LEGACY_SNAPSHOT_NAME_FORMAT, IndexMetaData.PROTO, parseFieldMatcher);
        this.snapshotFormat = new ChecksumBlobStoreFormat<Snapshot>(SNAPSHOT_CODEC, SNAPSHOT_NAME_FORMAT, Snapshot.PROTO, parseFieldMatcher, this.isCompress());
        this.snapshotLegacyFormat = new LegacyBlobStoreFormat<Snapshot>(LEGACY_SNAPSHOT_NAME_FORMAT, Snapshot.PROTO, parseFieldMatcher);
    }

    @Override
    protected void doStop() {
    }

    @Override
    protected void doClose() {
        try {
            this.blobStore().close();
        }
        catch (Throwable t) {
            this.logger.warn("cannot close blob store", t, new Object[0]);
        }
    }

    protected abstract BlobStore blobStore();

    protected abstract BlobPath basePath();

    protected boolean isCompress() {
        return false;
    }

    protected ByteSizeValue chunkSize() {
        return null;
    }

    @Override
    public void initializeSnapshot(SnapshotId snapshotId, List<String> indices, MetaData metaData) {
        if (this.readOnly()) {
            throw new RepositoryException(this.repositoryName, "cannot create snapshot in a readonly repository");
        }
        try {
            if (this.snapshotFormat.exists(this.snapshotsBlobContainer, snapshotId.getSnapshot()) || this.snapshotLegacyFormat.exists(this.snapshotsBlobContainer, snapshotId.getSnapshot())) {
                throw new InvalidSnapshotNameException(snapshotId, "snapshot with such name already exists");
            }
            this.globalMetaDataFormat.write(metaData, this.snapshotsBlobContainer, snapshotId.getSnapshot());
            for (String index : indices) {
                IndexMetaData indexMetaData = metaData.index(index);
                BlobPath indexPath = this.basePath().add("indices").add(index);
                BlobContainer indexMetaDataBlobContainer = this.blobStore().blobContainer(indexPath);
                this.indexMetaDataFormat.write(indexMetaData, indexMetaDataBlobContainer, snapshotId.getSnapshot());
            }
        }
        catch (IOException ex) {
            throw new SnapshotCreationException(snapshotId, (Throwable)ex);
        }
    }

    @Override
    public void deleteSnapshot(SnapshotId snapshotId) {
        if (this.readOnly()) {
            throw new RepositoryException(this.repositoryName, "cannot delete snapshot from a readonly repository");
        }
        List<String> indices = Collections.EMPTY_LIST;
        Snapshot snapshot = null;
        try {
            snapshot = this.readSnapshot(snapshotId);
            indices = snapshot.indices();
        }
        catch (SnapshotMissingException ex) {
            throw ex;
        }
        catch (IllegalStateException | ElasticsearchParseException | SnapshotException ex) {
            this.logger.warn("cannot read snapshot file [{}]", ex, snapshotId);
        }
        MetaData metaData = null;
        try {
            metaData = snapshot != null ? this.readSnapshotMetaData(snapshotId, snapshot.version(), indices, true) : this.readSnapshotMetaData(snapshotId, null, indices, true);
        }
        catch (IOException | SnapshotException ex) {
            this.logger.warn("cannot read metadata for snapshot [{}]", ex, snapshotId);
        }
        try {
            if (snapshot != null) {
                this.snapshotFormat(snapshot.version()).delete(this.snapshotsBlobContainer, snapshotId.getSnapshot());
                this.globalMetaDataFormat(snapshot.version()).delete(this.snapshotsBlobContainer, snapshotId.getSnapshot());
            } else {
                this.snapshotFormat.delete(this.snapshotsBlobContainer, snapshotId.getSnapshot());
                this.snapshotLegacyFormat.delete(this.snapshotsBlobContainer, snapshotId.getSnapshot());
                this.globalMetaDataLegacyFormat.delete(this.snapshotsBlobContainer, snapshotId.getSnapshot());
                this.globalMetaDataFormat.delete(this.snapshotsBlobContainer, snapshotId.getSnapshot());
            }
            List<SnapshotId> snapshotIds = this.snapshots();
            if (snapshotIds.contains(snapshotId)) {
                ArrayList<SnapshotId> builder = new ArrayList<SnapshotId>();
                for (SnapshotId id : snapshotIds) {
                    if (snapshotId.equals(id)) continue;
                    builder.add(id);
                }
                snapshotIds = Collections.unmodifiableList(builder);
            }
            this.writeSnapshotList(snapshotIds);
            for (String index : indices) {
                IndexMetaData indexMetaData;
                BlobPath indexPath = this.basePath().add("indices").add(index);
                BlobContainer indexMetaDataBlobContainer = this.blobStore().blobContainer(indexPath);
                try {
                    this.indexMetaDataFormat(snapshot.version()).delete(indexMetaDataBlobContainer, snapshotId.getSnapshot());
                }
                catch (IOException ex) {
                    this.logger.warn("[{}] failed to delete metadata for index [{}]", ex, snapshotId, index);
                }
                if (metaData == null || (indexMetaData = metaData.index(index)) == null) continue;
                for (int i = 0; i < indexMetaData.getNumberOfShards(); ++i) {
                    ShardId shardId = new ShardId(index, i);
                    try {
                        this.indexShardRepository.delete(snapshotId, snapshot.version(), shardId);
                        continue;
                    }
                    catch (SnapshotException ex) {
                        this.logger.warn("[{}] failed to delete shard data for shard [{}]", ex, snapshotId, shardId);
                    }
                }
            }
        }
        catch (IOException ex) {
            throw new RepositoryException(this.repositoryName, "failed to update snapshot in repository", ex);
        }
    }

    @Override
    public Snapshot finalizeSnapshot(SnapshotId snapshotId, List<String> indices, long startTime, String failure, int totalShards, List<SnapshotShardFailure> shardFailures) {
        try {
            Snapshot blobStoreSnapshot = new Snapshot(snapshotId.getSnapshot(), indices, startTime, failure, System.currentTimeMillis(), totalShards, shardFailures);
            this.snapshotFormat.write(blobStoreSnapshot, this.snapshotsBlobContainer, snapshotId.getSnapshot());
            List<SnapshotId> snapshotIds = this.snapshots();
            if (!snapshotIds.contains(snapshotId)) {
                snapshotIds = new ArrayList<SnapshotId>(snapshotIds);
                snapshotIds.add(snapshotId);
                snapshotIds = Collections.unmodifiableList(snapshotIds);
            }
            this.writeSnapshotList(snapshotIds);
            return blobStoreSnapshot;
        }
        catch (IOException ex) {
            throw new RepositoryException(this.repositoryName, "failed to update snapshot in repository", ex);
        }
    }

    @Override
    public List<SnapshotId> snapshots() {
        try {
            Map<String, BlobMetaData> blobs;
            ArrayList<SnapshotId> snapshots = new ArrayList<SnapshotId>();
            try {
                blobs = this.snapshotsBlobContainer.listBlobsByPrefix(COMMON_SNAPSHOT_PREFIX);
            }
            catch (UnsupportedOperationException ex) {
                return this.readSnapshotList();
            }
            int prefixLength = SNAPSHOT_PREFIX.length();
            int suffixLength = SNAPSHOT_SUFFIX.length();
            int legacyPrefixLength = LEGACY_SNAPSHOT_PREFIX.length();
            for (BlobMetaData md : blobs.values()) {
                String name;
                String blobName = md.name();
                if (blobName.startsWith(SNAPSHOT_PREFIX) && blobName.length() > legacyPrefixLength) {
                    name = blobName.substring(prefixLength, blobName.length() - suffixLength);
                } else {
                    if (!blobName.startsWith(LEGACY_SNAPSHOT_PREFIX) || blobName.length() <= suffixLength + prefixLength) continue;
                    name = blobName.substring(legacyPrefixLength);
                }
                snapshots.add(new SnapshotId(this.repositoryName, name));
            }
            return Collections.unmodifiableList(snapshots);
        }
        catch (IOException ex) {
            throw new RepositoryException(this.repositoryName, "failed to list snapshots in repository", ex);
        }
    }

    @Override
    public MetaData readSnapshotMetaData(SnapshotId snapshotId, Snapshot snapshot, List<String> indices) throws IOException {
        return this.readSnapshotMetaData(snapshotId, snapshot.version(), indices, false);
    }

    @Override
    public Snapshot readSnapshot(SnapshotId snapshotId) {
        try {
            return (Snapshot)this.snapshotFormat.read(this.snapshotsBlobContainer, snapshotId.getSnapshot());
        }
        catch (FileNotFoundException | NoSuchFileException ex) {
            try {
                return (Snapshot)this.snapshotLegacyFormat.read(this.snapshotsBlobContainer, snapshotId.getSnapshot());
            }
            catch (FileNotFoundException | NoSuchFileException ex1) {
                throw new SnapshotMissingException(snapshotId, (Throwable)ex);
            }
            catch (IOException | NotXContentException ex1) {
                throw new SnapshotException(snapshotId, "failed to get snapshots", ex1);
            }
        }
        catch (IOException | NotXContentException ex) {
            throw new SnapshotException(snapshotId, "failed to get snapshots", ex);
        }
    }

    private MetaData readSnapshotMetaData(SnapshotId snapshotId, Version snapshotVersion, List<String> indices, boolean ignoreIndexErrors) throws IOException {
        MetaData metaData;
        if (snapshotVersion == null) {
            assert (ignoreIndexErrors);
            if (this.globalMetaDataFormat.exists(this.snapshotsBlobContainer, snapshotId.getSnapshot())) {
                snapshotVersion = Version.CURRENT;
            } else if (this.globalMetaDataLegacyFormat.exists(this.snapshotsBlobContainer, snapshotId.getSnapshot())) {
                snapshotVersion = Version.V_1_0_0;
            } else {
                throw new SnapshotMissingException(snapshotId);
            }
        }
        try {
            metaData = this.globalMetaDataFormat(snapshotVersion).read(this.snapshotsBlobContainer, snapshotId.getSnapshot());
        }
        catch (FileNotFoundException | NoSuchFileException ex) {
            throw new SnapshotMissingException(snapshotId, (Throwable)ex);
        }
        catch (IOException ex) {
            throw new SnapshotException(snapshotId, "failed to get snapshots", ex);
        }
        MetaData.Builder metaDataBuilder = MetaData.builder(metaData);
        for (String index : indices) {
            BlobPath indexPath = this.basePath().add("indices").add(index);
            BlobContainer indexMetaDataBlobContainer = this.blobStore().blobContainer(indexPath);
            try {
                metaDataBuilder.put(this.indexMetaDataFormat(snapshotVersion).read(indexMetaDataBlobContainer, snapshotId.getSnapshot()), false);
            }
            catch (IOException | ElasticsearchParseException ex) {
                if (ignoreIndexErrors) {
                    this.logger.warn("[{}] [{}] failed to read metadata for index", snapshotId, index, ex);
                    continue;
                }
                throw ex;
            }
        }
        return metaDataBuilder.build();
    }

    private RateLimiter getRateLimiter(RepositorySettings repositorySettings, String setting, ByteSizeValue defaultRate) {
        ByteSizeValue maxSnapshotBytesPerSec = repositorySettings.settings().getAsBytesSize(setting, this.settings.getAsBytesSize(setting, defaultRate));
        if (maxSnapshotBytesPerSec.bytes() <= 0L) {
            return null;
        }
        return new RateLimiter.SimpleRateLimiter(maxSnapshotBytesPerSec.mbFrac());
    }

    private BlobStoreFormat<MetaData> globalMetaDataFormat(Version version) {
        if (BlobStoreRepository.legacyMetaData(version)) {
            return this.globalMetaDataLegacyFormat;
        }
        return this.globalMetaDataFormat;
    }

    private BlobStoreFormat<Snapshot> snapshotFormat(Version version) {
        if (BlobStoreRepository.legacyMetaData(version)) {
            return this.snapshotLegacyFormat;
        }
        return this.snapshotFormat;
    }

    public static boolean legacyMetaData(Version version) {
        return version.before(Version.V_2_0_0_beta1);
    }

    private BlobStoreFormat<IndexMetaData> indexMetaDataFormat(Version version) {
        if (BlobStoreRepository.legacyMetaData(version)) {
            return this.indexMetaDataLegacyFormat;
        }
        return this.indexMetaDataFormat;
    }

    protected void writeSnapshotList(List<SnapshotId> snapshots) throws IOException {
        BytesReference bRef;
        try (BytesStreamOutput bStream = new BytesStreamOutput();){
            try (OutputStreamStreamOutput stream = new OutputStreamStreamOutput(bStream);){
                XContentBuilder builder = XContentFactory.contentBuilder(XContentType.JSON, stream);
                builder.startObject();
                builder.startArray("snapshots");
                for (SnapshotId snapshot : snapshots) {
                    builder.value(snapshot.getSnapshot());
                }
                builder.endArray();
                builder.endObject();
                builder.close();
            }
            bRef = bStream.bytes();
        }
        if (this.snapshotsBlobContainer.blobExists(SNAPSHOTS_FILE)) {
            this.snapshotsBlobContainer.deleteBlob(SNAPSHOTS_FILE);
        }
        this.snapshotsBlobContainer.writeBlob(SNAPSHOTS_FILE, bRef);
    }

    protected List<SnapshotId> readSnapshotList() throws IOException {
        try (InputStream blob = this.snapshotsBlobContainer.readBlob(SNAPSHOTS_FILE);){
            byte[] data = ByteStreams.toByteArray(blob);
            ArrayList<SnapshotId> snapshots = new ArrayList<SnapshotId>();
            try (XContentParser parser = XContentHelper.createParser(new BytesArray(data));){
                String currentFieldName;
                if (parser.nextToken() == XContentParser.Token.START_OBJECT && parser.nextToken() == XContentParser.Token.FIELD_NAME && "snapshots".equals(currentFieldName = parser.currentName()) && parser.nextToken() == XContentParser.Token.START_ARRAY) {
                    while (parser.nextToken() != XContentParser.Token.END_ARRAY) {
                        snapshots.add(new SnapshotId(this.repositoryName, parser.text()));
                    }
                }
            }
            List<SnapshotId> list = Collections.unmodifiableList(snapshots);
            return list;
        }
    }

    @Override
    public void onRestorePause(long nanos) {
        this.restoreRateLimitingTimeInNanos.inc(nanos);
    }

    @Override
    public void onSnapshotPause(long nanos) {
        this.snapshotRateLimitingTimeInNanos.inc(nanos);
    }

    @Override
    public long snapshotThrottleTimeInNanos() {
        return this.snapshotRateLimitingTimeInNanos.count();
    }

    @Override
    public long restoreThrottleTimeInNanos() {
        return this.restoreRateLimitingTimeInNanos.count();
    }

    @Override
    public String startVerification() {
        try {
            if (this.readOnly()) {
                return null;
            }
            String seed = Strings.randomBase64UUID();
            byte[] testBytes = Strings.toUTF8Bytes(seed);
            BlobContainer testContainer = this.blobStore().blobContainer(this.basePath().add(BlobStoreRepository.testBlobPrefix(seed)));
            String blobName = "master.dat";
            testContainer.writeBlob(blobName + "-temp", new BytesArray(testBytes));
            testContainer.move(blobName + "-temp", blobName);
            return seed;
        }
        catch (IOException exp) {
            throw new RepositoryVerificationException(this.repositoryName, "path " + this.basePath() + " is not accessible on master node", exp);
        }
    }

    @Override
    public void endVerification(String seed) {
        if (this.readOnly()) {
            throw new UnsupportedOperationException("shouldn't be called");
        }
        try {
            this.blobStore().delete(this.basePath().add(BlobStoreRepository.testBlobPrefix(seed)));
        }
        catch (IOException exp) {
            throw new RepositoryVerificationException(this.repositoryName, "cannot delete test data at " + this.basePath(), exp);
        }
    }

    public static String testBlobPrefix(String seed) {
        return TESTS_FILE + seed;
    }

    @Override
    public boolean readOnly() {
        return this.readOnly;
    }
}

