/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.threadpool;

import com.google.common.collect.ImmutableMap;
import com.google.common.util.concurrent.MoreExecutors;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.lucene.util.Counter;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.settings.Validator;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.collect.MapBuilder;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.io.stream.Streamable;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.settings.SettingsException;
import org.elasticsearch.common.unit.SizeValue;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.EsAbortPolicy;
import org.elasticsearch.common.util.concurrent.EsExecutors;
import org.elasticsearch.common.util.concurrent.EsThreadPoolExecutor;
import org.elasticsearch.common.util.concurrent.XRejectedExecutionHandler;
import org.elasticsearch.common.xcontent.ToXContent;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.common.xcontent.XContentBuilderString;
import org.elasticsearch.node.settings.NodeSettingsService;
import org.elasticsearch.threadpool.ThreadPoolInfo;
import org.elasticsearch.threadpool.ThreadPoolStats;

public class ThreadPool
extends AbstractComponent {
    public static Map<String, ThreadPoolType> THREAD_POOL_TYPES;
    public static final String THREADPOOL_GROUP = "threadpool.";
    private volatile ImmutableMap<String, ExecutorHolder> executors;
    private final ImmutableMap<String, Settings> defaultExecutorTypeSettings;
    private final Queue<ExecutorHolder> retiredExecutors = new ConcurrentLinkedQueue<ExecutorHolder>();
    private final ScheduledThreadPoolExecutor scheduler;
    private final EstimatedTimeThread estimatedTimeThread;
    private boolean settingsListenerIsSet = false;
    static final Executor DIRECT_EXECUTOR;
    public static ThreadPoolTypeSettingsValidator THREAD_POOL_TYPE_SETTINGS_VALIDATOR;

    private static void add(Map<String, Settings> executorSettings, ExecutorSettingsBuilder builder) {
        Settings settings = builder.build();
        String name = settings.get("name");
        executorSettings.put(name, settings);
    }

    public ThreadPool(String name) {
        this(Settings.builder().put("name", name).build());
    }

    public ThreadPool(Settings settings) {
        super(settings);
        assert (settings.get("name") != null) : "ThreadPool's settings should contain a name";
        Map<String, Settings> groupSettings = this.getThreadPoolSettingsGroup(settings);
        int availableProcessors = EsExecutors.boundedNumberOfProcessors(settings);
        int halfProcMaxAt5 = Math.min((availableProcessors + 1) / 2, 5);
        int halfProcMaxAt10 = Math.min((availableProcessors + 1) / 2, 10);
        HashMap<String, Settings> defaultExecutorTypeSettings = new HashMap<String, Settings>();
        ThreadPool.add(defaultExecutorTypeSettings, new ExecutorSettingsBuilder("generic").keepAlive("30s"));
        ThreadPool.add(defaultExecutorTypeSettings, new ExecutorSettingsBuilder("index").size(availableProcessors).queueSize(200));
        ThreadPool.add(defaultExecutorTypeSettings, new ExecutorSettingsBuilder("bulk").size(availableProcessors).queueSize(50));
        ThreadPool.add(defaultExecutorTypeSettings, new ExecutorSettingsBuilder("get").size(availableProcessors).queueSize(1000));
        ThreadPool.add(defaultExecutorTypeSettings, new ExecutorSettingsBuilder("search").size(availableProcessors * 3 / 2 + 1).queueSize(1000));
        ThreadPool.add(defaultExecutorTypeSettings, new ExecutorSettingsBuilder("suggest").size(availableProcessors).queueSize(1000));
        ThreadPool.add(defaultExecutorTypeSettings, new ExecutorSettingsBuilder("percolate").size(availableProcessors).queueSize(1000));
        ThreadPool.add(defaultExecutorTypeSettings, new ExecutorSettingsBuilder("management").size(5).keepAlive("5m"));
        ThreadPool.add(defaultExecutorTypeSettings, new ExecutorSettingsBuilder("listener").size(halfProcMaxAt10));
        ThreadPool.add(defaultExecutorTypeSettings, new ExecutorSettingsBuilder("flush").size(halfProcMaxAt5).keepAlive("5m"));
        ThreadPool.add(defaultExecutorTypeSettings, new ExecutorSettingsBuilder("refresh").size(halfProcMaxAt10).keepAlive("5m"));
        ThreadPool.add(defaultExecutorTypeSettings, new ExecutorSettingsBuilder("warmer").size(halfProcMaxAt5).keepAlive("5m"));
        ThreadPool.add(defaultExecutorTypeSettings, new ExecutorSettingsBuilder("snapshot").size(halfProcMaxAt5).keepAlive("5m"));
        ThreadPool.add(defaultExecutorTypeSettings, new ExecutorSettingsBuilder("force_merge").size(1));
        ThreadPool.add(defaultExecutorTypeSettings, new ExecutorSettingsBuilder("fetch_shard_started").size(availableProcessors * 2).keepAlive("5m"));
        ThreadPool.add(defaultExecutorTypeSettings, new ExecutorSettingsBuilder("fetch_shard_store").size(availableProcessors * 2).keepAlive("5m"));
        this.defaultExecutorTypeSettings = ImmutableMap.copyOf(defaultExecutorTypeSettings);
        HashMap<Object, ExecutorHolder> executors = new HashMap<Object, ExecutorHolder>();
        for (Map.Entry entry : defaultExecutorTypeSettings.entrySet()) {
            executors.put(entry.getKey(), this.build((String)entry.getKey(), groupSettings.get(entry.getKey()), (Settings)entry.getValue()));
        }
        for (Map.Entry<Object, Object> entry : groupSettings.entrySet()) {
            if (executors.containsKey(entry.getKey())) continue;
            executors.put(entry.getKey(), this.build((String)entry.getKey(), (Settings)entry.getValue(), Settings.EMPTY));
        }
        executors.put("same", new ExecutorHolder(DIRECT_EXECUTOR, new Info("same", ThreadPoolType.DIRECT)));
        if (!((ExecutorHolder)executors.get((Object)"generic")).info.getThreadPoolType().equals((Object)ThreadPoolType.CACHED)) {
            throw new IllegalArgumentException("generic thread pool must be of type cached");
        }
        this.executors = ImmutableMap.copyOf(executors);
        this.scheduler = new ScheduledThreadPoolExecutor(1, EsExecutors.daemonThreadFactory(settings, "scheduler"), new EsAbortPolicy());
        this.scheduler.setExecuteExistingDelayedTasksAfterShutdownPolicy(false);
        this.scheduler.setContinueExistingPeriodicTasksAfterShutdownPolicy(false);
        this.scheduler.setRemoveOnCancelPolicy(true);
        TimeValue estimatedTimeInterval = settings.getAsTime("threadpool.estimated_time_interval", TimeValue.timeValueMillis(200L));
        this.estimatedTimeThread = new EstimatedTimeThread(EsExecutors.threadName(settings, "[timer]"), estimatedTimeInterval.millis());
        this.estimatedTimeThread.start();
    }

    private Map<String, Settings> getThreadPoolSettingsGroup(Settings settings) {
        Map<String, Settings> groupSettings = settings.getGroups(THREADPOOL_GROUP);
        this.validate(groupSettings);
        return groupSettings;
    }

    public void setNodeSettingsService(NodeSettingsService nodeSettingsService) {
        if (this.settingsListenerIsSet) {
            throw new IllegalStateException("the node settings listener was set more then once");
        }
        nodeSettingsService.addListener(new ApplySettings());
        this.settingsListenerIsSet = true;
    }

    public long estimatedTimeInMillis() {
        return this.estimatedTimeThread.estimatedTimeInMillis();
    }

    public Counter estimatedTimeInMillisCounter() {
        return this.estimatedTimeThread.counter;
    }

    public ThreadPoolInfo info() {
        ArrayList<Info> infos = new ArrayList<Info>();
        for (ExecutorHolder holder : this.executors.values()) {
            String name = holder.info.getName();
            if ("same".equals(name)) continue;
            infos.add(holder.info);
        }
        return new ThreadPoolInfo(infos);
    }

    public Info info(String name) {
        ExecutorHolder holder = this.executors.get(name);
        if (holder == null) {
            return null;
        }
        return holder.info;
    }

    public ThreadPoolStats stats() {
        ArrayList<ThreadPoolStats.Stats> stats = new ArrayList<ThreadPoolStats.Stats>();
        for (ExecutorHolder holder : this.executors.values()) {
            String name = holder.info.getName();
            if ("same".equals(name)) continue;
            int threads = -1;
            int queue = -1;
            int active = -1;
            long rejected = -1L;
            int largest = -1;
            long completed = -1L;
            if (holder.executor() instanceof ThreadPoolExecutor) {
                ThreadPoolExecutor threadPoolExecutor = (ThreadPoolExecutor)holder.executor();
                threads = threadPoolExecutor.getPoolSize();
                queue = threadPoolExecutor.getQueue().size();
                active = threadPoolExecutor.getActiveCount();
                largest = threadPoolExecutor.getLargestPoolSize();
                completed = threadPoolExecutor.getCompletedTaskCount();
                RejectedExecutionHandler rejectedExecutionHandler = threadPoolExecutor.getRejectedExecutionHandler();
                if (rejectedExecutionHandler instanceof XRejectedExecutionHandler) {
                    rejected = ((XRejectedExecutionHandler)rejectedExecutionHandler).rejected();
                }
            }
            stats.add(new ThreadPoolStats.Stats(name, threads, queue, active, rejected, largest, completed));
        }
        return new ThreadPoolStats(stats);
    }

    public Executor generic() {
        return this.executor("generic");
    }

    public Executor executor(String name) {
        Executor executor = this.executors.get(name).executor();
        if (executor == null) {
            throw new IllegalArgumentException("No executor found for [" + name + "]");
        }
        return executor;
    }

    public ScheduledExecutorService scheduler() {
        return this.scheduler;
    }

    public ScheduledFuture<?> scheduleWithFixedDelay(Runnable command, TimeValue interval) {
        return this.scheduler.scheduleWithFixedDelay(new LoggingRunnable(command), interval.millis(), interval.millis(), TimeUnit.MILLISECONDS);
    }

    public ScheduledFuture<?> schedule(TimeValue delay, String name, Runnable command) {
        if (!"same".equals(name)) {
            command = new ThreadedRunnable(command, this.executor(name));
        }
        return this.scheduler.schedule(new LoggingRunnable(command), delay.millis(), TimeUnit.MILLISECONDS);
    }

    public void shutdown() {
        this.estimatedTimeThread.running = false;
        this.estimatedTimeThread.interrupt();
        this.scheduler.shutdown();
        for (ExecutorHolder executor : this.executors.values()) {
            if (!(executor.executor() instanceof ThreadPoolExecutor)) continue;
            ((ThreadPoolExecutor)executor.executor()).shutdown();
        }
    }

    public void shutdownNow() {
        this.estimatedTimeThread.running = false;
        this.estimatedTimeThread.interrupt();
        this.scheduler.shutdownNow();
        for (ExecutorHolder executor : this.executors.values()) {
            if (!(executor.executor() instanceof ThreadPoolExecutor)) continue;
            ((ThreadPoolExecutor)executor.executor()).shutdownNow();
        }
        while (!this.retiredExecutors.isEmpty()) {
            ((ThreadPoolExecutor)this.retiredExecutors.remove().executor()).shutdownNow();
        }
    }

    public boolean awaitTermination(long timeout, TimeUnit unit) throws InterruptedException {
        boolean result = this.scheduler.awaitTermination(timeout, unit);
        for (ExecutorHolder executor : this.executors.values()) {
            if (!(executor.executor() instanceof ThreadPoolExecutor)) continue;
            result &= ((ThreadPoolExecutor)executor.executor()).awaitTermination(timeout, unit);
        }
        while (!this.retiredExecutors.isEmpty()) {
            ThreadPoolExecutor executor = (ThreadPoolExecutor)this.retiredExecutors.remove().executor();
            result &= executor.awaitTermination(timeout, unit);
        }
        this.estimatedTimeThread.join(unit.toMillis(timeout));
        return result;
    }

    private ExecutorHolder build(String name, @Nullable Settings settings, Settings defaultSettings) {
        return this.rebuild(name, null, settings, defaultSettings);
    }

    private ExecutorHolder rebuild(String name, ExecutorHolder previousExecutorHolder, @Nullable Settings settings, Settings defaultSettings) {
        if ("same".equals(name)) {
            return previousExecutorHolder;
        }
        if (settings == null) {
            settings = Settings.Builder.EMPTY_SETTINGS;
        }
        Info previousInfo = previousExecutorHolder != null ? previousExecutorHolder.info : null;
        String type = settings.get("type", previousInfo != null ? previousInfo.getThreadPoolType().getType() : defaultSettings.get("type"));
        ThreadPoolType threadPoolType = ThreadPoolType.fromType(type);
        ThreadFactory threadFactory = EsExecutors.daemonThreadFactory(this.settings, name);
        if (ThreadPoolType.DIRECT == threadPoolType) {
            if (previousExecutorHolder != null) {
                this.logger.debug("updating thread_pool [{}], type [{}]", name, type);
            } else {
                this.logger.debug("creating thread_pool [{}], type [{}]", name, type);
            }
            return new ExecutorHolder(DIRECT_EXECUTOR, new Info(name, threadPoolType));
        }
        if (ThreadPoolType.CACHED == threadPoolType) {
            if (!"generic".equals(name)) {
                throw new IllegalArgumentException("thread pool type cached is reserved only for the generic thread pool and can not be applied to [" + name + "]");
            }
            TimeValue defaultKeepAlive = defaultSettings.getAsTime("keep_alive", TimeValue.timeValueMinutes(5L));
            if (previousExecutorHolder != null) {
                if (ThreadPoolType.CACHED == previousInfo.getThreadPoolType()) {
                    TimeValue updatedKeepAlive = settings.getAsTime("keep_alive", previousInfo.getKeepAlive());
                    if (!previousInfo.getKeepAlive().equals(updatedKeepAlive)) {
                        this.logger.debug("updating thread_pool [{}], type [{}], keep_alive [{}]", name, type, updatedKeepAlive);
                        ((EsThreadPoolExecutor)previousExecutorHolder.executor()).setKeepAliveTime(updatedKeepAlive.millis(), TimeUnit.MILLISECONDS);
                        return new ExecutorHolder(previousExecutorHolder.executor(), new Info(name, threadPoolType, -1, -1, updatedKeepAlive, null));
                    }
                    return previousExecutorHolder;
                }
                if (previousInfo.getKeepAlive() != null) {
                    defaultKeepAlive = previousInfo.getKeepAlive();
                }
            }
            TimeValue keepAlive = settings.getAsTime("keep_alive", defaultKeepAlive);
            if (previousExecutorHolder != null) {
                this.logger.debug("updating thread_pool [{}], type [{}], keep_alive [{}]", name, type, keepAlive);
            } else {
                this.logger.debug("creating thread_pool [{}], type [{}], keep_alive [{}]", name, type, keepAlive);
            }
            EsThreadPoolExecutor executor = EsExecutors.newCached(name, keepAlive.millis(), TimeUnit.MILLISECONDS, threadFactory);
            return new ExecutorHolder(executor, new Info(name, threadPoolType, -1, -1, keepAlive, null));
        }
        if (ThreadPoolType.FIXED == threadPoolType) {
            int defaultSize = defaultSettings.getAsInt("size", (Integer)EsExecutors.boundedNumberOfProcessors(settings));
            SizeValue defaultQueueSize = this.getAsSizeOrUnbounded(defaultSettings, "queue", this.getAsSizeOrUnbounded(defaultSettings, "queue_size", null));
            if (previousExecutorHolder != null) {
                if (ThreadPoolType.FIXED == previousInfo.getThreadPoolType()) {
                    SizeValue updatedQueueSize = this.getAsSizeOrUnbounded(settings, "capacity", this.getAsSizeOrUnbounded(settings, "queue", this.getAsSizeOrUnbounded(settings, "queue_size", previousInfo.getQueueSize())));
                    if (Objects.equals(previousInfo.getQueueSize(), updatedQueueSize)) {
                        int updatedSize = this.applyHardSizeLimit(name, settings.getAsInt("size", (Integer)previousInfo.getMax()));
                        if (previousInfo.getMax() != updatedSize) {
                            this.logger.debug("updating thread_pool [{}], type [{}], size [{}], queue_size [{}]", name, type, updatedSize, updatedQueueSize);
                            if (updatedSize > previousInfo.getMax()) {
                                ((EsThreadPoolExecutor)previousExecutorHolder.executor()).setMaximumPoolSize(updatedSize);
                                ((EsThreadPoolExecutor)previousExecutorHolder.executor()).setCorePoolSize(updatedSize);
                            } else {
                                ((EsThreadPoolExecutor)previousExecutorHolder.executor()).setCorePoolSize(updatedSize);
                                ((EsThreadPoolExecutor)previousExecutorHolder.executor()).setMaximumPoolSize(updatedSize);
                            }
                            return new ExecutorHolder(previousExecutorHolder.executor(), new Info(name, threadPoolType, updatedSize, updatedSize, null, updatedQueueSize));
                        }
                        return previousExecutorHolder;
                    }
                }
                if (previousInfo.getMax() >= 0) {
                    defaultSize = previousInfo.getMax();
                }
                defaultQueueSize = previousInfo.getQueueSize();
            }
            int size = this.applyHardSizeLimit(name, settings.getAsInt("size", (Integer)defaultSize));
            SizeValue queueSize = this.getAsSizeOrUnbounded(settings, "capacity", this.getAsSizeOrUnbounded(settings, "queue", this.getAsSizeOrUnbounded(settings, "queue_size", defaultQueueSize)));
            this.logger.debug("creating thread_pool [{}], type [{}], size [{}], queue_size [{}]", name, type, size, queueSize);
            EsThreadPoolExecutor executor = EsExecutors.newFixed(name, size, queueSize == null ? -1 : (int)queueSize.singles(), threadFactory);
            return new ExecutorHolder(executor, new Info(name, threadPoolType, size, size, null, queueSize));
        }
        if (ThreadPoolType.SCALING == threadPoolType) {
            TimeValue defaultKeepAlive = defaultSettings.getAsTime("keep_alive", TimeValue.timeValueMinutes(5L));
            int defaultMin = defaultSettings.getAsInt("min", (Integer)1);
            int defaultSize = defaultSettings.getAsInt("size", (Integer)EsExecutors.boundedNumberOfProcessors(settings));
            if (previousExecutorHolder != null) {
                if (ThreadPoolType.SCALING == previousInfo.getThreadPoolType()) {
                    TimeValue updatedKeepAlive = settings.getAsTime("keep_alive", previousInfo.getKeepAlive());
                    int updatedMin = settings.getAsInt("min", (Integer)previousInfo.getMin());
                    int updatedSize = settings.getAsInt("max", settings.getAsInt("size", (Integer)previousInfo.getMax()));
                    if (!previousInfo.getKeepAlive().equals(updatedKeepAlive) || previousInfo.getMin() != updatedMin || previousInfo.getMax() != updatedSize) {
                        this.logger.debug("updating thread_pool [{}], type [{}], keep_alive [{}]", name, type, updatedKeepAlive);
                        if (!previousInfo.getKeepAlive().equals(updatedKeepAlive)) {
                            ((EsThreadPoolExecutor)previousExecutorHolder.executor()).setKeepAliveTime(updatedKeepAlive.millis(), TimeUnit.MILLISECONDS);
                        }
                        if (previousInfo.getMin() != updatedMin) {
                            ((EsThreadPoolExecutor)previousExecutorHolder.executor()).setCorePoolSize(updatedMin);
                        }
                        if (previousInfo.getMax() != updatedSize) {
                            ((EsThreadPoolExecutor)previousExecutorHolder.executor()).setMaximumPoolSize(updatedSize);
                        }
                        return new ExecutorHolder(previousExecutorHolder.executor(), new Info(name, threadPoolType, updatedMin, updatedSize, updatedKeepAlive, null));
                    }
                    return previousExecutorHolder;
                }
                if (previousInfo.getKeepAlive() != null) {
                    defaultKeepAlive = previousInfo.getKeepAlive();
                }
                if (previousInfo.getMin() >= 0) {
                    defaultMin = previousInfo.getMin();
                }
                if (previousInfo.getMax() >= 0) {
                    defaultSize = previousInfo.getMax();
                }
            }
            TimeValue keepAlive = settings.getAsTime("keep_alive", defaultKeepAlive);
            int min = settings.getAsInt("min", (Integer)defaultMin);
            int size = settings.getAsInt("max", settings.getAsInt("size", (Integer)defaultSize));
            if (previousExecutorHolder != null) {
                this.logger.debug("updating thread_pool [{}], type [{}], min [{}], size [{}], keep_alive [{}]", name, type, min, size, keepAlive);
            } else {
                this.logger.debug("creating thread_pool [{}], type [{}], min [{}], size [{}], keep_alive [{}]", name, type, min, size, keepAlive);
            }
            EsThreadPoolExecutor executor = EsExecutors.newScaling(name, min, size, keepAlive.millis(), TimeUnit.MILLISECONDS, threadFactory);
            return new ExecutorHolder(executor, new Info(name, threadPoolType, min, size, keepAlive, null));
        }
        throw new IllegalArgumentException("No type found [" + type + "], for [" + name + "]");
    }

    private int applyHardSizeLimit(String name, int size) {
        int availableProcessors = EsExecutors.boundedNumberOfProcessors(this.settings);
        if ((name.equals("bulk") || name.equals("index")) && size > availableProcessors) {
            this.logger.warn("requested thread pool size [{}] for [{}] is too large; setting to maximum [{}] instead", size, name, availableProcessors);
            size = availableProcessors;
        }
        return size;
    }

    public void updateSettings(Settings settings) {
        Map<String, Settings> groupSettings = this.getThreadPoolSettingsGroup(settings);
        if (groupSettings.isEmpty()) {
            return;
        }
        for (Map.Entry executor : this.defaultExecutorTypeSettings.entrySet()) {
            ExecutorHolder newExecutorHolder;
            ExecutorHolder oldExecutorHolder;
            Settings updatedSettings = groupSettings.get(executor.getKey());
            if (updatedSettings == null || (oldExecutorHolder = this.executors.get(executor.getKey())).equals(newExecutorHolder = this.rebuild((String)executor.getKey(), oldExecutorHolder, updatedSettings, (Settings)executor.getValue()))) continue;
            this.executors = MapBuilder.newMapBuilder(this.executors).put((String)executor.getKey(), newExecutorHolder).immutableMap();
            if (oldExecutorHolder.executor().equals(newExecutorHolder.executor()) || !(oldExecutorHolder.executor() instanceof EsThreadPoolExecutor)) continue;
            this.retiredExecutors.add(oldExecutorHolder);
            ((EsThreadPoolExecutor)oldExecutorHolder.executor()).shutdown(new ExecutorShutdownListener(oldExecutorHolder));
        }
        for (Map.Entry<String, Settings> entry : groupSettings.entrySet()) {
            if (this.defaultExecutorTypeSettings.containsKey(entry.getKey())) continue;
            ExecutorHolder oldExecutorHolder = this.executors.get(entry.getKey());
            ExecutorHolder newExecutorHolder = this.rebuild(entry.getKey(), oldExecutorHolder, entry.getValue(), Settings.EMPTY);
            if (newExecutorHolder.equals(oldExecutorHolder)) continue;
            this.executors = MapBuilder.newMapBuilder(this.executors).put(entry.getKey(), newExecutorHolder).immutableMap();
            if (oldExecutorHolder.executor().equals(newExecutorHolder.executor()) || !(oldExecutorHolder.executor() instanceof EsThreadPoolExecutor)) continue;
            this.retiredExecutors.add(oldExecutorHolder);
            ((EsThreadPoolExecutor)oldExecutorHolder.executor()).shutdown(new ExecutorShutdownListener(oldExecutorHolder));
        }
    }

    private void validate(Map<String, Settings> groupSettings) {
        for (String key : groupSettings.keySet()) {
            if (!THREAD_POOL_TYPES.containsKey(key)) continue;
            String type = groupSettings.get(key).get("type");
            ThreadPoolType correctThreadPoolType = THREAD_POOL_TYPES.get(key);
            if (type == null || correctThreadPoolType.getType().equals(type)) continue;
            throw new IllegalArgumentException("setting threadpool." + key + ".type to " + type + " is not permitted; must be " + correctThreadPoolType.getType());
        }
    }

    private SizeValue getAsSizeOrUnbounded(Settings settings, String setting, SizeValue defaultValue) throws SettingsException {
        if ("-1".equals(settings.get(setting))) {
            return null;
        }
        return SizeValue.parseSizeValue(settings.get(setting), defaultValue);
    }

    public static boolean terminate(ExecutorService service, long timeout, TimeUnit timeUnit) {
        if (service != null) {
            service.shutdown();
            try {
                if (service.awaitTermination(timeout, timeUnit)) {
                    return true;
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            service.shutdownNow();
        }
        return false;
    }

    public static boolean terminate(ThreadPool pool, long timeout, TimeUnit timeUnit) {
        if (pool != null) {
            pool.shutdown();
            try {
                if (pool.awaitTermination(timeout, timeUnit)) {
                    return true;
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            pool.shutdownNow();
        }
        return false;
    }

    static {
        HashMap<String, ThreadPoolType> map = new HashMap<String, ThreadPoolType>();
        map.put("same", ThreadPoolType.DIRECT);
        map.put("generic", ThreadPoolType.CACHED);
        map.put("listener", ThreadPoolType.FIXED);
        map.put("get", ThreadPoolType.FIXED);
        map.put("index", ThreadPoolType.FIXED);
        map.put("bulk", ThreadPoolType.FIXED);
        map.put("search", ThreadPoolType.FIXED);
        map.put("suggest", ThreadPoolType.FIXED);
        map.put("percolate", ThreadPoolType.FIXED);
        map.put("management", ThreadPoolType.SCALING);
        map.put("flush", ThreadPoolType.SCALING);
        map.put("refresh", ThreadPoolType.SCALING);
        map.put("warmer", ThreadPoolType.SCALING);
        map.put("snapshot", ThreadPoolType.SCALING);
        map.put("force_merge", ThreadPoolType.FIXED);
        map.put("fetch_shard_started", ThreadPoolType.SCALING);
        map.put("fetch_shard_store", ThreadPoolType.SCALING);
        THREAD_POOL_TYPES = Collections.unmodifiableMap(map);
        DIRECT_EXECUTOR = MoreExecutors.directExecutor();
        THREAD_POOL_TYPE_SETTINGS_VALIDATOR = new ThreadPoolTypeSettingsValidator();
    }

    private static class ThreadPoolTypeSettingsValidator
    implements Validator {
        private ThreadPoolTypeSettingsValidator() {
        }

        @Override
        public String validate(String setting, String value, ClusterState clusterState) {
            ThreadPoolType threadPoolType;
            Matcher matcher = Pattern.compile("threadpool\\.(.*)\\.type").matcher(setting);
            if (!matcher.matches()) {
                return null;
            }
            String threadPool = matcher.group(1);
            ThreadPoolType defaultThreadPoolType = THREAD_POOL_TYPES.get(threadPool);
            try {
                threadPoolType = ThreadPoolType.fromType(value);
            }
            catch (IllegalArgumentException e) {
                return e.getMessage();
            }
            if (defaultThreadPoolType.equals((Object)threadPoolType)) {
                return null;
            }
            return String.format(Locale.ROOT, "thread pool type for [%s] can only be updated to [%s] but was [%s]", threadPool, defaultThreadPoolType.getType(), threadPoolType.getType());
        }
    }

    class ApplySettings
    implements NodeSettingsService.Listener {
        ApplySettings() {
        }

        @Override
        public void onRefreshSettings(Settings settings) {
            ThreadPool.this.updateSettings(settings);
        }
    }

    public static class Info
    implements Streamable,
    ToXContent {
        private String name;
        private ThreadPoolType type;
        private int min;
        private int max;
        private TimeValue keepAlive;
        private SizeValue queueSize;

        Info() {
        }

        public Info(String name, ThreadPoolType type) {
            this(name, type, -1);
        }

        public Info(String name, ThreadPoolType type, int size) {
            this(name, type, size, size, null, null);
        }

        public Info(String name, ThreadPoolType type, int min, int max, @Nullable TimeValue keepAlive, @Nullable SizeValue queueSize) {
            this.name = name;
            this.type = type;
            this.min = min;
            this.max = max;
            this.keepAlive = keepAlive;
            this.queueSize = queueSize;
        }

        public String getName() {
            return this.name;
        }

        public ThreadPoolType getThreadPoolType() {
            return this.type;
        }

        public int getMin() {
            return this.min;
        }

        public int getMax() {
            return this.max;
        }

        @Nullable
        public TimeValue getKeepAlive() {
            return this.keepAlive;
        }

        @Nullable
        public SizeValue getQueueSize() {
            return this.queueSize;
        }

        @Override
        public void readFrom(StreamInput in) throws IOException {
            this.name = in.readString();
            this.type = ThreadPoolType.fromType(in.readString());
            this.min = in.readInt();
            this.max = in.readInt();
            if (in.readBoolean()) {
                this.keepAlive = TimeValue.readTimeValue(in);
            }
            if (in.readBoolean()) {
                this.queueSize = SizeValue.readSizeValue(in);
            }
            in.readBoolean();
            in.readBoolean();
            in.readBoolean();
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            out.writeString(this.name);
            out.writeString(this.type.getType());
            out.writeInt(this.min);
            out.writeInt(this.max);
            if (this.keepAlive == null) {
                out.writeBoolean(false);
            } else {
                out.writeBoolean(true);
                this.keepAlive.writeTo(out);
            }
            if (this.queueSize == null) {
                out.writeBoolean(false);
            } else {
                out.writeBoolean(true);
                this.queueSize.writeTo(out);
            }
            out.writeBoolean(false);
            out.writeBoolean(false);
            out.writeBoolean(false);
        }

        @Override
        public XContentBuilder toXContent(XContentBuilder builder, ToXContent.Params params) throws IOException {
            builder.startObject(this.name, XContentBuilder.FieldCaseConversion.NONE);
            builder.field(Fields.TYPE, this.type.getType());
            if (this.min != -1) {
                builder.field(Fields.MIN, this.min);
            }
            if (this.max != -1) {
                builder.field(Fields.MAX, this.max);
            }
            if (this.keepAlive != null) {
                builder.field(Fields.KEEP_ALIVE, this.keepAlive.toString());
            }
            if (this.queueSize == null) {
                builder.field(Fields.QUEUE_SIZE, -1);
            } else {
                builder.field(Fields.QUEUE_SIZE, this.queueSize.singles());
            }
            builder.endObject();
            return builder;
        }

        static final class Fields {
            static final XContentBuilderString TYPE = new XContentBuilderString("type");
            static final XContentBuilderString MIN = new XContentBuilderString("min");
            static final XContentBuilderString MAX = new XContentBuilderString("max");
            static final XContentBuilderString KEEP_ALIVE = new XContentBuilderString("keep_alive");
            static final XContentBuilderString QUEUE_SIZE = new XContentBuilderString("queue_size");

            Fields() {
            }
        }
    }

    static class ExecutorHolder {
        private final Executor executor;
        public final Info info;

        ExecutorHolder(Executor executor, Info info) {
            assert (executor instanceof EsThreadPoolExecutor || executor == MoreExecutors.directExecutor());
            this.executor = executor;
            this.info = info;
        }

        Executor executor() {
            return this.executor;
        }
    }

    static class EstimatedTimeThread
    extends Thread {
        final long interval;
        final TimeCounter counter;
        volatile boolean running = true;
        volatile long estimatedTimeInMillis;

        EstimatedTimeThread(String name, long interval) {
            super(name);
            this.interval = interval;
            this.estimatedTimeInMillis = TimeValue.nsecToMSec(System.nanoTime());
            this.counter = new TimeCounter();
            this.setDaemon(true);
        }

        public long estimatedTimeInMillis() {
            return this.estimatedTimeInMillis;
        }

        @Override
        public void run() {
            while (this.running) {
                this.estimatedTimeInMillis = TimeValue.nsecToMSec(System.nanoTime());
                try {
                    Thread.sleep(this.interval);
                }
                catch (InterruptedException e) {
                    this.running = false;
                    return;
                }
            }
        }

        private class TimeCounter
        extends Counter {
            private TimeCounter() {
            }

            @Override
            public long addAndGet(long delta) {
                throw new UnsupportedOperationException();
            }

            @Override
            public long get() {
                return EstimatedTimeThread.this.estimatedTimeInMillis;
            }
        }
    }

    class ThreadedRunnable
    implements Runnable {
        private final Runnable runnable;
        private final Executor executor;

        ThreadedRunnable(Runnable runnable, Executor executor) {
            this.runnable = runnable;
            this.executor = executor;
        }

        @Override
        public void run() {
            this.executor.execute(this.runnable);
        }

        public int hashCode() {
            return this.runnable.hashCode();
        }

        public boolean equals(Object obj) {
            return this.runnable.equals(obj);
        }

        public String toString() {
            return "[threaded] " + this.runnable.toString();
        }
    }

    class LoggingRunnable
    implements Runnable {
        private final Runnable runnable;

        LoggingRunnable(Runnable runnable) {
            this.runnable = runnable;
        }

        @Override
        public void run() {
            try {
                this.runnable.run();
            }
            catch (Throwable t) {
                ThreadPool.this.logger.warn("failed to run {}", t, this.runnable.toString());
                throw t;
            }
        }

        public int hashCode() {
            return this.runnable.hashCode();
        }

        public boolean equals(Object obj) {
            return this.runnable.equals(obj);
        }

        public String toString() {
            return "[threaded] " + this.runnable.toString();
        }
    }

    class ExecutorShutdownListener
    implements EsThreadPoolExecutor.ShutdownListener {
        private ExecutorHolder holder;

        public ExecutorShutdownListener(ExecutorHolder holder) {
            this.holder = holder;
        }

        @Override
        public void onTerminated() {
            ThreadPool.this.retiredExecutors.remove(this.holder);
        }
    }

    private static class ExecutorSettingsBuilder {
        Map<String, String> settings = new HashMap<String, String>();

        public ExecutorSettingsBuilder(String name) {
            this.settings.put("name", name);
            this.settings.put("type", THREAD_POOL_TYPES.get(name).getType());
        }

        public ExecutorSettingsBuilder size(int availableProcessors) {
            return this.add("size", Integer.toString(availableProcessors));
        }

        public ExecutorSettingsBuilder queueSize(int queueSize) {
            return this.add("queue_size", Integer.toString(queueSize));
        }

        public ExecutorSettingsBuilder keepAlive(String keepAlive) {
            return this.add("keep_alive", keepAlive);
        }

        private ExecutorSettingsBuilder add(String key, String value) {
            this.settings.put(key, value);
            return this;
        }

        public Settings build() {
            return Settings.settingsBuilder().put(this.settings).build();
        }
    }

    public static enum ThreadPoolType {
        CACHED("cached"),
        DIRECT("direct"),
        FIXED("fixed"),
        SCALING("scaling");

        private final String type;
        private static final Map<String, ThreadPoolType> TYPE_MAP;

        public String getType() {
            return this.type;
        }

        private ThreadPoolType(String type) {
            this.type = type;
        }

        public static ThreadPoolType fromType(String type) {
            ThreadPoolType threadPoolType = TYPE_MAP.get(type);
            if (threadPoolType == null) {
                throw new IllegalArgumentException("no ThreadPoolType for " + type);
            }
            return threadPoolType;
        }

        static {
            HashMap<String, ThreadPoolType> typeMap = new HashMap<String, ThreadPoolType>();
            for (ThreadPoolType threadPoolType : ThreadPoolType.values()) {
                typeMap.put(threadPoolType.getType(), threadPoolType);
            }
            TYPE_MAP = Collections.unmodifiableMap(typeMap);
        }
    }

    public static class Names {
        public static final String SAME = "same";
        public static final String GENERIC = "generic";
        public static final String LISTENER = "listener";
        public static final String GET = "get";
        public static final String INDEX = "index";
        public static final String BULK = "bulk";
        public static final String SEARCH = "search";
        public static final String SUGGEST = "suggest";
        public static final String PERCOLATE = "percolate";
        public static final String MANAGEMENT = "management";
        public static final String FLUSH = "flush";
        public static final String REFRESH = "refresh";
        public static final String WARMER = "warmer";
        public static final String SNAPSHOT = "snapshot";
        public static final String FORCE_MERGE = "force_merge";
        public static final String FETCH_SHARD_STARTED = "fetch_shard_started";
        public static final String FETCH_SHARD_STORE = "fetch_shard_store";
    }
}

