/*
 * Decompiled with CFR 0.152.
 */
package org.jclouds.blobstore.strategy.internal;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.inject.Inject;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.annotation.Resource;
import javax.inject.Named;
import javax.inject.Singleton;
import org.jclouds.blobstore.BlobStore;
import org.jclouds.blobstore.ContainerNotFoundException;
import org.jclouds.blobstore.domain.PageSet;
import org.jclouds.blobstore.domain.StorageMetadata;
import org.jclouds.blobstore.internal.BlobRuntimeException;
import org.jclouds.blobstore.options.ListContainerOptions;
import org.jclouds.blobstore.strategy.ClearContainerStrategy;
import org.jclouds.blobstore.strategy.ClearListStrategy;
import org.jclouds.http.handlers.BackoffLimitedRetryHandler;
import org.jclouds.logging.Logger;

@Singleton
public class DeleteAllKeysInList
implements ClearListStrategy,
ClearContainerStrategy {
    @Resource
    @Named(value="jclouds.blobstore")
    protected Logger logger = Logger.NULL;
    protected final BackoffLimitedRetryHandler retryHandler;
    private final ListeningExecutorService executorService;
    protected final BlobStore blobStore;
    protected long maxTime = Long.MAX_VALUE;
    protected int maxErrors = 3;
    private int maxParallelDeletes;

    @Inject
    DeleteAllKeysInList(@Named(value="jclouds.user-threads") ListeningExecutorService executorService, BlobStore blobStore, BackoffLimitedRetryHandler retryHandler, @Named(value="jclouds.max-parallel-deletes") int maxParallelDeletes) {
        this.executorService = executorService;
        this.blobStore = blobStore;
        this.retryHandler = retryHandler;
        this.maxParallelDeletes = maxParallelDeletes;
    }

    @Inject(optional=true)
    void setMaxTime(@Named(value="jclouds.request-timeout") long maxTime) {
        this.maxTime = maxTime;
    }

    @Inject(optional=true)
    void setMaxErrors(@Named(value="jclouds.max-retries") int maxErrors) {
        this.maxErrors = maxErrors;
    }

    @Override
    public void execute(String containerName) {
        this.execute(containerName, ListContainerOptions.Builder.recursive());
    }

    private boolean parentIsFolder(ListContainerOptions options, StorageMetadata md) {
        return options.getDir() != null && md.getName().indexOf(47) == -1;
    }

    private void cancelOutstandingFutures(Set<ListenableFuture<Void>> outstandingFutures) {
        for (ListenableFuture<Void> future : outstandingFutures) {
            future.cancel(true);
        }
    }

    private String getMessage(String containerName, ListContainerOptions options) {
        return options.getDir() != null ? String.format("clearing path %s/%s", containerName, options.getDir()) : String.format("clearing container %s", containerName);
    }

    private PageSet<? extends StorageMetadata> getListing(String containerName, ListContainerOptions options, Semaphore semaphore, Set<ListenableFuture<Void>> outstandingFutures, AtomicBoolean deleteFailure) {
        PageSet<? extends StorageMetadata> listing = null;
        try {
            listing = this.blobStore.list(containerName, options);
        }
        catch (ContainerNotFoundException ce) {
            return listing;
        }
        if (options.isRecursive()) {
            for (StorageMetadata storageMetadata : listing) {
                String fullPath = this.parentIsFolder(options, storageMetadata) ? options.getDir() + "/" + storageMetadata.getName() : storageMetadata.getName();
                switch (storageMetadata.getType()) {
                    case BLOB: {
                        break;
                    }
                    case FOLDER: 
                    case RELATIVE_PATH: {
                        if (fullPath.equals(options.getDir())) break;
                        this.executeOneIteration(containerName, options.clone().inDirectory(fullPath), semaphore, outstandingFutures, deleteFailure, true);
                        break;
                    }
                    case CONTAINER: {
                        throw new IllegalArgumentException("Container type not supported");
                    }
                }
            }
        }
        return listing;
    }

    private ListenableFuture<Void> deleteDirectory(ListContainerOptions options, final String containerName, final String dirName) {
        ListenableFuture<Void> blobDelFuture = options.isRecursive() ? this.executorService.submit(new Callable<Void>(){

            @Override
            public Void call() {
                DeleteAllKeysInList.this.blobStore.deleteDirectory(containerName, dirName);
                return null;
            }
        }) : null;
        return blobDelFuture;
    }

    private void deleteBlobsAndEmptyDirs(final String containerName, ListContainerOptions options, PageSet<? extends StorageMetadata> listing, final Semaphore semaphore, final AtomicBoolean deleteFailure, final Set<ListenableFuture<Void>> outstandingFutures) throws TimeoutException {
        for (StorageMetadata storageMetadata : listing) {
            ListenableFuture<Void> blobDelFuture;
            final String fullPath = this.parentIsFolder(options, storageMetadata) ? options.getDir() + "/" + storageMetadata.getName() : storageMetadata.getName();
            try {
                if (!semaphore.tryAcquire(this.maxTime, TimeUnit.MILLISECONDS)) {
                    throw new TimeoutException("Timeout waiting for semaphore");
                }
            }
            catch (InterruptedException ie) {
                this.logger.debug("Interrupted while deleting blobs", new Object[0]);
                Thread.currentThread().interrupt();
            }
            switch (storageMetadata.getType()) {
                case BLOB: 
                case FOLDER: {
                    blobDelFuture = this.executorService.submit(new Callable<Void>(){

                        @Override
                        public Void call() {
                            DeleteAllKeysInList.this.blobStore.removeBlob(containerName, fullPath);
                            return null;
                        }
                    });
                    break;
                }
                case RELATIVE_PATH: {
                    blobDelFuture = this.deleteDirectory(options, containerName, storageMetadata.getName());
                    break;
                }
                case CONTAINER: {
                    throw new IllegalArgumentException("Container type not supported");
                }
                default: {
                    blobDelFuture = null;
                }
            }
            if (blobDelFuture != null) {
                outstandingFutures.add(blobDelFuture);
                Futures.addCallback(blobDelFuture, new FutureCallback<Object>(){

                    @Override
                    public void onSuccess(Object o) {
                        outstandingFutures.remove(blobDelFuture);
                        semaphore.release();
                    }

                    @Override
                    public void onFailure(Throwable t) {
                        deleteFailure.set(true);
                        outstandingFutures.remove(blobDelFuture);
                        semaphore.release();
                    }
                }, MoreExecutors.directExecutor());
                continue;
            }
            semaphore.release();
        }
    }

    @VisibleForTesting
    void executeOneIteration(String containerName, ListContainerOptions listOptions, Semaphore semaphore, Set<ListenableFuture<Void>> outstandingFutures, AtomicBoolean deleteFailure, boolean blocking) {
        ListContainerOptions options = listOptions.clone();
        String message = this.getMessage(containerName, listOptions);
        if (options.isRecursive()) {
            message = message + " recursively";
        }
        this.logger.debug(message, new Object[0]);
        PageSet<? extends StorageMetadata> listing = this.getListing(containerName, options, semaphore, outstandingFutures, deleteFailure);
        while (listing != null && !listing.isEmpty()) {
            try {
                this.deleteBlobsAndEmptyDirs(containerName, options, listing, semaphore, deleteFailure, outstandingFutures);
            }
            catch (TimeoutException te) {
                this.logger.debug("TimeoutException while deleting blobs: {}", te.getMessage());
                this.cancelOutstandingFutures(outstandingFutures);
                deleteFailure.set(true);
            }
            String marker = listing.getNextMarker();
            if (marker == null) break;
            this.logger.debug("%s with marker %s", message, marker);
            options = options.afterMarker(marker);
            listing = this.getListing(containerName, options, semaphore, outstandingFutures, deleteFailure);
        }
        if (blocking) {
            this.waitForCompletion(semaphore, outstandingFutures);
        }
    }

    private void waitForCompletion(Semaphore semaphore, Set<ListenableFuture<Void>> outstandingFutures) {
        try {
            semaphore.acquire(this.maxParallelDeletes);
            semaphore.release(this.maxParallelDeletes);
        }
        catch (InterruptedException e) {
            this.logger.debug("Interrupted while waiting for blobs to be deleted", new Object[0]);
            this.cancelOutstandingFutures(outstandingFutures);
            Thread.currentThread().interrupt();
        }
    }

    @Override
    public void execute(String containerName, ListContainerOptions listOptions) {
        AtomicBoolean deleteFailure = new AtomicBoolean();
        int retries = this.maxErrors;
        Semaphore semaphore = new Semaphore(this.maxParallelDeletes);
        Set<ListenableFuture<Void>> outstandingFutures = Collections.synchronizedSet(new HashSet());
        while (retries > 0) {
            deleteFailure.set(false);
            this.executeOneIteration(containerName, listOptions, semaphore, outstandingFutures, deleteFailure, false);
            this.waitForCompletion(semaphore, outstandingFutures);
            if (!deleteFailure.get() || --retries <= 0) break;
            String message = this.getMessage(containerName, listOptions);
            this.retryHandler.imposeBackoffExponentialDelay(this.maxErrors - retries, message);
        }
        if (retries == 0) {
            this.cancelOutstandingFutures(outstandingFutures);
            throw new BlobRuntimeException("Exceeded maximum retry attempts");
        }
    }
}

