/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.core.security.authentication.token;

import java.io.UnsupportedEncodingException;
import java.security.NoSuchAlgorithmException;
import java.security.Principal;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.jcr.AccessDeniedException;
import javax.jcr.Node;
import javax.jcr.Property;
import javax.jcr.PropertyIterator;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.SimpleCredentials;
import javax.jcr.Value;
import javax.jcr.ValueFactory;
import org.apache.jackrabbit.api.security.authentication.token.TokenCredentials;
import org.apache.jackrabbit.api.security.principal.ItemBasedPrincipal;
import org.apache.jackrabbit.api.security.user.Authorizable;
import org.apache.jackrabbit.api.security.user.User;
import org.apache.jackrabbit.api.security.user.UserManager;
import org.apache.jackrabbit.core.NodeImpl;
import org.apache.jackrabbit.core.ProtectedItemModifier;
import org.apache.jackrabbit.core.SessionImpl;
import org.apache.jackrabbit.core.id.NodeId;
import org.apache.jackrabbit.core.security.authentication.token.TokenInfo;
import org.apache.jackrabbit.core.security.user.PasswordUtility;
import org.apache.jackrabbit.core.security.user.UserImpl;
import org.apache.jackrabbit.spi.Name;
import org.apache.jackrabbit.spi.commons.name.NameConstants;
import org.apache.jackrabbit.util.ISO8601;
import org.apache.jackrabbit.util.Text;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TokenProvider
extends ProtectedItemModifier {
    private static final Logger log = LoggerFactory.getLogger(TokenProvider.class);
    private static final String TOKEN_ATTRIBUTE = ".token";
    private static final String TOKEN_ATTRIBUTE_EXPIRY = "rep:token.exp";
    private static final String TOKEN_ATTRIBUTE_KEY = "rep:token.key";
    private static final String TOKENS_NODE_NAME = ".tokens";
    private static final String TOKEN_NT_NAME = "rep:Token";
    private static final Name TOKENS_NT_NAME = NameConstants.NT_UNSTRUCTURED;
    private static final char DELIM = '_';
    private static final Set<String> RESERVED_ATTRIBUTES = new HashSet<String>(3);
    private static final Collection<String> RESERVED_PREFIXES;
    private final SessionImpl session;
    private final UserManager userManager;
    private final long tokenExpiration;

    TokenProvider(SessionImpl session, long tokenExpiration) throws RepositoryException {
        this.session = session;
        this.userManager = session.getUserManager();
        this.tokenExpiration = tokenExpiration;
    }

    public TokenInfo createToken(User user, SimpleCredentials sc) throws RepositoryException {
        TokenInfo tokenInfo = null;
        if (sc != null && user != null && user.getID().equalsIgnoreCase(sc.getUserID())) {
            String[] attrNames = sc.getAttributeNames();
            HashMap<String, String> attributes = new HashMap<String, String>(attrNames.length);
            for (String attrName : sc.getAttributeNames()) {
                attributes.put(attrName, sc.getAttribute(attrName).toString());
            }
            tokenInfo = this.createToken(user, attributes);
            if (tokenInfo != null) {
                sc.setAttribute(TOKEN_ATTRIBUTE, tokenInfo.getToken());
            }
        }
        return tokenInfo;
    }

    private TokenInfo createToken(User user, Map<String, ?> attributes) throws RepositoryException {
        String error = "Failed to create login token. ";
        NodeImpl tokenParent = this.getTokenParent(user);
        if (tokenParent != null) {
            try {
                ValueFactory vf = this.session.getValueFactory();
                long creationTime = new Date().getTime();
                Calendar creation = GregorianCalendar.getInstance();
                creation.setTimeInMillis(creationTime);
                Name tokenName = this.session.getQName(Text.replace(ISO8601.format(creation), ":", "."));
                NodeImpl tokenNode = super.addNode(tokenParent, tokenName, this.session.getQName(TOKEN_NT_NAME), NodeId.randomId());
                String key = TokenProvider.generateKey(8);
                String token = tokenNode.getId().toString() + '_' + key;
                String keyHash = PasswordUtility.buildPasswordHash(TokenProvider.getKeyValue(key, user.getID()));
                this.setProperty(tokenNode, this.session.getQName(TOKEN_ATTRIBUTE_KEY), vf.createValue(keyHash));
                this.setProperty(tokenNode, this.session.getQName(TOKEN_ATTRIBUTE_EXPIRY), this.createExpirationValue(creationTime, this.session));
                for (String name : attributes.keySet()) {
                    if (RESERVED_ATTRIBUTES.contains(name)) continue;
                    String attr = attributes.get(name).toString();
                    this.setProperty(tokenNode, this.session.getQName(name), vf.createValue(attr));
                }
                this.session.save();
                return new TokenInfoImpl(tokenNode, token, user.getID());
            }
            catch (NoSuchAlgorithmException e) {
                log.error(error, (Throwable)e);
            }
            catch (UnsupportedEncodingException e) {
                log.error(error, (Throwable)e);
            }
            catch (AccessDeniedException e) {
                log.warn(error, (Throwable)e);
            }
        } else {
            log.warn("Unable to get/create token store for user {}", (Object)user.getID());
        }
        return null;
    }

    private Value createExpirationValue(long creationTime, Session session) throws RepositoryException {
        Calendar cal = Calendar.getInstance();
        cal.setTimeInMillis(TokenProvider.createExpirationTime(creationTime, this.tokenExpiration));
        return session.getValueFactory().createValue(cal);
    }

    public TokenInfo getTokenInfo(String token) throws RepositoryException {
        if (token == null) {
            return null;
        }
        NodeImpl tokenNode = (NodeImpl)TokenProvider.getTokenNode(token, this.session);
        String userId = TokenProvider.getUserId(tokenNode, this.userManager);
        if (userId == null || !TokenProvider.isValidTokenTree(tokenNode)) {
            return null;
        }
        return new TokenInfoImpl(tokenNode, token, userId);
    }

    static Node getTokenNode(String token, Session session) throws RepositoryException {
        int pos = token.indexOf(95);
        String id = pos == -1 ? token : token.substring(0, pos);
        return session.getNodeByIdentifier(id);
    }

    static String getUserId(NodeImpl tokenNode, UserManager userManager) throws RepositoryException {
        if (tokenNode != null) {
            final NodeImpl userNode = (NodeImpl)tokenNode.getParent().getParent();
            final String principalName = userNode.getProperty(UserImpl.P_PRINCIPAL_NAME).getString();
            if (userNode.isNodeType(UserImpl.NT_REP_USER)) {
                Authorizable a = userManager.getAuthorizable(new ItemBasedPrincipal(){

                    @Override
                    public String getPath() throws RepositoryException {
                        return userNode.getPath();
                    }

                    @Override
                    public String getName() {
                        return principalName;
                    }
                });
                if (a != null && !a.isGroup() && !((User)a).isDisabled()) {
                    return a.getID();
                }
            } else {
                throw new RepositoryException("Failed to calculate userId from token credentials");
            }
        }
        return null;
    }

    static boolean isMandatoryAttribute(String attributeName) {
        return attributeName != null && attributeName.startsWith(TOKEN_ATTRIBUTE);
    }

    static boolean isInfoAttribute(String attributeName) {
        String prefix = Text.getNamespacePrefix(attributeName);
        return !RESERVED_PREFIXES.contains(prefix);
    }

    private static long createExpirationTime(long creationTime, long tokenExpiration) {
        return creationTime + tokenExpiration;
    }

    private static long getExpirationTime(NodeImpl tokenNode, long defaultValue) throws RepositoryException {
        if (tokenNode.hasProperty(TOKEN_ATTRIBUTE_EXPIRY)) {
            return tokenNode.getProperty(TOKEN_ATTRIBUTE_EXPIRY).getLong();
        }
        return defaultValue;
    }

    private static String generateKey(int size) {
        SecureRandom random = new SecureRandom();
        byte[] key = new byte[size];
        random.nextBytes(key);
        StringBuilder res = new StringBuilder(key.length * 2);
        for (byte b : key) {
            res.append(Text.hexTable[b >> 4 & 0xF]);
            res.append(Text.hexTable[b & 0xF]);
        }
        return res.toString();
    }

    private static String getKeyValue(String key, String userId) {
        return key + userId;
    }

    private static boolean isValidTokenTree(NodeImpl tokenNode) throws RepositoryException {
        if (tokenNode == null) {
            return false;
        }
        return TOKENS_NODE_NAME.equals(tokenNode.getParent().getName()) && TOKEN_NT_NAME.equals(tokenNode.getPrimaryNodeType().getName());
    }

    private NodeImpl getTokenParent(User user) throws RepositoryException {
        NodeImpl tokenParent;
        block7: {
            tokenParent = null;
            String parentPath = null;
            try {
                if (user != null) {
                    Principal pr = user.getPrincipal();
                    if (pr instanceof ItemBasedPrincipal) {
                        String userPath = ((ItemBasedPrincipal)pr).getPath();
                        NodeImpl userNode = (NodeImpl)this.session.getNode(userPath);
                        if (userNode.hasNode(TOKENS_NODE_NAME)) {
                            tokenParent = (NodeImpl)userNode.getNode(TOKENS_NODE_NAME);
                        } else {
                            tokenParent = userNode.addNode(this.session.getQName(TOKENS_NODE_NAME), TOKENS_NT_NAME, NodeId.randomId());
                            parentPath = userPath + '/' + TOKENS_NODE_NAME;
                            this.session.save();
                        }
                    }
                } else {
                    log.debug("Cannot create login token: No user specified. (null)");
                }
            }
            catch (RepositoryException e) {
                log.debug("Conflict while creating token store -> retrying", (Throwable)e);
                this.session.refresh(false);
                if (parentPath == null || !this.session.nodeExists(parentPath)) break block7;
                tokenParent = (NodeImpl)this.session.getNode(parentPath);
            }
        }
        return tokenParent;
    }

    static {
        RESERVED_ATTRIBUTES.add(TOKEN_ATTRIBUTE);
        RESERVED_ATTRIBUTES.add(TOKEN_ATTRIBUTE_EXPIRY);
        RESERVED_ATTRIBUTES.add(TOKEN_ATTRIBUTE_KEY);
        RESERVED_PREFIXES = Collections.unmodifiableList(Arrays.asList("xml", "jcr", "nt", "mix", "xmlns", "rep", "sv"));
    }

    private class TokenInfoImpl
    implements TokenInfo {
        private final String token;
        private final String tokenPath;
        private final String userId;
        private final long expirationTime;
        private final String key;
        private final Map<String, String> mandatoryAttributes;
        private final Map<String, String> publicAttributes;

        private TokenInfoImpl(NodeImpl tokenNode, String token, String userId) throws RepositoryException {
            this.token = token;
            this.tokenPath = tokenNode.getPath();
            this.userId = userId;
            this.expirationTime = TokenProvider.getExpirationTime(tokenNode, Long.MIN_VALUE);
            this.key = tokenNode.getProperty(TokenProvider.TOKEN_ATTRIBUTE_KEY).getString();
            this.mandatoryAttributes = new HashMap<String, String>();
            this.publicAttributes = new HashMap<String, String>();
            PropertyIterator pit = tokenNode.getProperties();
            while (pit.hasNext()) {
                Property property = pit.nextProperty();
                String name = property.getName();
                String value = property.getString();
                if (RESERVED_ATTRIBUTES.contains(name)) continue;
                if (TokenProvider.isMandatoryAttribute(name)) {
                    this.mandatoryAttributes.put(name, value);
                    continue;
                }
                if (!TokenProvider.isInfoAttribute(name)) continue;
                this.publicAttributes.put(name, value);
            }
        }

        @Override
        public String getToken() {
            return this.token;
        }

        @Override
        public boolean isExpired(long loginTime) {
            return this.expirationTime < loginTime;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public boolean resetExpiration(long loginTime) throws RepositoryException {
            if (this.isExpired(loginTime)) {
                log.debug("Attempt to reset an expired token.");
                return false;
            }
            Session s = null;
            try {
                if (this.expirationTime - loginTime <= TokenProvider.this.tokenExpiration / 2L) {
                    s = TokenProvider.this.session.createSession(TokenProvider.this.session.getWorkspace().getName());
                    TokenProvider.this.setProperty((NodeImpl)s.getNode(this.tokenPath), TokenProvider.this.session.getQName(TokenProvider.TOKEN_ATTRIBUTE_EXPIRY), TokenProvider.this.createExpirationValue(loginTime, TokenProvider.this.session));
                    s.save();
                    log.debug("Successfully reset token expiration time.");
                    boolean bl = true;
                    return bl;
                }
            }
            catch (RepositoryException e) {
                log.warn("Error while resetting token expiration", (Throwable)e);
            }
            finally {
                if (s != null) {
                    s.logout();
                }
            }
            return false;
        }

        @Override
        public boolean matches(TokenCredentials tokenCredentials) {
            String tk = tokenCredentials.getToken();
            int pos = tk.lastIndexOf(95);
            if (pos > -1) {
                tk = tk.substring(pos + 1);
            }
            if (this.key == null || !PasswordUtility.isSame(this.key, TokenProvider.getKeyValue(tk, this.userId))) {
                return false;
            }
            for (String name : this.mandatoryAttributes.keySet()) {
                String expectedValue = this.mandatoryAttributes.get(name);
                if (expectedValue.equals(tokenCredentials.getAttribute(name))) continue;
                return false;
            }
            List<String> attrNames = Arrays.asList(tokenCredentials.getAttributeNames());
            for (String name : this.publicAttributes.keySet()) {
                if (attrNames.contains(name)) continue;
                tokenCredentials.setAttribute(name, this.publicAttributes.get(name).toString());
            }
            return true;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public boolean remove() {
            Session s = null;
            try {
                s = TokenProvider.this.session.createSession(TokenProvider.this.session.getWorkspace().getName());
                Node node = s.getNode(this.tokenPath);
                node.remove();
                s.save();
                boolean bl = true;
                return bl;
            }
            catch (RepositoryException e) {
                log.warn("Internal error while removing token node.", (Throwable)e);
            }
            finally {
                if (s != null) {
                    s.logout();
                }
            }
            return false;
        }

        @Override
        public TokenCredentials getCredentials() {
            TokenCredentials tc = new TokenCredentials(this.token);
            for (String name : this.mandatoryAttributes.keySet()) {
                tc.setAttribute(name, this.mandatoryAttributes.get(name));
            }
            for (String name : this.publicAttributes.keySet()) {
                tc.setAttribute(name, this.publicAttributes.get(name));
            }
            return tc;
        }
    }
}

