/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.component.splunk.support;

import com.splunk.Event;
import com.splunk.Job;
import com.splunk.JobArgs;
import com.splunk.JobResultsArgs;
import com.splunk.ResultsReader;
import com.splunk.ResultsReaderJson;
import com.splunk.SavedSearch;
import com.splunk.SavedSearchCollection;
import com.splunk.SavedSearchDispatchArgs;
import com.splunk.Service;
import com.splunk.ServiceArgs;
import java.io.Closeable;
import java.io.InputStream;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import org.apache.camel.component.splunk.ConsumerType;
import org.apache.camel.component.splunk.SplunkEndpoint;
import org.apache.camel.component.splunk.event.SplunkEvent;
import org.apache.camel.component.splunk.support.SplunkResultProcessor;
import org.apache.camel.util.IOHelper;
import org.apache.camel.util.ObjectHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SplunkDataReader {
    private static final Logger LOG = LoggerFactory.getLogger(SplunkDataReader.class);
    private static final String DATE_FORMAT = "MM/dd/yy HH:mm:ss:SSS";
    private static final String SPLUNK_TIME_FORMAT = "%m/%d/%y %H:%M:%S:%3N";
    private transient Calendar lastSuccessfulReadTime;
    private SplunkEndpoint endpoint;
    private ConsumerType consumerType;

    public SplunkDataReader(SplunkEndpoint endpoint, ConsumerType consumerType) {
        this.endpoint = endpoint;
        this.consumerType = consumerType;
    }

    public int getCount() {
        return this.endpoint.getConfiguration().getCount();
    }

    public String getSearch() {
        return this.endpoint.getConfiguration().getSearch();
    }

    public String getEarliestTime() {
        return this.endpoint.getConfiguration().getEarliestTime();
    }

    public String getLatestTime() {
        return this.endpoint.getConfiguration().getLatestTime();
    }

    public String getInitEarliestTime() {
        return this.endpoint.getConfiguration().getInitEarliestTime();
    }

    private String getSavedSearch() {
        return this.endpoint.getConfiguration().getSavedSearch();
    }

    public List<SplunkEvent> read() throws Exception {
        return this.read(null);
    }

    public List<SplunkEvent> read(SplunkResultProcessor callback) throws Exception {
        switch (this.consumerType) {
            case NORMAL: {
                return this.nonBlockingSearch(callback);
            }
            case REALTIME: {
                return this.realtimeSearch(callback);
            }
            case SAVEDSEARCH: {
                return this.savedSearch(callback);
            }
        }
        throw new RuntimeException("Unknown search mode " + (Object)((Object)this.consumerType));
    }

    private String calculateEarliestTime(Calendar startTime, boolean realtime) {
        String result;
        if (realtime) {
            result = this.calculateEarliestTimeForRealTime(startTime);
        } else {
            SimpleDateFormat df = new SimpleDateFormat(DATE_FORMAT);
            result = df.format(this.lastSuccessfulReadTime.getTime());
        }
        return result;
    }

    private String calculateEarliestTimeForRealTime(Calendar startTime) {
        long diff = startTime.getTimeInMillis() - this.lastSuccessfulReadTime.getTimeInMillis();
        String result = "-" + diff / 1000L + "s";
        return result;
    }

    private void populateArgs(JobArgs queryArgs, Calendar startTime, boolean realtime) {
        String latestTime;
        String earliestTime = this.getEarliestTime(startTime, realtime);
        if (ObjectHelper.isNotEmpty((Object)earliestTime)) {
            queryArgs.setEarliestTime(earliestTime);
        }
        if (ObjectHelper.isNotEmpty((Object)(latestTime = this.getLatestTime(startTime, realtime)))) {
            queryArgs.setLatestTime(latestTime);
        }
        queryArgs.setTimeFormat(SPLUNK_TIME_FORMAT);
    }

    private String getLatestTime(Calendar startTime, boolean realtime) {
        String lTime;
        if (ObjectHelper.isNotEmpty((Object)this.getLatestTime())) {
            lTime = this.getLatestTime();
        } else if (realtime) {
            lTime = "rt";
        } else {
            SimpleDateFormat df = new SimpleDateFormat(DATE_FORMAT);
            lTime = df.format(startTime.getTime());
        }
        return lTime;
    }

    private String getEarliestTime(Calendar startTime, boolean realtime) {
        String eTime = null;
        if (this.lastSuccessfulReadTime == null) {
            eTime = this.getInitEarliestTime();
        } else if (ObjectHelper.isNotEmpty((Object)this.getEarliestTime())) {
            eTime = this.getEarliestTime();
        } else {
            String calculatedEarliestTime = this.calculateEarliestTime(startTime, realtime);
            if (calculatedEarliestTime != null) {
                eTime = realtime ? "rt" + calculatedEarliestTime : calculatedEarliestTime;
            }
        }
        return eTime;
    }

    private List<SplunkEvent> savedSearch(SplunkResultProcessor callback) throws Exception {
        LOG.trace("saved search start");
        ServiceArgs queryArgs = new ServiceArgs();
        queryArgs.setApp("search");
        if (ObjectHelper.isNotEmpty((Object)this.endpoint.getConfiguration().getOwner())) {
            queryArgs.setOwner(this.endpoint.getConfiguration().getOwner());
        }
        if (ObjectHelper.isNotEmpty((Object)this.endpoint.getConfiguration().getApp())) {
            queryArgs.setApp(this.endpoint.getConfiguration().getApp());
        }
        Calendar startTime = Calendar.getInstance();
        SavedSearch search = null;
        Job job = null;
        String latestTime = this.getLatestTime(startTime, false);
        String earliestTime = this.getEarliestTime(startTime, false);
        Service service = this.endpoint.getService();
        SavedSearchCollection savedSearches = service.getSavedSearches(queryArgs);
        for (SavedSearch s : savedSearches.values()) {
            if (!s.getName().equals(this.getSavedSearch())) continue;
            search = s;
            break;
        }
        if (search != null) {
            SavedSearchDispatchArgs args = new SavedSearchDispatchArgs();
            args.setForceDispatch(true);
            args.setDispatchEarliestTime(earliestTime);
            args.setDispatchLatestTime(latestTime);
            job = search.dispatch(args);
        }
        while (!job.isDone()) {
            Thread.sleep(2000L);
        }
        List<SplunkEvent> data = this.extractData(job, false, callback);
        this.lastSuccessfulReadTime = startTime;
        return data;
    }

    private List<SplunkEvent> nonBlockingSearch(SplunkResultProcessor callback) throws Exception {
        LOG.debug("non block search start");
        JobArgs queryArgs = new JobArgs();
        queryArgs.setExecutionMode(JobArgs.ExecutionMode.NORMAL);
        Calendar startTime = Calendar.getInstance();
        this.populateArgs(queryArgs, startTime, false);
        List<SplunkEvent> data = this.runQuery(queryArgs, false, callback);
        this.lastSuccessfulReadTime = startTime;
        return data;
    }

    private List<SplunkEvent> realtimeSearch(SplunkResultProcessor callback) throws Exception {
        LOG.debug("realtime search start");
        JobArgs queryArgs = new JobArgs();
        queryArgs.setExecutionMode(JobArgs.ExecutionMode.NORMAL);
        queryArgs.setSearchMode(JobArgs.SearchMode.REALTIME);
        Calendar startTime = Calendar.getInstance();
        this.populateArgs(queryArgs, startTime, true);
        List<SplunkEvent> data = this.runQuery(queryArgs, true, callback);
        this.lastSuccessfulReadTime = startTime;
        return data;
    }

    private List<SplunkEvent> runQuery(JobArgs queryArgs, boolean realtime, SplunkResultProcessor callback) throws Exception {
        Service service = this.endpoint.getService();
        Job job = service.getJobs().create(this.getSearch(), queryArgs);
        LOG.debug("Running search : {} with queryArgs : {}", (Object)this.getSearch(), (Object)queryArgs);
        if (realtime) {
            while (!job.isReady()) {
                Thread.sleep(500L);
            }
            Thread.sleep(4000L);
        } else {
            while (!job.isDone()) {
                Thread.sleep(500L);
            }
        }
        return this.extractData(job, realtime, callback);
    }

    private List<SplunkEvent> extractData(Job job, boolean realtime, SplunkResultProcessor callback) throws Exception {
        ArrayList<SplunkEvent> result = new ArrayList<SplunkEvent>();
        ResultsReader resultsReader = null;
        int total = 0;
        total = realtime ? job.getResultPreviewCount() : job.getResultCount();
        if (this.getCount() == 0 || total < this.getCount()) {
            Event data;
            InputStream stream = null;
            JobResultsArgs outputArgs = new JobResultsArgs();
            outputArgs.setOutputMode(JobResultsArgs.OutputMode.JSON);
            if (realtime) {
                if (this.getCount() > 0) {
                    outputArgs.setCount(this.getCount());
                }
                stream = job.getResultsPreview(outputArgs);
            } else {
                stream = job.getResults(outputArgs);
            }
            resultsReader = new ResultsReaderJson(stream);
            while ((data = resultsReader.getNextEvent()) != null) {
                SplunkEvent splunkData = new SplunkEvent(data);
                if (callback != null) {
                    callback.process(splunkData);
                    continue;
                }
                result.add(splunkData);
            }
            IOHelper.close((Closeable)stream);
        } else {
            for (int offset = 0; offset < total; offset += this.getCount()) {
                Event data;
                JobResultsArgs outputArgs = new JobResultsArgs();
                outputArgs.setOutputMode(JobResultsArgs.OutputMode.JSON);
                outputArgs.setCount(this.getCount());
                outputArgs.setOffset(offset);
                InputStream stream = realtime ? job.getResultsPreview(outputArgs) : job.getResults(outputArgs);
                resultsReader = new ResultsReaderJson(stream);
                while ((data = resultsReader.getNextEvent()) != null) {
                    SplunkEvent splunkData = new SplunkEvent(data);
                    if (callback != null) {
                        callback.process(splunkData);
                        continue;
                    }
                    result.add(splunkData);
                }
                IOHelper.close((Closeable)stream);
            }
        }
        if (resultsReader != null) {
            resultsReader.close();
        }
        job.cancel();
        return result;
    }
}

