/*
 * Decompiled with CFR 0.152.
 */
package org.talend.dataprep.api.filter;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.text.ParseException;
import java.time.DateTimeException;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Iterator;
import java.util.List;
import java.util.function.Predicate;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.talend.daikon.exception.TalendRuntimeException;
import org.talend.daikon.exception.error.ErrorCode;
import org.talend.daikon.number.BigDecimalParser;
import org.talend.dataprep.BaseErrorCodes;
import org.talend.dataprep.api.dataset.ColumnMetadata;
import org.talend.dataprep.api.dataset.RowMetadata;
import org.talend.dataprep.api.dataset.row.DataSetRow;
import org.talend.dataprep.api.filter.FilterService;
import org.talend.dataprep.api.type.Type;
import org.talend.dataprep.date.DateManipulator;
import org.talend.dataprep.quality.AnalyzerService;
import org.talend.dataprep.transformation.actions.Providers;
import org.talend.dataprep.transformation.actions.date.DateParser;
import org.talend.dataprep.util.NumericHelper;

public class SimpleFilterService
implements FilterService {
    private static final String EQ = "eq";
    private static final String GT = "gt";
    private static final String LT = "lt";
    private static final String GTE = "gte";
    private static final String LTE = "lte";
    private static final String CONTAINS = "contains";
    private static final String MATCHES = "matches";
    private static final String INVALID = "invalid";
    private static final String VALID = "valid";
    private static final String EMPTY = "empty";
    private static final String RANGE = "range";
    private static final String AND = "and";
    private static final String OR = "or";
    private static final String NOT = "not";
    private static final Logger LOGGER = LoggerFactory.getLogger(SimpleFilterService.class);
    private DateParser dateParser;

    private static Predicate<DataSetRow> safeDate(Predicate<DataSetRow> inner) {
        return r -> {
            try {
                return inner.test((DataSetRow)r);
            }
            catch (DateTimeException e) {
                LOGGER.debug("Unable to parse date.", (Throwable)e);
                return false;
            }
        };
    }

    @Override
    public Predicate<DataSetRow> build(String filterAsString, RowMetadata rowMetadata) {
        if (StringUtils.isEmpty((String)filterAsString)) {
            return r -> true;
        }
        try {
            ObjectMapper mapper = new ObjectMapper();
            JsonNode root = mapper.reader().readTree(filterAsString);
            Iterator elements = root.elements();
            if (!elements.hasNext()) {
                throw new IllegalArgumentException("Malformed filter: " + filterAsString);
            }
            return this.buildFilter(root, rowMetadata);
        }
        catch (Exception e) {
            throw new TalendRuntimeException((ErrorCode)BaseErrorCodes.UNABLE_TO_PARSE_FILTER, (Throwable)e);
        }
    }

    private Predicate<DataSetRow> buildFilter(JsonNode currentNode, RowMetadata rowMetadata) {
        Iterator children = currentNode.elements();
        JsonNode operationContent = (JsonNode)children.next();
        String columnId = operationContent.has("field") ? operationContent.get("field").asText() : null;
        String value = operationContent.has("value") ? operationContent.get("value").asText() : null;
        Iterator propertiesIterator = currentNode.fieldNames();
        if (!propertiesIterator.hasNext()) {
            throw new UnsupportedOperationException("Unsupported query, empty filter definition: " + currentNode.toString());
        }
        String operation = (String)propertiesIterator.next();
        if (columnId == null && SimpleFilterService.allowFullFilter(operation)) {
            Predicate<DataSetRow> predicate;
            List<ColumnMetadata> columns = rowMetadata.getColumns();
            if (!columns.isEmpty()) {
                predicate = this.buildOperationFilter(currentNode, rowMetadata, columns.get(0).getId(), operation, value);
                for (int i = 1; i < columns.size(); ++i) {
                    predicate = predicate.or(this.buildOperationFilter(currentNode, rowMetadata, columns.get(i).getId(), operation, value));
                }
            } else {
                predicate = dsr -> true;
            }
            return predicate;
        }
        return this.buildOperationFilter(currentNode, rowMetadata, columnId, operation, value);
    }

    private static boolean allowFullFilter(String operation) {
        switch (operation) {
            case "eq": 
            case "gt": 
            case "lt": 
            case "gte": 
            case "lte": 
            case "contains": 
            case "matches": 
            case "invalid": 
            case "valid": 
            case "empty": 
            case "range": {
                return true;
            }
        }
        return false;
    }

    private Predicate<DataSetRow> buildOperationFilter(JsonNode currentNode, RowMetadata rowMetadata, String columnId, String operation, String value) {
        switch (operation) {
            case "eq": {
                return this.createEqualsPredicate(currentNode, columnId, value);
            }
            case "gt": {
                return this.createGreaterThanPredicate(currentNode, columnId, value);
            }
            case "lt": {
                return this.createLowerThanPredicate(currentNode, columnId, value);
            }
            case "gte": {
                return this.createGreaterOrEqualsPredicate(currentNode, columnId, value);
            }
            case "lte": {
                return this.createLowerOrEqualsPredicate(currentNode, columnId, value);
            }
            case "contains": {
                return this.createContainsPredicate(currentNode, columnId, value);
            }
            case "matches": {
                return this.createMatchesPredicate(currentNode, columnId, value);
            }
            case "invalid": {
                return this.createInvalidPredicate(columnId);
            }
            case "valid": {
                return this.createValidPredicate(columnId);
            }
            case "empty": {
                return this.createEmptyPredicate(columnId);
            }
            case "range": {
                return this.createRangePredicate(columnId, (JsonNode)currentNode.elements().next(), rowMetadata);
            }
            case "and": {
                return this.createAndPredicate((JsonNode)currentNode.elements().next(), rowMetadata);
            }
            case "or": {
                return this.createOrPredicate((JsonNode)currentNode.elements().next(), rowMetadata);
            }
            case "not": {
                return this.createNotPredicate((JsonNode)currentNode.elements().next(), rowMetadata);
            }
        }
        throw new UnsupportedOperationException("Unsupported query, unknown filter '" + operation + "': " + currentNode.toString());
    }

    private Predicate<DataSetRow> createAndPredicate(JsonNode nodeContent, RowMetadata rowMetadata) {
        this.checkValidMultiPredicate(nodeContent);
        Predicate<DataSetRow> leftFilter = this.buildFilter(nodeContent.get(0), rowMetadata);
        Predicate<DataSetRow> rightFilter = this.buildFilter(nodeContent.get(1), rowMetadata);
        return leftFilter.and(rightFilter);
    }

    private Predicate<DataSetRow> createOrPredicate(JsonNode nodeContent, RowMetadata rowMetadata) {
        this.checkValidMultiPredicate(nodeContent);
        Predicate<DataSetRow> leftFilter = this.buildFilter(nodeContent.get(0), rowMetadata);
        Predicate<DataSetRow> rightFilter = this.buildFilter(nodeContent.get(1), rowMetadata);
        return leftFilter.or(rightFilter);
    }

    private Predicate<DataSetRow> createNotPredicate(JsonNode nodeContent, RowMetadata rowMetadata) {
        if (!nodeContent.isObject()) {
            throw new IllegalArgumentException("Unsupported query, malformed 'not' (expected 1 object child).");
        }
        if (nodeContent.size() == 0) {
            throw new IllegalArgumentException("Unsupported query, malformed 'not' (object child is empty).");
        }
        return this.buildFilter(nodeContent, rowMetadata).negate();
    }

    private Predicate<DataSetRow> createEqualsPredicate(JsonNode node, String columnId, String value) {
        this.checkValidValue(node, value);
        return r -> {
            if (StringUtils.equals((String)r.get(columnId), (String)value)) {
                return true;
            }
            return NumericHelper.isBigDecimal(r.get(columnId)) && NumericHelper.isBigDecimal(value) && NumberUtils.compare((double)this.toBigDecimal(r.get(columnId)), (double)this.toBigDecimal(value)) == 0;
        };
    }

    private Predicate<DataSetRow> createGreaterThanPredicate(JsonNode node, String columnId, String value) {
        this.checkValidValue(node, value);
        return r -> NumericHelper.isBigDecimal(r.get(columnId)) && NumericHelper.isBigDecimal(value) && this.toBigDecimal(r.get(columnId)) > this.toBigDecimal(value);
    }

    private Predicate<DataSetRow> createLowerThanPredicate(JsonNode node, String columnId, String value) {
        this.checkValidValue(node, value);
        return r -> NumericHelper.isBigDecimal(r.get(columnId)) && NumericHelper.isBigDecimal(value) && this.toBigDecimal(r.get(columnId)) < this.toBigDecimal(value);
    }

    private Predicate<DataSetRow> createGreaterOrEqualsPredicate(JsonNode node, String columnId, String value) {
        this.checkValidValue(node, value);
        return r -> NumericHelper.isBigDecimal(r.get(columnId)) && NumericHelper.isBigDecimal(value) && this.toBigDecimal(r.get(columnId)) >= this.toBigDecimal(value);
    }

    private Predicate<DataSetRow> createLowerOrEqualsPredicate(JsonNode node, String columnId, String value) {
        this.checkValidValue(node, value);
        return r -> NumericHelper.isBigDecimal(r.get(columnId)) && NumericHelper.isBigDecimal(value) && this.toBigDecimal(r.get(columnId)) <= this.toBigDecimal(value);
    }

    private Predicate<DataSetRow> createContainsPredicate(JsonNode node, String columnId, String value) {
        this.checkValidValue(node, value);
        return r -> StringUtils.containsIgnoreCase((String)r.get(columnId), (String)value);
    }

    private Predicate<DataSetRow> createMatchesPredicate(JsonNode node, String columnId, String value) {
        this.checkValidValue(node, value);
        return r -> this.matches(r.get(columnId), value);
    }

    private Predicate<DataSetRow> createInvalidPredicate(String columnId) {
        return r -> r.isInvalid(columnId);
    }

    private Predicate<DataSetRow> createValidPredicate(String columnId) {
        return r -> !r.isInvalid(columnId) && !StringUtils.isEmpty((String)r.get(columnId));
    }

    private Predicate<DataSetRow> createEmptyPredicate(String columnId) {
        return r -> StringUtils.isEmpty((String)r.get(columnId));
    }

    private Predicate<DataSetRow> createRangePredicate(String columnId, JsonNode nodeContent, RowMetadata rowMetadata) {
        String start = nodeContent.get("start").asText();
        String end = nodeContent.get("end").asText();
        return r -> {
            String columnType = rowMetadata.getById(columnId).getType();
            Type parsedType = Type.get(columnType);
            if (Type.DATE.isAssignableFrom(parsedType)) {
                return this.createDateRangePredicate(columnId, start, end, rowMetadata).test((DataSetRow)r);
            }
            return this.createNumberRangePredicate(columnId, start, end).test((DataSetRow)r);
        };
    }

    private Predicate<DataSetRow> createDateRangePredicate(String columnId, String start, String end, RowMetadata rowMetadata) {
        try {
            long minTimestamp = Long.parseLong(start);
            long maxTimestamp = Long.parseLong(end);
            LocalDateTime minDate = DateManipulator.fromEpochMillisecondsWithSystemOffset(minTimestamp);
            LocalDateTime maxDate = DateManipulator.fromEpochMillisecondsWithSystemOffset(maxTimestamp);
            return SimpleFilterService.safeDate(r -> {
                ColumnMetadata columnMetadata = rowMetadata.getById(columnId);
                LocalDateTime columnValue = this.getDateParser().parse(r.get(columnId), columnMetadata);
                return minDate.compareTo(columnValue) == 0 || minDate.isBefore(columnValue) && maxDate.isAfter(columnValue);
            });
        }
        catch (Exception e) {
            LOGGER.debug("Unable to create date range predicate.", (Throwable)e);
            throw new IllegalArgumentException("Unsupported query, malformed date 'range' (expected timestamps in min and max properties).");
        }
    }

    private synchronized DateParser getDateParser() {
        if (this.dateParser == null) {
            this.dateParser = new DateParser(Providers.get(AnalyzerService.class));
        }
        return this.dateParser;
    }

    void setDateParser(DateParser dateParser) {
        this.dateParser = dateParser;
    }

    private Predicate<DataSetRow> createNumberRangePredicate(String columnId, String start, String end) {
        try {
            double min = this.toBigDecimal(start);
            double max = this.toBigDecimal(end);
            return r -> {
                String value = r.get(columnId);
                if (NumericHelper.isBigDecimal(value)) {
                    double columnValue = this.toBigDecimal(value);
                    return NumberUtils.compare((double)columnValue, (double)min) == 0 || columnValue > min && columnValue < max;
                }
                return false;
            };
        }
        catch (Exception e) {
            LOGGER.debug("Unable to create number range predicate.", (Throwable)e);
            throw new IllegalArgumentException("Unsupported query, malformed 'range' (expected number min and max properties).");
        }
    }

    private void checkValidValue(JsonNode node, String value) {
        if (value == null) {
            throw new UnsupportedOperationException("Unsupported query, the filter needs a value : " + node.toString());
        }
    }

    private void checkValidMultiPredicate(JsonNode nodeContent) {
        if (nodeContent.size() != 2) {
            throw new IllegalArgumentException("Unsupported query, malformed 'and' (expected 2 children).");
        }
    }

    private boolean matches(String value, String pattern) {
        if (value == null && pattern == null) {
            return true;
        }
        if (value == null) {
            return false;
        }
        if (StringUtils.containsAny((String)pattern, (char[])new char[]{'A', 'a', '9'})) {
            if (value.length() != pattern.length()) {
                return false;
            }
            char[] valueArray = value.toCharArray();
            char[] patternArray = pattern.toCharArray();
            for (int i = 0; i < valueArray.length; ++i) {
                if (!(patternArray[i] == 'A' ? !Character.isUpperCase(valueArray[i]) : (patternArray[i] == 'a' ? !Character.isLowerCase(valueArray[i]) : (patternArray[i] == '9' ? !Character.isDigit(valueArray[i]) : valueArray[i] != patternArray[i])))) continue;
                return false;
            }
        } else {
            DateTimeFormatter formatter = DateTimeFormatter.ofPattern(pattern);
            try {
                formatter.toFormat().parseObject(value);
            }
            catch (ParseException e) {
                return false;
            }
        }
        return true;
    }

    private double toBigDecimal(String value) {
        return BigDecimalParser.toBigDecimal((String)value).doubleValue();
    }
}

