/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.fs.UnresolvedLinkException;
import org.apache.hadoop.hdfs.server.common.Util;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.INodeFileUnderConstruction;
import org.apache.hadoop.util.Daemon;

@InterfaceAudience.Private
public class LeaseManager {
    public static final Log LOG = LogFactory.getLog(LeaseManager.class);
    private final FSNamesystem fsnamesystem;
    private long softLimit = 60000L;
    private long hardLimit = 3600000L;
    private SortedMap<String, Lease> leases = new TreeMap<String, Lease>();
    private SortedSet<Lease> sortedLeases = new TreeSet<Lease>();
    private SortedMap<String, Lease> sortedLeasesByPath = new TreeMap<String, Lease>();
    private Daemon lmthread;
    private volatile boolean shouldRunMonitor;

    LeaseManager(FSNamesystem fsnamesystem) {
        this.fsnamesystem = fsnamesystem;
    }

    Lease getLease(String holder) {
        return (Lease)this.leases.get(holder);
    }

    SortedSet<Lease> getSortedLeases() {
        return this.sortedLeases;
    }

    public Lease getLeaseByPath(String src) {
        return (Lease)this.sortedLeasesByPath.get(src);
    }

    public synchronized int countLease() {
        return this.sortedLeases.size();
    }

    synchronized int countPath() {
        int count = 0;
        for (Lease lease : this.sortedLeases) {
            count += lease.getPaths().size();
        }
        return count;
    }

    synchronized Lease addLease(String holder, String src) {
        Lease lease = this.getLease(holder);
        if (lease == null) {
            lease = new Lease(holder);
            this.leases.put(holder, lease);
            this.sortedLeases.add(lease);
        } else {
            this.renewLease(lease);
        }
        this.sortedLeasesByPath.put(src, lease);
        lease.paths.add(src);
        return lease;
    }

    synchronized void removeLease(Lease lease, String src) {
        this.sortedLeasesByPath.remove(src);
        if (!lease.removePath(src)) {
            LOG.error((Object)(src + " not found in lease.paths (=" + lease.paths + ")"));
        }
        if (!lease.hasPath()) {
            this.leases.remove(lease.holder);
            if (!this.sortedLeases.remove(lease)) {
                LOG.error((Object)(lease + " not found in sortedLeases"));
            }
        }
    }

    synchronized void removeLease(String holder, String src) {
        Lease lease = this.getLease(holder);
        if (lease != null) {
            this.removeLease(lease, src);
        } else {
            LOG.warn((Object)("Removing non-existent lease! holder=" + holder + " src=" + src));
        }
    }

    synchronized Lease reassignLease(Lease lease, String src, String newHolder) {
        assert (newHolder != null) : "new lease holder is null";
        if (lease != null) {
            this.removeLease(lease, src);
        }
        return this.addLease(newHolder, src);
    }

    public synchronized String findPath(INodeFileUnderConstruction pendingFile) throws IOException {
        String src;
        Lease lease = this.getLease(pendingFile.getClientName());
        if (lease != null && (src = lease.findPath(pendingFile)) != null) {
            return src;
        }
        throw new IOException("pendingFile (=" + pendingFile + ") not found." + "(lease=" + lease + ")");
    }

    synchronized void renewLease(String holder) {
        this.renewLease(this.getLease(holder));
    }

    synchronized void renewLease(Lease lease) {
        if (lease != null) {
            this.sortedLeases.remove(lease);
            lease.renew();
            this.sortedLeases.add(lease);
        }
    }

    synchronized void renewAllLeases() {
        for (Lease l : this.leases.values()) {
            this.renewLease(l);
        }
    }

    synchronized void changeLease(String src, String dst, String overwrite, String replaceBy) {
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)(this.getClass().getSimpleName() + ".changelease: " + " src=" + src + ", dest=" + dst + ", overwrite=" + overwrite + ", replaceBy=" + replaceBy));
        }
        int len = overwrite.length();
        for (Map.Entry<String, Lease> entry : LeaseManager.findLeaseWithPrefixPath(src, this.sortedLeasesByPath)) {
            String oldpath = entry.getKey();
            Lease lease = entry.getValue();
            String newpath = replaceBy + oldpath.substring(len);
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("changeLease: replacing " + oldpath + " with " + newpath));
            }
            lease.replacePath(oldpath, newpath);
            this.sortedLeasesByPath.remove(oldpath);
            this.sortedLeasesByPath.put(newpath, lease);
        }
    }

    synchronized void removeLeaseWithPrefixPath(String prefix) {
        for (Map.Entry<String, Lease> entry : LeaseManager.findLeaseWithPrefixPath(prefix, this.sortedLeasesByPath)) {
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)(LeaseManager.class.getSimpleName() + ".removeLeaseWithPrefixPath: entry=" + entry));
            }
            this.removeLease(entry.getValue(), entry.getKey());
        }
    }

    private static List<Map.Entry<String, Lease>> findLeaseWithPrefixPath(String prefix, SortedMap<String, Lease> path2lease) {
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)(LeaseManager.class.getSimpleName() + ".findLease: prefix=" + prefix));
        }
        ArrayList<Map.Entry<String, Lease>> entries = new ArrayList<Map.Entry<String, Lease>>();
        int srclen = prefix.length();
        for (Map.Entry<String, Lease> entry : path2lease.tailMap(prefix).entrySet()) {
            String p = entry.getKey();
            if (!p.startsWith(prefix)) {
                return entries;
            }
            if (p.length() != srclen && p.charAt(srclen) != '/') continue;
            entries.add(entry);
        }
        return entries;
    }

    public void setLeasePeriod(long softLimit, long hardLimit) {
        this.softLimit = softLimit;
        this.hardLimit = hardLimit;
    }

    private synchronized void checkLeases() {
        assert (this.fsnamesystem.hasWriteLock());
        while (this.sortedLeases.size() > 0) {
            Lease oldest = this.sortedLeases.first();
            if (!oldest.expiredHardLimit()) {
                return;
            }
            LOG.info((Object)("Lease " + oldest + " has expired hard limit"));
            ArrayList<String> removing = new ArrayList<String>();
            String[] leasePaths = new String[oldest.getPaths().size()];
            oldest.getPaths().toArray(leasePaths);
            for (String p : leasePaths) {
                try {
                    if (this.fsnamesystem.internalReleaseLease(oldest, p, "HDFS_NameNode")) {
                        LOG.info((Object)("Lease recovery for file " + p + " is complete. File closed."));
                        removing.add(p);
                        continue;
                    }
                    LOG.info((Object)("Started block recovery for file " + p + " lease " + oldest));
                }
                catch (IOException e) {
                    LOG.error((Object)("Cannot release the path " + p + " in the lease " + oldest), (Throwable)e);
                    removing.add(p);
                }
            }
            for (String p : removing) {
                this.removeLease(oldest, p);
            }
        }
    }

    public synchronized String toString() {
        return this.getClass().getSimpleName() + "= {" + "\n leases=" + this.leases + "\n sortedLeases=" + this.sortedLeases + "\n sortedLeasesByPath=" + this.sortedLeasesByPath + "\n}";
    }

    void startMonitor() {
        Preconditions.checkState((this.lmthread == null ? 1 : 0) != 0, (Object)"Lease Monitor already running");
        this.shouldRunMonitor = true;
        this.lmthread = new Daemon((Runnable)new Monitor());
        this.lmthread.start();
    }

    void stopMonitor() {
        if (this.lmthread != null) {
            this.shouldRunMonitor = false;
            try {
                this.lmthread.interrupt();
                this.lmthread.join(3000L);
            }
            catch (InterruptedException ie) {
                LOG.warn((Object)"Encountered exception ", (Throwable)ie);
            }
            this.lmthread = null;
        }
    }

    @VisibleForTesting
    void triggerMonitorCheckNow() {
        Preconditions.checkState((this.lmthread != null ? 1 : 0) != 0, (Object)"Lease monitor is not running");
        this.lmthread.interrupt();
    }

    class Monitor
    implements Runnable {
        final String name = this.getClass().getSimpleName();

        Monitor() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            while (LeaseManager.this.shouldRunMonitor && LeaseManager.this.fsnamesystem.isRunning()) {
                try {
                    LeaseManager.this.fsnamesystem.writeLockInterruptibly();
                    try {
                        if (!LeaseManager.this.fsnamesystem.isInSafeMode()) {
                            LeaseManager.this.checkLeases();
                        }
                    }
                    finally {
                        LeaseManager.this.fsnamesystem.writeUnlock();
                    }
                    Thread.sleep(2000L);
                }
                catch (InterruptedException ie) {
                    if (!LOG.isDebugEnabled()) continue;
                    LOG.debug((Object)(this.name + " is interrupted"), (Throwable)ie);
                }
            }
        }
    }

    class Lease
    implements Comparable<Lease> {
        private final String holder;
        private long lastUpdate;
        private final Collection<String> paths = new TreeSet<String>();

        private Lease(String holder) {
            this.holder = holder;
            this.renew();
        }

        private void renew() {
            this.lastUpdate = Util.now();
        }

        public boolean expiredHardLimit() {
            return Util.now() - this.lastUpdate > LeaseManager.this.hardLimit;
        }

        public boolean expiredSoftLimit() {
            return Util.now() - this.lastUpdate > LeaseManager.this.softLimit;
        }

        private String findPath(INodeFileUnderConstruction pendingFile) {
            try {
                for (String src : this.paths) {
                    if (((LeaseManager)LeaseManager.this).fsnamesystem.dir.getFileINode(src) != pendingFile) continue;
                    return src;
                }
            }
            catch (UnresolvedLinkException e) {
                throw new AssertionError((Object)"Lease files should reside on this FS");
            }
            return null;
        }

        boolean hasPath() {
            return !this.paths.isEmpty();
        }

        boolean removePath(String src) {
            return this.paths.remove(src);
        }

        public String toString() {
            return "[Lease.  Holder: " + this.holder + ", pendingcreates: " + this.paths.size() + "]";
        }

        @Override
        public int compareTo(Lease o) {
            Lease l1 = this;
            Lease l2 = o;
            long lu1 = l1.lastUpdate;
            long lu2 = l2.lastUpdate;
            if (lu1 < lu2) {
                return -1;
            }
            if (lu1 > lu2) {
                return 1;
            }
            return l1.holder.compareTo(l2.holder);
        }

        public boolean equals(Object o) {
            if (!(o instanceof Lease)) {
                return false;
            }
            Lease obj = (Lease)o;
            return this.lastUpdate == obj.lastUpdate && this.holder.equals(obj.holder);
        }

        public int hashCode() {
            return this.holder.hashCode();
        }

        Collection<String> getPaths() {
            return this.paths;
        }

        String getHolder() {
            return this.holder;
        }

        void replacePath(String oldpath, String newpath) {
            this.paths.remove(oldpath);
            this.paths.add(newpath);
        }

        @VisibleForTesting
        long getLastUpdate() {
            return this.lastUpdate;
        }
    }
}

