package org.talend.net;

import java.io.File;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.Date;
import java.util.HashMap;
import java.util.TimeZone;
import java.text.SimpleDateFormat;
import java.text.ParseException;
import java.util.Map;

/**
 * Base wrapper for all classes from within .NET.  Because all of these classes on the .NET side extend System::Object
 * or System.Object (depending on the implementation language), they can all be stored as an Object on the native side.
 * Because of this, on the java side, we have this external representation.  After all, it's either an Object or a
 * primitive.  By using this class, you can access all of the public functionality of .NET classes from within Java.
 *
 * @author Richard Baldwin
 */
public final class Object {

    /**
     * Map containing the classes.  The idea behind this is much like said in the class header:
     * it's one of three things from our point of view.  A primitive (int, long, etc), a String, a Date
     * or a miscellaneous object that we may not be able to convert.  We convert the types we know implicitly
     * by dealing with them as strings during the transfer from java to .NET and vice versa.  On the .NET side
     * these can be converted dynamically through the Convert class.  On the java side it's nearly as simple,
     * we use the constructor that takes a String and create the instance from that.
     */
    private static final Map<String, String> CLASS_MAP = new HashMap<String, String>();
    /**
     * A map containing the loaded assemblies.  The key is the absolute path to them and the value
     * is the internal Assembly representation.
     */
    private static final Map<String, Object> LOADED_ASSEMBLIES = new HashMap<String, Object>();

    /**
     * Helper class to facilitate the transformation from .NET to java objects.  While it currently only
     * supports primitives plus String and Date - it expects that all others are instances of {@link org.talend.net.Object}
     * that you (the caller) should know what to do with them anyways.
     */
    private final class Helper {

        /**
         * The .NET type of the object.  i.e. System.Int32, etc.
         */
        private String type;
        /**
         * the physical value, stored in a string.
         */
        private String value;

        /**
         * Default constructor.
         */
        public Helper() {
        }

        /**
         * parameterized constructor.
         * @param t the .NET type
         * @param v the value.
         */
        public Helper(final String t, final String v) {
            this.type = t;
            this.value = v;
        }

        /**
         * Simple getter for the type value.
         * @return the value of the type instance variable.
         */
        public String getType() {
            return type;
        }

        /**
         * simple setter for the type.
         * @param t the value to apply to the the type instance variable.
         */
        public void setType(final String t) {
            this.type = t;
        }

        /**
         * getter for the value in string form.
         * @return the value that we got from .NET
         */
        public String getValue() {
            return value;
        }

        /**
         * Setter for the value instance variable.
         * @param v the value to set to the value instance variable.
         */
        public void setValue(final String v) {
            this.value = v;
        }

        /**
         * method to remove the colon from within the timezone section of the
         * date string.
         * @param date the string containing a date with a timezone formatted as -07:00
         * @return the String now contains the timezone formatted as -0700
         */
        private String removeTZColon(final String date) {
            //gotta remove that last friggin' colon...
            StringBuffer sb = new StringBuffer(date);
            sb.deleteCharAt(date.lastIndexOf(":"));
            return sb.toString();
        }

        /**
         * Converter method to convert - somewhat implicitly - between the .NET value and
         * java value.  Thus far, only Date has a special case for the conversion otherwise, we invoke
         * the constructor that accepts a string.  Thus, theoretically, any {@link java.lang.Object}
         * that has a constructor that takes 1 string can be used for conversion.
         *
         * @return the converted value
         * @throws NoSuchMethodException if there isn't a constructor that takes one string argument
         * @throws ClassNotFoundException if it can't find the class.  Not likely.
         * @throws InstantiationException if it can't create the destination type via the constructor.
         * @throws IllegalAccessException if the constructor is private, not likely because we force accessibility.
         * @throws InvocationTargetException can't invoke the constructor for some other reason.
         * @throws ParseException invalid date format.
         */
        public java.lang.Object convert() throws NoSuchMethodException, ClassNotFoundException, InstantiationException,
                IllegalAccessException, InvocationTargetException, ParseException {
            if (type == null || value == null) {
                return null;
            }
            
            //try {
            if (type.equals("System.DateTime")) {
                return DOT_NET_DATE_FORMAT.parse(removeTZColon(value));
            } else if (type.equals("System.DBNull")) {//special case for nulls from OleDb.
                return null;
            }
            Constructor c = Class.forName(CLASS_MAP.get(type)).getConstructor(String.class);
            c.setAccessible(true);
            return c.newInstance(value);
        }
    }
    /**
     * Hold a pointer to the internalized representation of the object in question,
     * this variable will hold the pointer.
     */
    private int plongObj = -1;
    /**
     * Date format as we get it from .NET.
     */
    private static final SimpleDateFormat DOT_NET_DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd HH:mm:ssZ");
    /**
     * Date format to send the dates to .NET.
     */
    private static final SimpleDateFormat DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss'Z'");

    /*
     * load the dll.
     */
    static {
        loadLibrary();
        initClassMap();
    }

    /**
     * initialize the static map of convertible classes.
     */
    private static void initClassMap() {
        CLASS_MAP.put("System.Int32", "java.lang.Integer");
        CLASS_MAP.put("System.Int64", "java.lang.Long");
        CLASS_MAP.put("System.Int16", "java.lang.Short");
        CLASS_MAP.put("System.String", "java.lang.String");
        CLASS_MAP.put("System.Double", "java.lang.Double");
        CLASS_MAP.put("System.Char", "java.lang.Character");
        CLASS_MAP.put("System.Byte", "java.lang.Byte");
        CLASS_MAP.put("System.Boolean", "java.lang.Boolean");
        CLASS_MAP.put("System.Single", "java.lang.Float");
        CLASS_MAP.put("System.Decimal", "java.math.BigDecimal");
        CLASS_MAP.put("System.DateTime", "java.util.Date");
        CLASS_MAP.put("System.Int8", "java.lang.Byte");
        CLASS_MAP.put("System.UInt16", "java.lang.Short");
        CLASS_MAP.put("System.UInt32", "java.lang.Integer");
        CLASS_MAP.put("System.UInt64", "java.lang.Long");
    }

    /**
     * Load the janet library.  This uses the system property that contains the architecture of the CPU
     * to determine exactly which one to load.
     */
    private static void loadLibrary() {
        String osName = System.getProperty("os.name");
        String arch = System.getProperty("sun.arch.data.model");
        if (!osName.startsWith("Windows")) {
            throw new UnsupportedOperationException("Operating systems other "
                    + "than Windows are not supported for .NET integration.");
        }
        try {
            System.loadLibrary("janet-win" + arch);
            //System.loadLibrary("Janet");
        } catch (Exception ule) {
            throw new UnsupportedOperationException("Please insure that janet-win" + arch
                    + ".dll is in a Path directory.  Please see http://talendforge.org/wiki/doku.php?id=doc:dotnet for more information", ule);
        }
    }

    /**
     * initialize the helper object on the .NET side.
     * @return the wrapped pointer to the C++ Helper object.
     */
    private native int initHelper();

    /**
     * default constructor.
     */
    private Object() {
        super();
    }

    /**
     * Instantiate a .NET object with the internal reference value.
     * @param pLongObj the reference to the .NET object.  This is a pointer
     *                 on the C++ side of things.
     */
    private Object(final int pLongObj) {
        this.plongObj = pLongObj;
    }

    /**
     * Create an instance of a .NET object.  We'll use a loaded assembly to do so.
     * @param className the name of the class to load and instantiate.
     * @param assembly the representation of the assembly.  {@see Object#loadAssembly}
     * @param params the list of parameters for the constructor.
     * @param classNames the array containing the class names for the parameters.
     * @return the internal representation of the new instance.
     */
    private native int createInstanceLib(final String className, final Object assembly, final java.lang.Object[] params,
            final String[] classNames);

    /**
     * Create an instance of a .NET class.  This can be a Visual C++ (WITH CLR), C# or VB.  The library value has
     * to be the absolute path to the dll.  All dependent dlls must be in the PATH variable or it may not properly
     * be able to load them.
     *
     * @param library the absolute path to the dll containing the class or the internal assembly name for example:<br />
     *                <i>System.Data, Version=2.0.0.0, Culture=neutral, PublicKeyToken=b77a5c561934e089<br />
     *                C:\Windows\System32\some.dll</i>
     * @param className the fully qualified class name.
     * @param params the array of parameters for the constructor.  This can be a mixture between java primitives
     *               like int, long and so forth - or recursively an instance of {@link org.talend.net.Object}
     * @return the representation of the new class in the form of an {@link org.talend.net.Object}
     */
    public static Object createInstance(final String library, final String className, final java.lang.Object[] params) {
        Object tmp = new Object();
        Object theAssembly = loadAssem(library);
        
        String[] classNames = new String[params.length];
        for (int i = 0; i < params.length; i++) {
            classNames[i] = params[i].getClass().getName();
        }
        
        java.lang.Object[] convParams = new java.lang.Object[params.length];
        for (int i = 0; i < params.length; i++) {
            classNames[i] = params[i].getClass().getName();
            if (params[i] == null) {
                convParams[i] = "";
            } else if (!(params[i] instanceof org.talend.net.Object) && !(params[i] instanceof java.util.Date)) {
                convParams[i] = params[i].toString();
            } else if (params[i] instanceof Date) {
                DATE_FORMAT.setTimeZone(TimeZone.getTimeZone("GMT"));
                convParams[i] = DATE_FORMAT.format((Date) params[i]);
            } else {
                convParams[i] = params[i];
            }
        }
        
        return new Object(tmp.createInstanceLib(className, theAssembly,
        		convParams, classNames));
    }

    /**
     * Create an instance of a .NET class.  This can be a Visual C++ (WITH CLR), C# or VB.  The library value has
     * to be the absolute path to the dll.  All dependent dlls must be in the PATH variable or it may not properly
     * be able to load them.
     *
     * @param library the absolute path to the dll containing the class or the internal assembly name for example:<br />
     *                <i>System.Data, Version=2.0.0.0, Culture=neutral, PublicKeyToken=b77a5c561934e089<br />
     *                C:\Windows\System32\some.dll</i>
     * @param className the fully qualified class name.
     * @return the representation of the new class in the form of an {@link org.talend.net.Object}
     */
    public static Object createInstance(final String library, final String className) {
        Object tmp = new Object();
        Object theAssembly = loadAssem(library);
        return new Object(tmp.createInstanceLib(className, theAssembly,
                new java.lang.Object[]{}, new String[]{}));
    }

    /**
     * Call a .NET method that returns a String.  Needed for the .NET Helper class.
     * @param methodName the name of the method to call.
     * @return the string that the .NET method returns.
     */
    private native String invokeForString(final String methodName);

    /**
     * Invoke a .NET method.  This is a more general call that accepts an Object array and will
     * return a Generic {@link org.talend.net.Object} instance.
     * @param methodName the name of the method to invoke.
     * @param params the object array containing the parameters to the method.
     * @return the results of the call.
     */
    public Object invoke(final String methodName, final java.lang.Object[] params) {
        String[] classNames = new String[params.length];
        java.lang.Object[] convParams = new java.lang.Object[params.length];
        for (int i = 0; i < params.length; i++) {
            classNames[i] = params[i].getClass().getName();
            if (params[i] == null) {
                convParams[i] = "";
            } else if (!(params[i] instanceof org.talend.net.Object) && !(params[i] instanceof java.util.Date)) {
                convParams[i] = params[i].toString();
            } else if (params[i] instanceof Date) {
                convParams[i] = DATE_FORMAT.format((Date) params[i]);
            } else {
                convParams[i] = params[i];
            }
        }
        return new Object(invokeNative(methodName, convParams, classNames));
    }

    /**
     * Invoke a .NET method.  This is a more general call that accepts an Object array and will
     * return a Generic {@link org.talend.net.Object} instance.
     * @param methodName the name of the method to invoke.
     * @return the results of the call.
     */
    public Object invoke(final String methodName) {
        return new Object(invokeNative(methodName, new java.lang.Object[]{}, new String[]{}));
    }

    /**
     * Invoke a static method on the .NET side.  This is different from invoking an instance object because
     * during the invocation, we don't need or will we have an actual instance to invoke the method upon.  This
     * works essentially the same way as the regular {@link Object#invoke(java.lang.String, java.lang.Object[])}
     * except that the internal type information is resolved from the passed assembly instead of the instance
     * of the object itself.
     *
     * @param assembly the name of the assembly that contains the class that we want to invoke the static method
     *                 upon.
     * @param className The fully qualified class name for the type.  For example: TestClassLibrary.TestClass
     * @param methodName the name of the static method we're going to invoke.
     * @param params the list of parameters for the method.
     * @return an instance of {@link Object} that results from invoking the method.  If this is a void method,
     *         you can ignore the return value.
     */
    public static Object invokeStatic(final String assembly, final String className, final String methodName,
            final java.lang.Object[] params) {
        String[] classNames = new String[params.length];
        java.lang.Object[] convParams = new java.lang.Object[params.length];
        for (int i = 0; i < params.length; i++) {
            classNames[i] = params[i].getClass().getName();
            if (params[i] == null) {
                convParams[i] = "";
            } else if (!(params[i] instanceof org.talend.net.Object) && !(params[i] instanceof java.util.Date)) {
                convParams[i] = params[i].toString();
            } else if (params[i] instanceof Date) {
                convParams[i] = DATE_FORMAT.format((Date) params[i]);
            } else {
                convParams[i] = params[i];
            }
        }
        Object theAssembly = loadAssem(assembly);
        return new Object(invokeStaticNative(theAssembly, className, methodName, convParams, classNames));
    }

    /**
     * invoke a static method on the .NET side, this is a wrapper for
     * {@link Object#invokeStatic(java.lang.String, java.lang.String, java.lang.String, java.lang.Object[])} that
     * doesn't expect a parameter list.  If the method you wish to invoke doesn't take parameters, this method is the
     * one that you should utilize.
     * @param assembly the name or path of the assembly containing the type that we want to invoke the static method on.
     * @param className the fully qualified class name of the desired type.
     * @param methodName the name of the method that we want to invoke.
     * @return an instance of {@link Object} that results from invoking the method.  If the return type is void,
     *         you can disregard the results of the call.
     */
    public static Object invokeStatic(final String assembly, final String className, final String methodName) {
        return invokeStatic(assembly, className, methodName, new java.lang.Object[]{});
    }

    /**
     * invoke a method on the .NET side that returns a .NET object other than the types we know how to convert.
     * This works very similar to how {@link Object#invokeNative(java.lang.String, java.lang.Object[], java.lang.String[])}
     * except this one utilizes static methods on the .NET side.
     * @param assembly the Assembly containing the target type.
     * @param className the fully qualified class name of the target type.
     * @param methodName the name of the method to invoke.
     * @param params the parameters for the target method.
     * @param classNames the java type names for the parameters.
     * @return the internal wrapper for the .NET object.
     */
    private static native int invokeStaticNative(final Object assembly, final String className, final String methodName,
            final java.lang.Object[] params, final String[] classNames);

    /**
     * Invoke a method with a series of objects that serve as the parameters.
     * @param methodName the name of the method to invoke.
     * @param params the Object array containing the parameters.
     * @param classNames the classes for each of the parameters.
     * @return the embedded instance that contains the results of the method call.
     */
    private native int invokeNative(final String methodName, final java.lang.Object[] params,
            final String[] classNames);

    /**
     * Convert a java String into a .NET System::String.
     * @param value the string to convert to the .NET equivalent.
     * @return the .NET System::String embedded in an int.
     */
    private native int nativeString(final String value);

    /**
     * Get an {@link org.talend.net.Object} that encapsulates a native version of a String.
     *
     * @param value the String to put into a native (.NET) string.
     * @return the {@link org.talend.net.Object} containing the value.
     */
    public Object getNativeString(final String value) {
        return new Object(nativeString(value));
    }

    /**
     * invoke a method on the native side that will return a value that can easily and implicitly
     * be converted to a String.  It's stored internally in a C++ Helper that will be used
     * to pull the value out and convert it to the destination type.
     *
     * @param methodName the name of the method to invoke.
     * @param params the array of parameters to pass to the method.
     * @param classNames the names of the classes of the types in the parameter list.
     * @param helper the C++ helper class.
     */
    private native void invokeGenericNative(final String methodName, final java.lang.Object[] params,
            final String[] classNames, final Object helper);

    /**
     * Invoke a method on the .NET side that returns one of the following:
     * {@link java.lang.String}, {@link java.lang.Integer}, {@link java.lang.Long}, {@link java.lang.Float},
     * {@link java.lang.Boolean}, {@link java.lang.Byte}, {@link java.lang.Character}, {@link java.lang.Double},
     * {@link java.util.Date}, {@link java.lang.Short}, and {@link java.math.BigDecimal}
     *
     * All of the parameters are converted to a String prior to being sent to .NET.  They will be converted to the
     * target type from within there using the Convert class.  Unless it's an instance of {@link org.talend.net.Object}
     * in which case it will be passed as is.<br />
     * See also: {@link org.talend.net.Object.Helper}
     *
     * @param methodName the name of the method.
     * @return the re-converted value.
     * @throws NoSuchMethodException if there isn't a String constructor.
     * @throws ClassNotFoundException if it can't find the target class - not likely as they're all implicit to the JVM.
     * @throws InstantiationException if it can't instantiate.
     * @throws IllegalAccessException not likely, but if we didn't set the constructor to explicitly accessible.
     * @throws InvocationTargetException can't invoke the constructor.
     * @throws ParseException a date is returned in an invalid format.
     */
    public java.lang.Object invokeGeneric(final String methodName)
            throws NoSuchMethodException, ClassNotFoundException, InstantiationException,
            IllegalAccessException, InvocationTargetException, ParseException {
        return invokeGeneric(methodName, new java.lang.Object[]{});
    }

    /**
     * Invoke a method on the .NET side that returns one of the following:
     * {@link java.lang.String}, {@link java.lang.Integer}, {@link java.lang.Long}, {@link java.lang.Float},
     * {@link java.lang.Boolean}, {@link java.lang.Byte}, {@link java.lang.Character}, {@link java.lang.Double},
     * {@link java.util.Date}, {@link java.lang.Short}, and {@link java.math.BigDecimal}
     *
     * All of the parameters are converted to a String prior to being sent to .NET.  They will be converted to the
     * target type from within there using the Convert class.  Unless it's an instance of {@link org.talend.net.Object}
     * in which case it will be passed as is.<br />
     * See also: {@link org.talend.net.Object.Helper}
     *
     * @param methodName the name of the method.
     * @param params the list of parameters.
     * @return the re-converted value.
     * @throws NoSuchMethodException if there isn't a String constructor.
     * @throws ClassNotFoundException if it can't find the target class - not likely as they're all implicit to the JVM.
     * @throws InstantiationException if it can't instantiate.
     * @throws IllegalAccessException not likely, but if we didn't set the constructor to explicitly accessible.
     * @throws InvocationTargetException can't invoke the constructor.
     * @throws ParseException a date is returned in an invalid format.
     */
    public java.lang.Object invokeGeneric(final String methodName, final java.lang.Object[] params)
            throws NoSuchMethodException, ClassNotFoundException, InstantiationException,
            IllegalAccessException, InvocationTargetException, ParseException {
        String[] classNames = new String[params.length];
        java.lang.Object[] convParams = new java.lang.Object[params.length];
        for (int i = 0; i < params.length; i++) {
            classNames[i] = params[i].getClass().getName();
            if (params[i] == null) {
                convParams[i] = "";
            } else if (!(params[i] instanceof org.talend.net.Object) && !(params[i] instanceof java.util.Date)) {
                convParams[i] = params[i].toString();
            } else if (params[i] instanceof Date) {
                DATE_FORMAT.setTimeZone(TimeZone.getTimeZone("GMT"));
                convParams[i] = DATE_FORMAT.format((Date) params[i]);
            } else {
                convParams[i] = params[i];
            }
        }
        Object nHelper = new Object(initHelper());
        invokeGenericNative(methodName, convParams, classNames, nHelper);
        Helper tmp = new Helper();
        tmp.setType(nHelper.invokeForString("getType"));
        tmp.setValue(nHelper.invokeForString("getValue"));
        return tmp.convert();
    }

    /**
     * Simply return a new instance of a {@link Helper}.
     * @return a new instance of {@link Helper}
     */
    private Helper getNewHelper() {
        return new Helper();
    }

    /**
     * static wrapper to get an instance of {@link Helper}.
     * @return an instance of {@link Helper}
     */
    private static Helper getHelper() {
        Object t = new Object();
        return t.getNewHelper();
    }

    /**
     * Invoke a static method on the .NET side that returns a type that we can implicitly handle.  This is
     * similar to {@link Object#invokeGeneric(java.lang.String, java.lang.Object[])} in the types we can convert.
     *
     *
     * @param assembly the name or path of the assembly containing the desired type.
     * @param className the fully qaulified class name for the desired type.
     * @param methodName the name of the method you wish to invoke.
     * @param params an array containing all of the parameters to the desired method.
     * @return an implicitly converted value from the .NET side.
     *
     * @throws NoSuchMethodException shouldn't happen, but in the off chance we can't find the target constructor
     *                               on the java side.
     * @throws ClassNotFoundException shouldn't happen because the converter - {@link Helper} - at this time only supports
     *                                classes that are built into the JVM.
     * @throws InstantiationException shouldn't happen if it can't create an instance of the target java type.
     * @throws IllegalAccessException shouldn't happen because we force accessible on the target type.
     * @throws InvocationTargetException shouldn't happen again because we force accessible on target types
     *                                   that have to exist in the JVM.
     * @throws ParseException shouldn't happen but could if something goes wrong with date formatting on the .NET side.
     */
    public static java.lang.Object invokeStaticGeneric(final String assembly, final String className,
            final String methodName, final java.lang.Object[] params)
            throws NoSuchMethodException, ClassNotFoundException, InstantiationException,
            IllegalAccessException, InvocationTargetException, ParseException {
        String[] classNames = new String[params.length];
        java.lang.Object[] convParams = new java.lang.Object[params.length];
        for (int i = 0; i < params.length; i++) {
            classNames[i] = params[i].getClass().getName();
            if (params[i] == null) {
                convParams[i] = "";
            } else if (!(params[i] instanceof org.talend.net.Object) && !(params[i] instanceof java.util.Date)) {
                convParams[i] = params[i].toString();
            } else if (params[i] instanceof Date) {
                DATE_FORMAT.setTimeZone(TimeZone.getTimeZone("GMT"));
                convParams[i] = DATE_FORMAT.format((Date) params[i]);
            } else {
                convParams[i] = params[i];
            }
        }
        Object tmp2 = new Object();
        Object nHelper = new Object(tmp2.initHelper());
        invokeStaticGenericNative(loadAssem(assembly), className, methodName, convParams, classNames, nHelper);
        Helper tmp = getHelper();
        tmp.setType(nHelper.invokeForString("getType"));
        tmp.setValue(nHelper.invokeForString("getValue"));
        return tmp.convert();
    }

    /**
     * wrapper for
     * {@link Object#invokeStaticGeneric(java.lang.String, java.lang.String, java.lang.String, java.lang.Object[])}
     * that doesn't expect a parameter list for the desired method.
     * @param assembly the name of the assembly containing the desired type.
     * @param className the fully qualified class name for the desired type.
     * @param methodName the name of the method.
     * @return the implicitly converted type from the .NET side.
     *
     * @throws NoSuchMethodException shouldn't happen, but in the off chance we can't find the target constructor
     *                               on the java side.
     * @throws ClassNotFoundException shouldn't happen because the converter - {@link Helper} - at this time only supports
     *                                classes that are built into the JVM.
     * @throws InstantiationException shouldn't happen if it can't create an instance of the target java type.
     * @throws IllegalAccessException shouldn't happen because we force accessible on the target type.
     * @throws InvocationTargetException shouldn't happen again because we force accessible on target types
     *                                   that have to exist in the JVM.
     * @throws ParseException shouldn't happen but could if something goes wrong with date formatting on the .NET side.
     */
    public static java.lang.Object invokeStaticGeneric(final String assembly, final String className,
            final String methodName)
            throws NoSuchMethodException, ClassNotFoundException, InstantiationException,
            IllegalAccessException, InvocationTargetException, ParseException {
        return invokeStaticGeneric(assembly, className, methodName, new java.lang.Object[]{});
    }

    /**
     * Native entry point to invoke a static method on the .NET side that returns a type that we know how to implicitly
     * convert to the java equivalent.
     *
     * @param assembly the Assembly that contains the target class.
     * @param className the fully qualified class name of the target type.
     * @param methodName the name of the method.
     * @param params the array of parameters that will be converted and passed to the target method.
     * @param classNames the java types of the parameters.
     * @param helper the helper to help with the implicit conversion on the .NET side.
     */
    private static native void invokeStaticGenericNative(final Object assembly, final String className,
            final String methodName, final java.lang.Object[] params, final String[] classNames, final Object helper);

    /**
     * load a .NET assembly.
     * @param assemblyPath the absolute path the the dll containing the assembly.
     * @return the internal wrapped pointer to the Assembly.
     */
    private static native int loadAssmbly(final String assemblyPath);

    /**
     * Loads a .NET assembly into the system.
     * @param assembly the absolute path to the dll containing the assembly.
     * @return the {@link org.talend.net.Object} containing the Assembly.
     */
    public static Object loadAssembly(final String assembly) {
        if (!LOADED_ASSEMBLIES.containsKey(assembly)) {
            Object rc = new Object(loadAssmbly(assembly));
            LOADED_ASSEMBLIES.put(assembly, rc);
            return rc;
        } else {
            return LOADED_ASSEMBLIES.get(assembly);
        }
    }

    /**
     * load a .NET assembly.
     * @param assemblyName the full name of the assembly. For example: <br/>
     *                     <i>System.Data, Version=2.0.0.0, Culture=neutral, PublicKeyToken=b77a5c561934e089</i>
     * @return the internal wrapped pointer to the Assembly.
     */
    private static native int loadAssmblyByName(final String assemblyName);

    /**
     * Loads a .NET assembly into the system.
     * @param assemblyName the full name of the assembly.
     * @return the {@link org.talend.net.Object} containing the Assembly.
     */
    public static Object loadAssemblyByName(final String assemblyName) {
        if (!LOADED_ASSEMBLIES.containsKey(assemblyName)) {
            Object rc = new Object(loadAssmblyByName(assemblyName));
            LOADED_ASSEMBLIES.put(assemblyName, rc);
            return rc;
        } else {
            return LOADED_ASSEMBLIES.get(assemblyName);
        }
    }

    /**
     * Wrapper to intelligently load an assembly, this method will determine (or try to...) if the target
     * is a dll or assembly name.  It will use this to then try to load the assembly accordingly.
     * @param library the name OR absolute path to the assembly or dll that contains the target types.
     * @return the instance of the assembly that we're trying to load.
     */
    public static Object loadAssem(final String library) {
        File dllFile = new File(library);
        Object theAssembly;
        if (dllFile.exists()) {
            theAssembly = loadAssembly(library);
        } else {
            theAssembly = loadAssemblyByName(library);
        }
        return theAssembly;
    }

    /*
     * START OF PROPERTY ACCESS FUNCTIONALITY May 5, 2011
     */
    /**
     * pull the value of the property back from within an instance of a native object.
     * 
     * @param propertyName the name of the property to pull back.
     * @param helper the conversion helper class.
     */
    private native void accessGenericNativeProperty(final String propertyName, final Object helper);

    /**
     * Obtain the value of a native property for an Object to manipulate it, this only works
     * for properties that contain implicitly convertible fields.
     * @param propertyName the name of the property to pull the value of back.
     * @return the genericized version of the property's value.
     * @throws NoSuchMethodException shouldn't happen, but in the off chance we can't find the target constructor
     *                               on the java side.
     * @throws ClassNotFoundException shouldn't happen because the converter - {@link Helper} - at this time only supports
     *                                classes that are built into the JVM.
     * @throws InstantiationException shouldn't happen if it can't create an instance of the target java type.
     * @throws IllegalAccessException shouldn't happen because we force accessible on the target type.
     * @throws InvocationTargetException shouldn't happen again because we force accessible on target types
     *                                   that have to exist in the JVM.
     * @throws ParseException shouldn't happen but could if something goes wrong with date formatting on the .NET side.
     */
    public java.lang.Object accessGenericProperty(final String propertyName)
            throws NoSuchMethodException, ClassNotFoundException, InstantiationException,
            IllegalAccessException, InvocationTargetException, ParseException {
        Object nHelper = new Object(initHelper());
        accessGenericNativeProperty(propertyName, nHelper);
        Helper tmp = getHelper();
        tmp.setType(nHelper.invokeForString("getType"));
        tmp.setValue(nHelper.invokeForString("getValue"));
        return tmp.convert();
    }

    /**
     * retrieve the value of a native property that cannot be implicitly converted
     * from the .NET type to the java type and/or vice versa.
     * @param propertyName the name of the property to retrieve the value of.
     * @param targetType the .NET type that we'll be returning.
     * @return the internal pointer to the instance that the property returns.
     */
    private native int accessNativeProperty(final String propertyName, final String targetType);

    /**
     * Retrieve the value of a native field that contains a non-convertible Object.
     * See {@link Helper} for the convertible types.
     * @param propertyName the name of the field to pull back.
     * @param targetType the .NET type that the value will be.  This is extremely important in the
     *        event that there are two properties with the same name.  Can be left null if there's
     *        no chance of collision.
     * @return the value of the field wrapped in a java object.
     */
    public Object accessProperty(final String propertyName, final String targetType) {
        return new Object(accessNativeProperty(propertyName,(targetType == null ? "" : targetType)));
    }

    /**
     * Mutate a native property, which is to alter its value by providing another value.
     * Only for instances where the property is a non-implicitly convertible type.
     * @param propertyName the property that we're altering.
     * @param value the value that we're setting the property to.
     * @param type the {@link Class} of the value we're passing.
     */
    private native void mutateNativeProperty(final String propertyName, final java.lang.Object value, String type);

    /**
     * Alter the value of a property in a native class.  Properties are typically used to
     * wrap private fields to provide get/set functionality for the field without
     * exposing the fields directly.  This is more or less the equivalent of bean
     * get/set methods in java.  This is only for non-convertible values.
     * @param propertyName the name of the property to alter the value of.
     * @param value the value that you want to set the property to.
     */
    public void mutateProperty(final String propertyName, final java.lang.Object value) {
        String className = "";
        java.lang.Object arg = value;
        if (value == null) {
            arg = "";
        } else {
            className = value.getClass().getName();
            if (!(value instanceof Object) && !(value instanceof Date)) {
                arg = value.toString();
            } else if (value instanceof Date) {
                DATE_FORMAT.setTimeZone(TimeZone.getTimeZone("GMT"));
                arg = DATE_FORMAT.format((Date) value);
            }            
        }
        mutateNativeProperty(propertyName, arg, className);
    }
}
