/*
 * Decompiled with CFR 0.152.
 */
package parquet.hadoop;

import java.lang.management.ManagementFactory;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import parquet.Log;
import parquet.ParquetRuntimeException;
import parquet.Preconditions;
import parquet.hadoop.InternalParquetRecordWriter;

public class MemoryManager {
    private static final Log LOG = Log.getLog(MemoryManager.class);
    static final float DEFAULT_MEMORY_POOL_RATIO = 0.95f;
    static final long DEFAULT_MIN_MEMORY_ALLOCATION = 0x100000L;
    private final float memoryPoolRatio;
    private final long totalMemoryPool;
    private final long minMemoryAllocation;
    private final Map<InternalParquetRecordWriter, Long> writerList = new HashMap<InternalParquetRecordWriter, Long>();
    private final Map<String, Runnable> callBacks = new HashMap<String, Runnable>();
    private double scale = 1.0;

    public MemoryManager(float ratio, long minAllocation) {
        this.checkRatio(ratio);
        this.memoryPoolRatio = ratio;
        this.minMemoryAllocation = minAllocation;
        this.totalMemoryPool = Math.round((double)ManagementFactory.getMemoryMXBean().getHeapMemoryUsage().getMax() * (double)ratio);
        LOG.debug(String.format("Allocated total memory pool is: %,d", this.totalMemoryPool));
    }

    private void checkRatio(float ratio) {
        if (ratio <= 0.0f || ratio > 1.0f) {
            throw new IllegalArgumentException("The configured memory pool ratio " + ratio + " is " + "not between 0 and 1.");
        }
    }

    synchronized void addWriter(InternalParquetRecordWriter writer, Long allocation) {
        Long oldValue = this.writerList.get(writer);
        if (oldValue != null) {
            throw new IllegalArgumentException("[BUG] The Parquet Memory Manager should not add an instance of InternalParquetRecordWriter more than once. The Manager already contains the writer: " + writer);
        }
        this.writerList.put(writer, allocation);
        this.updateAllocation();
    }

    synchronized void removeWriter(InternalParquetRecordWriter writer) {
        if (this.writerList.containsKey(writer)) {
            this.writerList.remove(writer);
        }
        if (!this.writerList.isEmpty()) {
            this.updateAllocation();
        }
    }

    private void updateAllocation() {
        long totalAllocations = 0L;
        for (Long allocation : this.writerList.values()) {
            totalAllocations += allocation.longValue();
        }
        if (totalAllocations <= this.totalMemoryPool) {
            this.scale = 1.0;
        } else {
            this.scale = (double)this.totalMemoryPool / (double)totalAllocations;
            LOG.warn(String.format("Total allocation exceeds %.2f%% (%,d bytes) of heap memory\nScaling row group sizes to %.2f%% for %d writers", Float.valueOf(100.0f * this.memoryPoolRatio), this.totalMemoryPool, 100.0 * this.scale, this.writerList.size()));
            for (Runnable callBack : this.callBacks.values()) {
                callBack.run();
            }
        }
        int maxColCount = 0;
        for (InternalParquetRecordWriter internalParquetRecordWriter : this.writerList.keySet()) {
            maxColCount = Math.max(internalParquetRecordWriter.getSchema().getColumns().size(), maxColCount);
        }
        for (Map.Entry entry : this.writerList.entrySet()) {
            long newSize = (long)Math.floor((double)((Long)entry.getValue()).longValue() * this.scale);
            if (this.scale < 1.0 && this.minMemoryAllocation > 0L && newSize < this.minMemoryAllocation) {
                throw new ParquetRuntimeException(String.format("New Memory allocation %d bytes is smaller than the minimum allocation size of %d bytes.", newSize, this.minMemoryAllocation)){};
            }
            ((InternalParquetRecordWriter)entry.getKey()).setRowGroupSizeThreshold(newSize);
            LOG.debug(String.format("Adjust block size from %,d to %,d for writer: %s", entry.getValue(), newSize, entry.getKey()));
        }
    }

    long getTotalMemoryPool() {
        return this.totalMemoryPool;
    }

    Map<InternalParquetRecordWriter, Long> getWriterList() {
        return this.writerList;
    }

    float getMemoryPoolRatio() {
        return this.memoryPoolRatio;
    }

    public void registerScaleCallBack(String callBackName, Runnable callBack) {
        Preconditions.checkNotNull(callBackName, "callBackName");
        Preconditions.checkNotNull(callBack, "callBack");
        if (this.callBacks.containsKey(callBackName)) {
            throw new IllegalArgumentException("The callBackName " + callBackName + " is duplicated and has been registered already.");
        }
        this.callBacks.put(callBackName, callBack);
    }

    Map<String, Runnable> getScaleCallBacks() {
        return Collections.unmodifiableMap(this.callBacks);
    }

    double getScale() {
        return this.scale;
    }
}

