/*
 * Copyright (c) 2022, 2025, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package io.github.dmlloyd.classfile.attribute;

import io.github.dmlloyd.classfile.*;
import io.github.dmlloyd.classfile.AttributeMapper.AttributeStability;
import io.github.dmlloyd.classfile.extras.reflect.AccessFlag;

import io.github.dmlloyd.classfile.impl.BoundAttribute;
import io.github.dmlloyd.classfile.impl.UnboundAttribute;

/**
 * Models the {@link Attributes#synthetic() Synthetic} attribute (JVMS {@jvms
 * 4.7.8}), which marks a class member as implementation-specific artifacts.
 * <p>
 * This attribute can appear on classes, methods, and fields, and permits
 * {@linkplain AttributeMapper#allowMultiple multiple instances} in a structure.
 * It has {@linkplain AttributeStability#STATELESS no data dependency}.
 * <p>
 * This attribute was introduced in the Java SE Platform version 1.1, major
 * version {@value ClassFile#JAVA_1_VERSION}.
 *
 * @apiNote
 * This attribute has been largely superseded by the {@link AccessFlag#SYNTHETIC
 * ACC_SYNTHETIC} modifier in newer {@code class} files.  See {@link
 * java.compiler/javax.lang.model.util.Elements.Origin} for a more in-depth
 * description about the differences between synthetic and implicit (mandated)
 * constructs generated by compilers.
 *
 * @see Attributes#synthetic()
 * @see AccessFlag#SYNTHETIC
 * @jvms 4.7.8 The {@code Synthetic} Attribute
 * @since 24
 */
@SuppressWarnings("doclint:reference")
public sealed interface SyntheticAttribute
        extends Attribute<SyntheticAttribute>,
                ClassElement, MethodElement, FieldElement
        permits BoundAttribute.BoundSyntheticAttribute, UnboundAttribute.UnboundSyntheticAttribute {

    /**
     * {@return a {@code Synthetic} attribute}
     */
    static SyntheticAttribute of() {
        return new UnboundAttribute.UnboundSyntheticAttribute();
    }
}
