/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.s3a.s3guard;

import com.amazonaws.services.s3.model.MultipartUpload;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.Closeable;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.PrintStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.AccessDeniedException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Scanner;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DurationFormatUtils;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FilterFileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.s3a.Constants;
import org.apache.hadoop.fs.s3a.Invoker;
import org.apache.hadoop.fs.s3a.MultipartUtils;
import org.apache.hadoop.fs.s3a.S3AFileStatus;
import org.apache.hadoop.fs.s3a.S3AFileSystem;
import org.apache.hadoop.fs.s3a.S3AUtils;
import org.apache.hadoop.fs.s3a.WriteOperationHelper;
import org.apache.hadoop.fs.s3a.audit.AuditSpanS3A;
import org.apache.hadoop.fs.s3a.auth.delegation.S3ADelegationTokens;
import org.apache.hadoop.fs.s3a.commit.CommitConstants;
import org.apache.hadoop.fs.s3a.impl.DirectoryPolicy;
import org.apache.hadoop.fs.s3a.impl.DirectoryPolicyImpl;
import org.apache.hadoop.fs.s3a.s3guard.AuthoritativeAuditOperation;
import org.apache.hadoop.fs.s3a.s3guard.DirListingMetadata;
import org.apache.hadoop.fs.s3a.s3guard.DynamoDBMetadataStore;
import org.apache.hadoop.fs.s3a.s3guard.ITtlTimeProvider;
import org.apache.hadoop.fs.s3a.s3guard.ImportOperation;
import org.apache.hadoop.fs.s3a.s3guard.LocalMetadataStore;
import org.apache.hadoop.fs.s3a.s3guard.MetadataStore;
import org.apache.hadoop.fs.s3a.s3guard.NullMetadataStore;
import org.apache.hadoop.fs.s3a.s3guard.PathMetadata;
import org.apache.hadoop.fs.s3a.s3guard.PathMetadataDynamoDBTranslation;
import org.apache.hadoop.fs.s3a.s3guard.S3Guard;
import org.apache.hadoop.fs.s3a.s3guard.S3GuardFsck;
import org.apache.hadoop.fs.s3a.s3guard.TableDeleteTimeoutException;
import org.apache.hadoop.fs.s3a.select.SelectTool;
import org.apache.hadoop.fs.s3a.tools.MarkerTool;
import org.apache.hadoop.fs.shell.CommandFormat;
import org.apache.hadoop.fs.statistics.IOStatistics;
import org.apache.hadoop.fs.statistics.IOStatisticsLogging;
import org.apache.hadoop.fs.statistics.IOStatisticsSupport;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.util.ExitCodeProvider;
import org.apache.hadoop.util.ExitUtil;
import org.apache.hadoop.util.GenericOptionsParser;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.LimitedPrivate(value={"management tools"})
@InterfaceStability.Evolving
public abstract class S3GuardTool
extends Configured
implements Tool,
Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(S3GuardTool.class);
    private static final String NAME = "s3guard";
    private static final String COMMON_USAGE = "When possible and not overridden by more specific options, metadata\nrepository information will be inferred from the S3A URL (if provided)\n\nGeneric options supported are:\n  -conf <config file> - specify an application configuration file\n  -D <property=value> - define a value for a given property\n";
    private static final String USAGE = "s3guard [command] [OPTIONS] [s3a://BUCKET]\n\nCommands: \n\tinit - initialize metadata repository\n\tdestroy - destroy the Metadata Store including its contents(all data in S3 is preserved)\n\tauthoritative - Audits a DynamoDB S3Guard repository for all the entries being 'authoritative'\n\tbucket-info - provide/check S3Guard information about a specific bucket\n\tdiff - report on delta between S3 and repository\n\tfsck - Compares S3 with MetadataStore, and returns a failure status if any rules or invariants are violated. Only works with DynamoDB metadata stores.\n\timport - import metadata from existing S3 data\n\tmarkers - View and manipulate S3 directory markers\n\tprune - truncate older metadata from repository (all data in S3 is preserved)\n\tset-capacity - Alter metadata store IO capacity\n\tselect - make an S3 Select call\n\tuploads - list or abort pending multipart uploads\n";
    private static final String DATA_IN_S3_IS_PRESERVED = "(all data in S3 is preserved)";
    static final int SUCCESS = 0;
    static final int INVALID_ARGUMENT = 40;
    static final int E_USAGE = 42;
    static final int ERROR = -1;
    static final int E_BAD_STATE = 46;
    static final int E_NOT_FOUND = 44;
    @VisibleForTesting
    public static final String WRONG_FILESYSTEM = "Wrong filesystem for ";
    private FileSystem baseFS;
    private S3AFileSystem filesystem;
    private MetadataStore store;
    private final CommandFormat commandFormat;
    public static final String META_FLAG = "meta";
    public static final String DAYS_FLAG = "days";
    public static final String HOURS_FLAG = "hours";
    public static final String MINUTES_FLAG = "minutes";
    public static final String SECONDS_FLAG = "seconds";
    public static final String AGE_OPTIONS_USAGE = "[-days <days>] [-hours <hours>] [-minutes <minutes>] [-seconds <seconds>]";
    public static final String REGION_FLAG = "region";
    public static final String READ_FLAG = "read";
    public static final String WRITE_FLAG = "write";
    public static final String SSE_FLAG = "sse";
    public static final String CMK_FLAG = "cmk";
    public static final String TAG_FLAG = "tag";
    public static final String VERBOSE = "verbose";
    private static S3GuardTool command;

    public abstract String getUsage();

    protected S3GuardTool(Configuration conf, String ... opts) {
        super(conf);
        conf.set("fs.s3a.s3guard.disabled.warn.level", S3Guard.DisabledWarnLevel.SILENT.toString());
        this.commandFormat = new CommandFormat(0, Integer.MAX_VALUE, opts);
        this.commandFormat.addOptionWithValue(META_FLAG);
        this.commandFormat.addOptionWithValue(REGION_FLAG);
    }

    public abstract String getName();

    @Override
    public void close() throws IOException {
        IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{this.baseFS, this.store});
        this.baseFS = null;
        this.filesystem = null;
        this.store = null;
    }

    protected void parseDynamoDBRegion(List<String> paths) throws IOException {
        boolean hasS3Path;
        Configuration conf = this.getConf();
        String fromCli = this.getCommandFormat().getOptValue(REGION_FLAG);
        String fromConf = conf.get("fs.s3a.s3guard.ddb.region");
        boolean bl = hasS3Path = !paths.isEmpty();
        if (fromCli != null) {
            if (fromCli.isEmpty()) {
                throw S3GuardTool.invalidArgs("No region provided with -region flag", new Object[0]);
            }
            if (hasS3Path) {
                throw S3GuardTool.invalidArgs("Providing both an S3 path and the -region flag is not supported. If you need to specify a different region than the S3 bucket, configure fs.s3a.s3guard.ddb.region", new Object[0]);
            }
            conf.set("fs.s3a.s3guard.ddb.region", fromCli);
            return;
        }
        if (fromConf != null) {
            if (fromConf.isEmpty()) {
                throw S3GuardTool.invalidArgs("No region provided with config %s", "fs.s3a.s3guard.ddb.region");
            }
            return;
        }
        if (hasS3Path) {
            String s3Path = paths.get(0);
            this.initS3AFileSystem(s3Path);
            return;
        }
        throw S3GuardTool.invalidArgs("No region found from -region flag, config, or S3 bucket", new Object[0]);
    }

    private long getDeltaComponent(TimeUnit unit, String arg) {
        String raw = this.getCommandFormat().getOptValue(arg);
        if (raw == null || raw.isEmpty()) {
            return 0L;
        }
        Long parsed = Long.parseLong(raw);
        return unit.toMillis(parsed);
    }

    long ageOptionsToMsec() {
        long cliDelta = 0L;
        cliDelta += this.getDeltaComponent(TimeUnit.DAYS, DAYS_FLAG);
        cliDelta += this.getDeltaComponent(TimeUnit.HOURS, HOURS_FLAG);
        cliDelta += this.getDeltaComponent(TimeUnit.MINUTES, MINUTES_FLAG);
        return cliDelta += this.getDeltaComponent(TimeUnit.SECONDS, SECONDS_FLAG);
    }

    protected void addAgeOptions() {
        CommandFormat format = this.getCommandFormat();
        format.addOptionWithValue(DAYS_FLAG);
        format.addOptionWithValue(HOURS_FLAG);
        format.addOptionWithValue(MINUTES_FLAG);
        format.addOptionWithValue(SECONDS_FLAG);
    }

    protected void checkIfS3BucketIsGuarded(List<String> paths) throws IOException {
        String metadataStoreUri;
        String s3Path = "";
        if (!paths.isEmpty()) {
            s3Path = paths.get(0);
        }
        if ((metadataStoreUri = this.getCommandFormat().getOptValue(META_FLAG)) == null || metadataStoreUri.isEmpty()) {
            try (S3AFileSystem s3AFileSystem = (S3AFileSystem)S3AFileSystem.newInstance((URI)S3GuardTool.toUri(s3Path), (Configuration)this.getConf());){
                Preconditions.checkState((boolean)s3AFileSystem.hasMetadataStore(), (Object)("The S3 bucket is unguarded. " + this.getName() + " can not be used on an unguarded bucket."));
            }
        }
    }

    protected void checkBucketNameOrDDBTableNameProvided(List<String> paths) {
        String metadataStoreUri;
        String s3Path = null;
        if (!paths.isEmpty()) {
            s3Path = paths.get(0);
        }
        if ((metadataStoreUri = this.getCommandFormat().getOptValue(META_FLAG)) == null && s3Path == null) {
            throw S3GuardTool.invalidArgs("S3 bucket url or DDB table name have to be provided explicitly to use " + this.getName() + " command.", new Object[0]);
        }
    }

    protected MetadataStore initMetadataStore(boolean forceCreate) throws IOException {
        Configuration conf;
        block14: {
            block13: {
                if (this.getStore() != null) {
                    return this.getStore();
                }
                conf = this.filesystem == null ? this.getConf() : this.filesystem.getConf();
                String metaURI = this.getCommandFormat().getOptValue(META_FLAG);
                if (metaURI == null || metaURI.isEmpty()) break block13;
                URI uri = URI.create(metaURI);
                LOG.info("Create metadata store: {}", (Object)(uri + " scheme: " + uri.getScheme()));
                switch (uri.getScheme().toLowerCase(Locale.ENGLISH)) {
                    case "local": {
                        this.setStore(new LocalMetadataStore());
                        break;
                    }
                    case "dynamodb": {
                        this.setStore(new DynamoDBMetadataStore());
                        conf.set("fs.s3a.s3guard.ddb.table", uri.getAuthority());
                        if (forceCreate) {
                            conf.setBoolean("fs.s3a.s3guard.ddb.table.create", true);
                            break;
                        }
                        break block14;
                    }
                    default: {
                        throw new IOException(String.format("Metadata store %s is not supported", uri));
                    }
                }
                break block14;
            }
            this.setStore(new DynamoDBMetadataStore());
            if (forceCreate) {
                conf.setBoolean("fs.s3a.s3guard.ddb.table.create", true);
            }
        }
        if (this.filesystem == null) {
            this.getStore().initialize(conf, (ITtlTimeProvider)new S3Guard.TtlTimeProvider(conf));
        } else {
            this.getStore().initialize(this.filesystem, (ITtlTimeProvider)new S3Guard.TtlTimeProvider(conf));
        }
        LOG.info("Metadata store {} is initialized.", (Object)this.getStore());
        return this.getStore();
    }

    protected void initS3AFileSystem(String path) throws IOException {
        LOG.debug("Initializing S3A FS to {}", (Object)path);
        URI uri = S3GuardTool.toUri(path);
        Configuration conf = new Configuration(this.getConf());
        String nullStore = NullMetadataStore.class.getName();
        conf.set("fs.s3a.metadatastore.impl", nullStore);
        String bucket = uri.getHost();
        S3AUtils.setBucketOption(conf, bucket, "fs.s3a.metadatastore.impl", "org.apache.hadoop.fs.s3a.s3guard.NullMetadataStore");
        String updatedBucketOption = S3AUtils.getBucketOption(conf, bucket, "fs.s3a.metadatastore.impl");
        LOG.debug("updated bucket store option {}", (Object)updatedBucketOption);
        Preconditions.checkState((boolean)"org.apache.hadoop.fs.s3a.s3guard.NullMetadataStore".equals(updatedBucketOption), (String)"Expected bucket option to be %s but was %s", (Object)"org.apache.hadoop.fs.s3a.s3guard.NullMetadataStore", (Object)updatedBucketOption);
        this.bindFilesystem(FileSystem.newInstance((URI)uri, (Configuration)conf));
    }

    @VisibleForTesting
    boolean maybeInitFilesystem(List<String> paths) throws IOException {
        if (this.getFilesystem() == null) {
            if (!paths.isEmpty()) {
                this.initS3AFileSystem(paths.get(0));
            } else {
                LOG.debug("No path on command line, so not instantiating FS");
            }
        }
        return this.getFilesystem() != null;
    }

    protected List<String> parseArgs(String[] args) {
        return this.getCommandFormat().parse(args, 1);
    }

    protected S3AFileSystem getFilesystem() {
        return this.filesystem;
    }

    protected S3AFileSystem bindFilesystem(FileSystem bindingFS) {
        FileSystem fs = bindingFS;
        this.baseFS = bindingFS;
        while (fs instanceof FilterFileSystem) {
            fs = ((FilterFileSystem)fs).getRawFileSystem();
        }
        if (!(fs instanceof S3AFileSystem)) {
            throw new ExitUtil.ExitException(53, "Wrong filesystem for URI " + fs.getUri() + " : " + fs.getClass().getName());
        }
        this.filesystem = (S3AFileSystem)fs;
        return this.filesystem;
    }

    @VisibleForTesting
    public MetadataStore getStore() {
        return this.store;
    }

    @VisibleForTesting
    protected void setStore(MetadataStore store) {
        Preconditions.checkNotNull((Object)store);
        this.store = store;
    }

    protected void resetBindings() {
        this.store = null;
        this.filesystem = null;
    }

    protected CommandFormat getCommandFormat() {
        return this.commandFormat;
    }

    public final int run(String[] args) throws Exception {
        return this.run(args, System.out);
    }

    public abstract int run(String[] var1, PrintStream var2) throws Exception, ExitUtil.ExitException;

    protected void dumpFileSystemStatistics(PrintStream stream) {
        S3AFileSystem fs = this.getFilesystem();
        if (fs == null) {
            return;
        }
        S3GuardTool.println(stream, "%nIO Statistics for %s%n", fs.getUri());
        IOStatistics iostats = IOStatisticsSupport.retrieveIOStatistics((Object)fs);
        if (iostats != null) {
            S3GuardTool.println(stream, IOStatisticsLogging.ioStatisticsToPrettyString((IOStatistics)iostats), new Object[0]);
        } else {
            S3GuardTool.println(stream, "FileSystem does not provide IOStatistics", new Object[0]);
        }
        S3GuardTool.println(stream, "", new Object[0]);
    }

    protected static URI toUri(String s3Path) {
        URI uri;
        try {
            uri = new URI(s3Path);
        }
        catch (URISyntaxException e) {
            throw S3GuardTool.invalidArgs("Not a valid fileystem path: %s", s3Path);
        }
        return uri;
    }

    private static void printHelp() {
        if (command == null) {
            S3GuardTool.errorln("Usage: hadoop s3guard [command] [OPTIONS] [s3a://BUCKET]\n\nCommands: \n\tinit - initialize metadata repository\n\tdestroy - destroy the Metadata Store including its contents(all data in S3 is preserved)\n\tauthoritative - Audits a DynamoDB S3Guard repository for all the entries being 'authoritative'\n\tbucket-info - provide/check S3Guard information about a specific bucket\n\tdiff - report on delta between S3 and repository\n\tfsck - Compares S3 with MetadataStore, and returns a failure status if any rules or invariants are violated. Only works with DynamoDB metadata stores.\n\timport - import metadata from existing S3 data\n\tmarkers - View and manipulate S3 directory markers\n\tprune - truncate older metadata from repository (all data in S3 is preserved)\n\tset-capacity - Alter metadata store IO capacity\n\tselect - make an S3 Select call\n\tuploads - list or abort pending multipart uploads\n");
            S3GuardTool.errorln("\tperform S3Guard metadata store administrative commands.");
        } else {
            S3GuardTool.errorln("Usage: hadoop " + command.getUsage());
        }
        S3GuardTool.errorln();
        S3GuardTool.errorln(COMMON_USAGE);
    }

    protected static void errorln() {
        System.err.println();
    }

    protected static void errorln(String x) {
        System.err.println(x);
    }

    protected static void println(PrintStream out, String format, Object ... args) {
        out.println(String.format(format, args));
    }

    protected static void printStoreDiagnostics(PrintStream out, MetadataStore store) throws IOException {
        Map<String, String> diagnostics = store.getDiagnostics();
        out.println("Metadata Store Diagnostics:");
        for (Map.Entry<String, String> entry : diagnostics.entrySet()) {
            S3GuardTool.println(out, "\t%s=%s", entry.getKey(), entry.getValue());
        }
    }

    protected static ExitUtil.ExitException storeNotFound(FileNotFoundException e) {
        return new ExitUtil.ExitException(44, e.toString(), (Throwable)e);
    }

    protected static ExitUtil.ExitException invalidArgs(String format, Object ... args) {
        return S3GuardTool.exitException(40, format, args);
    }

    protected static ExitUtil.ExitException badState(String format, Object ... args) {
        int exitCode = 46;
        return S3GuardTool.exitException(exitCode, format, args);
    }

    protected static ExitUtil.ExitException userAborted(String format, Object ... args) {
        return S3GuardTool.exitException(-1, format, args);
    }

    protected static ExitUtil.ExitException exitException(int exitCode, String format, Object ... args) {
        return new ExitUtil.ExitException(exitCode, String.format(format, args));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static int run(Configuration conf, String ... args) throws Exception {
        int n;
        String[] otherArgs = new GenericOptionsParser(conf, args).getRemainingArgs();
        if (otherArgs.length == 0) {
            S3GuardTool.printHelp();
            throw new ExitUtil.ExitException(42, "No arguments provided");
        }
        String subCommand = otherArgs[0];
        LOG.debug("Executing command {}", (Object)subCommand);
        switch (subCommand) {
            case "init": {
                command = new Init(conf);
                break;
            }
            case "destroy": {
                command = new Destroy(conf);
                break;
            }
            case "import": {
                command = new Import(conf);
                break;
            }
            case "bucket-info": {
                command = new BucketInfo(conf);
                break;
            }
            case "diff": {
                command = new Diff(conf);
                break;
            }
            case "markers": {
                command = new MarkerTool(conf);
                break;
            }
            case "prune": {
                command = new Prune(conf);
                break;
            }
            case "set-capacity": {
                command = new SetCapacity(conf);
                break;
            }
            case "uploads": {
                command = new Uploads(conf);
                break;
            }
            case "select": {
                command = new SelectTool(conf);
                break;
            }
            case "fsck": {
                command = new Fsck(conf);
                break;
            }
            case "authoritative": {
                command = new Authoritative(conf);
                break;
            }
            default: {
                S3GuardTool.printHelp();
                throw new ExitUtil.ExitException(42, "Unknown command " + subCommand);
            }
        }
        try {
            n = ToolRunner.run((Configuration)conf, (Tool)command, (String[])otherArgs);
        }
        catch (Throwable throwable) {
            IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{command});
            throw throwable;
        }
        IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{command});
        return n;
    }

    public static void main(String[] args) {
        try {
            int ret = S3GuardTool.run(new Configuration(), args);
            S3GuardTool.exit(ret, "");
        }
        catch (CommandFormat.UnknownOptionException e) {
            S3GuardTool.errorln(e.getMessage());
            S3GuardTool.printHelp();
            S3GuardTool.exit(42, e.getMessage());
        }
        catch (ExitUtil.ExitException e) {
            LOG.debug("Exception raised", (Throwable)e);
            S3GuardTool.exit(e.getExitCode(), e.toString());
        }
        catch (FileNotFoundException e) {
            S3GuardTool.errorln(e.toString());
            LOG.debug("Not found:", (Throwable)e);
            S3GuardTool.exit(44, e.toString());
        }
        catch (Throwable e) {
            if (e instanceof ExitCodeProvider) {
                ExitCodeProvider ec = (ExitCodeProvider)e;
                LOG.debug("Exception raised", e);
                S3GuardTool.exit(ec.getExitCode(), e.toString());
            }
            e.printStackTrace(System.err);
            S3GuardTool.exit(-1, e.toString());
        }
    }

    protected static void exit(int status, String text) {
        ExitUtil.terminate((int)status, (String)text);
    }

    static class Authoritative
    extends S3GuardTool {
        public static final String NAME = "authoritative";
        public static final String CHECK_FLAG = "check-config";
        public static final String REQUIRE_AUTH = "required";
        public static final String PURPOSE = "Audits a DynamoDB S3Guard repository for all the entries being 'authoritative'";
        private static final String USAGE = "authoritative [OPTIONS] [s3a://PATH]\n\tAudits a DynamoDB S3Guard repository for all the entries being 'authoritative'\n\nOptions:\n  -required - Require directories under the path to be authoritative.\n  -check-config - Check the configuration for the path to be authoritative\n  -verbose - Verbose Output.\n";

        Authoritative(Configuration conf) {
            super(conf, CHECK_FLAG, REQUIRE_AUTH, S3GuardTool.VERBOSE);
        }

        @Override
        public String getName() {
            return NAME;
        }

        @Override
        public String getUsage() {
            return USAGE;
        }

        @Override
        public int run(String[] args, PrintStream out) throws InterruptedException, IOException {
            List<String> paths = this.parseArgs(args);
            if (paths.isEmpty()) {
                out.println(USAGE);
                throw Authoritative.invalidArgs("no arguments", new Object[0]);
            }
            this.maybeInitFilesystem(paths);
            this.initMetadataStore(false);
            String s3Path = paths.get(0);
            URI uri = Authoritative.toUri(s3Path);
            Path auditPath = uri.getPath().isEmpty() ? new Path("/") : new Path(uri.getPath());
            S3AFileSystem fs = this.getFilesystem();
            MetadataStore ms = this.getStore();
            if (!(ms instanceof DynamoDBMetadataStore)) {
                Authoritative.errorln(s3Path + " path uses MS: " + ms);
                Authoritative.errorln("authoritative can be only used with a DynamoDB-backed S3Guard table.");
                Authoritative.errorln(USAGE);
                return -1;
            }
            CommandFormat commandFormat = this.getCommandFormat();
            if (commandFormat.getOpt(CHECK_FLAG) && !fs.allowAuthoritative(auditPath)) {
                Authoritative.errorln("Path " + auditPath + " is not configured to be authoritative");
                return 5;
            }
            AuthoritativeAuditOperation audit = new AuthoritativeAuditOperation(fs.createStoreContext(), (DynamoDBMetadataStore)ms, commandFormat.getOpt(REQUIRE_AUTH), commandFormat.getOpt(S3GuardTool.VERBOSE));
            audit.audit(fs.qualify(auditPath));
            out.flush();
            return 0;
        }
    }

    static class Fsck
    extends S3GuardTool {
        public static final String CHECK_FLAG = "check";
        public static final String DDB_MS_CONSISTENCY_FLAG = "internal";
        public static final String FIX_FLAG = "fix";
        public static final String NAME = "fsck";
        public static final String PURPOSE = "Compares S3 with MetadataStore, and returns a failure status if any rules or invariants are violated. Only works with DynamoDB metadata stores.";
        private static final String USAGE = "fsck [OPTIONS] [s3a://BUCKET]\n\tCompares S3 with MetadataStore, and returns a failure status if any rules or invariants are violated. Only works with DynamoDB metadata stores.\n\nCommon options:\n  -check Check the metadata store for errors, but do not fix any issues.\n  -internal Check the dynamodb metadata store for internal consistency.\n  -fix Fix the errors found in the metadata store. Can be used with check or internal flags. \n\t\tFixes: \n\t\t\t- Remove orphan entries from DDB.\n";

        Fsck(Configuration conf) {
            super(conf, CHECK_FLAG, DDB_MS_CONSISTENCY_FLAG, FIX_FLAG);
        }

        @Override
        public String getName() {
            return NAME;
        }

        @Override
        public String getUsage() {
            return USAGE;
        }

        @Override
        public int run(String[] args, PrintStream out) throws InterruptedException, IOException {
            List<S3GuardFsck.ComparePair> violations;
            S3GuardFsck s3GuardFsck;
            List<String> paths = this.parseArgs(args);
            if (paths.isEmpty()) {
                out.println(USAGE);
                throw Fsck.invalidArgs("no arguments", new Object[0]);
            }
            int exitValue = 0;
            CommandFormat commandFormat = this.getCommandFormat();
            int flags = this.countTrue(commandFormat.getOpt(CHECK_FLAG), commandFormat.getOpt(DDB_MS_CONSISTENCY_FLAG));
            if (flags > 1) {
                out.println(USAGE);
                throw Fsck.invalidArgs("There should be only one parameter used for checking.", new Object[0]);
            }
            if (flags == 0 && commandFormat.getOpt(FIX_FLAG)) {
                Fsck.errorln("fix flag can be used with either check or internal flag, but not alone.");
                Fsck.errorln(USAGE);
                return -1;
            }
            String s3Path = paths.get(0);
            try {
                this.initS3AFileSystem(s3Path);
            }
            catch (Exception e) {
                Fsck.errorln("Failed to initialize S3AFileSystem from path: " + s3Path);
                throw e;
            }
            URI uri = Fsck.toUri(s3Path);
            Path root = uri.getPath().isEmpty() ? new Path("/") : new Path(uri.getPath());
            S3AFileSystem fs = this.getFilesystem();
            this.initMetadataStore(false);
            MetadataStore ms = this.getStore();
            if (ms == null || !(ms instanceof DynamoDBMetadataStore)) {
                Fsck.errorln(s3Path + " path uses metadata store: " + ms);
                Fsck.errorln("fsck can be only used with a DynamoDB backed s3a bucket.");
                Fsck.errorln(USAGE);
                return -1;
            }
            if (commandFormat.getOpt(CHECK_FLAG)) {
                s3GuardFsck = new S3GuardFsck(fs, ms);
                violations = s3GuardFsck.compareS3ToMs(fs.qualify(root));
            } else if (commandFormat.getOpt(DDB_MS_CONSISTENCY_FLAG)) {
                s3GuardFsck = new S3GuardFsck(fs, ms);
                violations = s3GuardFsck.checkDdbInternalConsistency(fs.qualify(root));
            } else {
                Fsck.errorln("No supported operation is selected.");
                Fsck.errorln(USAGE);
                return -1;
            }
            if (commandFormat.getOpt(FIX_FLAG)) {
                s3GuardFsck = new S3GuardFsck(fs, ms);
                s3GuardFsck.fixViolations(violations);
            }
            out.flush();
            if (violations == null || violations.size() > 0) {
                exitValue = -1;
            }
            return exitValue;
        }

        int countTrue(Boolean ... bools) {
            return (int)Arrays.stream(bools).filter(p -> p).count();
        }
    }

    static class Uploads
    extends S3GuardTool {
        public static final String NAME = "uploads";
        public static final String ABORT = "abort";
        public static final String LIST = "list";
        public static final String EXPECT = "expect";
        public static final String FORCE = "force";
        public static final String PURPOSE = "list or abort pending multipart uploads";
        private static final String USAGE = "uploads [OPTIONS] s3a://BUCKET[/path]\n\tlist or abort pending multipart uploads\n\nCommon options:\n (-list | -expect <num-uploads> | -abort) [-verbose] [<age-options>] [-force]\n\t - Under given path, list or delete all uploads, or only those \nolder than specified by <age-options>\n<age-options> are any combination of the integer-valued options:\n\t[-days <days>] [-hours <hours>] [-minutes <minutes>] [-seconds <seconds>]\n-expect is similar to list, except no output is printed,\n\tbut the exit code will be an error if the provided number\n\tis different that the number of uploads found by the command.\n-force option prevents the \"Are you sure\" prompt when\n\tusing -abort";
        public static final String TOTAL = "Total";
        private Mode mode = null;
        private int expectedCount;
        private long ageMsec = 0L;
        private boolean verbose = false;
        private boolean force = false;
        private String prefix;

        Uploads(Configuration conf) {
            super(conf, ABORT, LIST, S3GuardTool.VERBOSE, FORCE);
            this.addAgeOptions();
            this.getCommandFormat().addOptionWithValue(EXPECT);
        }

        @Override
        public String getName() {
            return NAME;
        }

        @Override
        public String getUsage() {
            return USAGE;
        }

        @Override
        public int run(String[] args, PrintStream out) throws InterruptedException, IOException {
            List<String> paths = this.parseArgs(args);
            if (paths.isEmpty()) {
                Uploads.errorln(this.getUsage());
                throw Uploads.invalidArgs("No options specified", new Object[0]);
            }
            this.processArgs(paths, out);
            Uploads.println(out, "Listing uploads under path \"%s\"", this.prefix);
            this.promptBeforeAbort(out);
            this.processUploads(out);
            if (this.verbose) {
                this.dumpFileSystemStatistics(out);
            }
            out.flush();
            return 0;
        }

        private void promptBeforeAbort(PrintStream out) throws IOException {
            if (this.mode != Mode.ABORT || this.force) {
                return;
            }
            Scanner scanner = new Scanner(System.in, "UTF-8");
            out.println("Are you sure you want to delete any pending uploads? (yes/no) >");
            String response = scanner.nextLine();
            if (!"yes".equalsIgnoreCase(response)) {
                throw S3GuardTool.userAborted("User did not answer yes, quitting.", new Object[0]);
            }
        }

        private void processUploads(PrintStream out) throws IOException {
            S3AFileSystem fs = this.getFilesystem();
            MultipartUtils.UploadIterator uploads = fs.listUploads(this.prefix);
            AuditSpanS3A span = fs.createSpan("multipart_upload_aborted", this.prefix, null);
            WriteOperationHelper writeOperationHelper = fs.getWriteOperationHelper();
            int count = 0;
            while (uploads.hasNext()) {
                MultipartUpload upload = uploads.next();
                if (!this.olderThan(upload, this.ageMsec)) continue;
                ++count;
                if (this.mode == Mode.ABORT || this.mode == Mode.LIST || this.verbose) {
                    Uploads.println(out, "%s%s %s", this.mode == Mode.ABORT ? "Deleting: " : "", upload.getKey(), upload.getUploadId());
                }
                if (this.mode != Mode.ABORT) continue;
                writeOperationHelper.abortMultipartUpload(upload.getKey(), upload.getUploadId(), true, Invoker.LOG_EVENT);
            }
            span.deactivate();
            if (this.mode != Mode.EXPECT || this.verbose) {
                Uploads.println(out, "%s %d uploads %s.", TOTAL, count, this.mode == Mode.ABORT ? "deleted" : "found");
            }
            if (this.mode == Mode.EXPECT && count != this.expectedCount) {
                throw Uploads.badState("Expected upload count under %s: %d, found %d", this.prefix, this.expectedCount, count);
            }
        }

        private boolean olderThan(MultipartUpload u, long msec) {
            if (msec == 0L) {
                return true;
            }
            Date ageDate = new Date(System.currentTimeMillis() - msec);
            return ageDate.compareTo(u.getInitiated()) >= 0;
        }

        private void processArgs(List<String> args, PrintStream out) throws IOException {
            String expectVal;
            CommandFormat commands = this.getCommandFormat();
            String err = "Can only specify one of -list,  -abort, and expect";
            if (commands.getOpt(LIST)) {
                this.mode = Mode.LIST;
            }
            if (commands.getOpt(ABORT)) {
                if (this.mode != null) {
                    throw Uploads.invalidArgs(err, new Object[0]);
                }
                this.mode = Mode.ABORT;
            }
            if ((expectVal = commands.getOptValue(EXPECT)) != null) {
                if (this.mode != null) {
                    throw Uploads.invalidArgs(err, new Object[0]);
                }
                this.mode = Mode.EXPECT;
                this.expectedCount = Integer.parseInt(expectVal);
            }
            if (this.mode == null) {
                this.vprintln(out, "No mode specified, defaulting to -list", new Object[0]);
                this.mode = Mode.LIST;
            }
            if (commands.getOpt(S3GuardTool.VERBOSE)) {
                this.verbose = true;
            }
            if (commands.getOpt(FORCE)) {
                this.force = true;
            }
            this.ageMsec = this.ageOptionsToMsec();
            String s3Path = args.get(0);
            URI uri = S3GuardTool.toUri(s3Path);
            this.prefix = uri.getPath();
            if (this.prefix.length() > 0) {
                this.prefix = this.prefix.substring(1);
            }
            this.vprintln(out, "Command: %s, age %d msec, path %s (prefix \"%s\")", this.mode.name(), this.ageMsec, s3Path, this.prefix);
            this.initS3AFileSystem(s3Path);
        }

        private void vprintln(PrintStream out, String format, Object ... args) {
            if (this.verbose) {
                out.println(String.format(format, args));
            }
        }

        private static enum Mode {
            LIST,
            EXPECT,
            ABORT;

        }
    }

    public static class BucketInfo
    extends S3GuardTool {
        public static final String BUCKET_INFO = "bucket-info";
        public static final String NAME = "bucket-info";
        public static final String GUARDED_FLAG = "guarded";
        public static final String UNGUARDED_FLAG = "unguarded";
        public static final String AUTH_FLAG = "auth";
        public static final String NONAUTH_FLAG = "nonauth";
        public static final String ENCRYPTION_FLAG = "encryption";
        public static final String MAGIC_FLAG = "magic";
        public static final String MARKERS_FLAG = "markers";
        public static final String MARKERS_AWARE = "aware";
        public static final String PURPOSE = "provide/check S3Guard information about a specific bucket";
        private static final String USAGE = "bucket-info [OPTIONS] s3a://BUCKET\n\tprovide/check S3Guard information about a specific bucket\n\nCommon options:\n  -guarded - Require S3Guard\n  -unguarded - Force S3Guard to be disabled\n  -auth - Require the S3Guard mode to be \"authoritative\"\n  -nonauth - Require the S3Guard mode to be \"non-authoritative\"\n  -magic - Require the S3 filesystem to be support the \"magic\" committer\n  -encryption (none, sse-s3, sse-kms) - Require encryption policy\n  -markers (aware, keep, delete, authoritative) - directory markers policy\n";
        @VisibleForTesting
        public static final String LOCATION_UNKNOWN = "Location unknown -caller lacks s3:GetBucketLocation permission";
        @VisibleForTesting
        public static final String IS_MARKER_AWARE = "\tThe S3A connector is compatible with buckets where directory markers are not deleted";

        public BucketInfo(Configuration conf) {
            super(conf, GUARDED_FLAG, UNGUARDED_FLAG, AUTH_FLAG, NONAUTH_FLAG, MAGIC_FLAG);
            CommandFormat format = this.getCommandFormat();
            format.addOptionWithValue(ENCRYPTION_FLAG);
            format.addOptionWithValue(MARKERS_FLAG);
        }

        @Override
        public String getName() {
            return "bucket-info";
        }

        @Override
        public String getUsage() {
            return USAGE;
        }

        @Override
        public int run(String[] args, PrintStream out) throws InterruptedException, IOException {
            List<String> paths = this.parseArgs(args);
            if (paths.isEmpty()) {
                BucketInfo.errorln(this.getUsage());
                throw BucketInfo.invalidArgs("No bucket specified", new Object[0]);
            }
            String s3Path = paths.get(0);
            CommandFormat commands = this.getCommandFormat();
            URI fsURI = BucketInfo.toUri(s3Path);
            Configuration unguardedConf = this.getConf();
            if (commands.getOpt(UNGUARDED_FLAG)) {
                LOG.debug("Unguarded flag is passed to command :" + this.getName());
                S3AUtils.clearBucketOption(unguardedConf, fsURI.getHost(), "fs.s3a.metadatastore.impl");
                unguardedConf.set("fs.s3a.metadatastore.impl", "org.apache.hadoop.fs.s3a.s3guard.NullMetadataStore");
            }
            S3AFileSystem fs = this.bindFilesystem(FileSystem.newInstance((URI)fsURI, (Configuration)unguardedConf));
            Configuration conf = fs.getConf();
            URI fsUri = fs.getUri();
            MetadataStore store = fs.getMetadataStore();
            BucketInfo.println(out, "Filesystem %s", fsUri);
            try {
                BucketInfo.println(out, "Location: %s", fs.getBucketLocation());
            }
            catch (AccessDeniedException e) {
                LOG.debug("failed to get bucket location", (Throwable)e);
                BucketInfo.println(out, LOCATION_UNKNOWN, new Object[0]);
            }
            boolean usingS3Guard = !(store instanceof NullMetadataStore);
            boolean authMode = false;
            if (usingS3Guard) {
                out.printf("Filesystem %s is using S3Guard with store %s%n", fsUri, store.toString());
                this.printOption(out, "Authoritative Metadata Store", "fs.s3a.metadatastore.authoritative", "false");
                this.printOption(out, "Authoritative Path", "fs.s3a.authoritative.path", "");
                Collection<String> authoritativePaths = S3Guard.getAuthoritativePaths(fs);
                if (!authoritativePaths.isEmpty()) {
                    BucketInfo.println(out, "Qualified Authoritative Paths:", new Object[0]);
                    for (String path : authoritativePaths) {
                        BucketInfo.println(out, "\t%s", path);
                    }
                    BucketInfo.println(out, "", new Object[0]);
                }
                authMode = conf.getBoolean("fs.s3a.metadatastore.authoritative", false);
                long ttl = conf.getTimeDuration("fs.s3a.metadatastore.metadata.ttl", Constants.DEFAULT_METADATASTORE_METADATA_TTL, TimeUnit.MILLISECONDS);
                BucketInfo.println(out, "\tMetadata time to live: (set in %s) = %s", "fs.s3a.metadatastore.metadata.ttl", DurationFormatUtils.formatDurationHMS((long)ttl));
                BucketInfo.printStoreDiagnostics(out, store);
            } else {
                BucketInfo.println(out, "Filesystem %s is not using S3Guard", fsUri);
            }
            BucketInfo.println(out, "%nS3A Client", new Object[0]);
            this.printOption(out, "\tSigning Algorithm", "fs.s3a.signing-algorithm", "(unset)");
            String endpoint = conf.getTrimmed("fs.s3a.endpoint", "");
            BucketInfo.println(out, "\tEndpoint: %s=%s", "fs.s3a.endpoint", StringUtils.isNotEmpty((CharSequence)endpoint) ? endpoint : "(unset)");
            String encryption = this.printOption(out, "\tEncryption", "fs.s3a.encryption.algorithm", "none");
            this.printOption(out, "\tInput seek policy", "fs.s3a.experimental.input.fadvise", "normal");
            this.printOption(out, "\tChange Detection Source", "fs.s3a.change.detection.source", "etag");
            this.printOption(out, "\tChange Detection Mode", "fs.s3a.change.detection.mode", "server");
            BucketInfo.println(out, "%nS3A Committers", new Object[0]);
            boolean magic = fs.hasPathCapability(new Path(s3Path), "fs.s3a.capability.magic.committer");
            BucketInfo.println(out, "\tThe \"magic\" committer %s supported in the filesystem", magic ? "is" : "is not");
            this.printOption(out, "\tS3A Committer factory class", CommitConstants.S3A_COMMITTER_FACTORY_KEY, "");
            String committer = conf.getTrimmed("fs.s3a.committer.name", "file");
            this.printOption(out, "\tS3A Committer name", "fs.s3a.committer.name", "file");
            switch (committer) {
                case "file": {
                    BucketInfo.println(out, "The original 'file' commmitter is active -this is slow and potentially unsafe", new Object[0]);
                    break;
                }
                case "staging": {
                    BucketInfo.println(out, "The 'staging' committer is used -prefer the 'directory' committer", new Object[0]);
                }
                case "directory": 
                case "partitioned": {
                    this.printOption(out, "\tCluster filesystem staging directory", "fs.s3a.committer.staging.tmp.path", "tmp/staging");
                    this.printOption(out, "\tLocal filesystem buffer directory", "fs.s3a.buffer.dir", "");
                    this.printOption(out, "\tFile conflict resolution", "fs.s3a.committer.staging.conflict-mode", "append");
                    break;
                }
                case "magic": {
                    this.printOption(out, "\tStore magic committer integration", "fs.s3a.committer.magic.enabled", Boolean.toString(true));
                    if (magic) break;
                    BucketInfo.println(out, "Warning: although the magic committer is enabled, the store does not support it", new Object[0]);
                    break;
                }
                default: {
                    BucketInfo.println(out, "\tWarning: committer '%s' is unknown", committer);
                }
            }
            BucketInfo.println(out, "%nSecurity", new Object[0]);
            if (fs.getDelegationTokens().isPresent()) {
                S3ADelegationTokens dtIntegration = fs.getDelegationTokens().get();
                BucketInfo.println(out, "\tDelegation Support enabled: token kind = %s", dtIntegration.getTokenKind());
                UserGroupInformation.AuthenticationMethod authenticationMethod = UserGroupInformation.getCurrentUser().getAuthenticationMethod();
                BucketInfo.println(out, "\tHadoop security mode: %s", authenticationMethod);
                if (UserGroupInformation.isSecurityEnabled()) {
                    BucketInfo.println(out, "\tWarning: security is disabled; tokens will not be collected", new Object[0]);
                }
            } else {
                BucketInfo.println(out, "\tDelegation token support is disabled", new Object[0]);
            }
            if (usingS3Guard) {
                if (commands.getOpt(UNGUARDED_FLAG)) {
                    throw BucketInfo.badState("S3Guard is enabled for %s", fsUri);
                }
                if (commands.getOpt(AUTH_FLAG) && !authMode) {
                    throw BucketInfo.badState("S3Guard is enabled for %s, but not in authoritative mode", fsUri);
                }
                if (commands.getOpt(NONAUTH_FLAG) && authMode) {
                    throw BucketInfo.badState("S3Guard is enabled in authoritative mode for %s", fsUri);
                }
            } else if (commands.getOpt(GUARDED_FLAG)) {
                throw BucketInfo.badState("S3Guard is not enabled for %s", fsUri);
            }
            if (commands.getOpt(MAGIC_FLAG) && !magic) {
                throw BucketInfo.badState("The magic committer is not enabled for %s", fsUri);
            }
            String desiredEncryption = this.getCommandFormat().getOptValue(ENCRYPTION_FLAG);
            if (StringUtils.isNotEmpty((CharSequence)desiredEncryption) && !desiredEncryption.equalsIgnoreCase(encryption)) {
                throw BucketInfo.badState("Bucket %s: required encryption is %s but actual encryption is %s", fsUri, desiredEncryption, encryption);
            }
            this.processMarkerOption(out, fs, this.getCommandFormat().getOptValue(MARKERS_FLAG));
            out.flush();
            return 0;
        }

        private void processMarkerOption(PrintStream out, S3AFileSystem fs, String marker) {
            BucketInfo.println(out, "%nSecurity", new Object[0]);
            DirectoryPolicy markerPolicy = fs.getDirectoryMarkerPolicy();
            String desc = markerPolicy.describe();
            BucketInfo.println(out, "\tThe directory marker policy is \"%s\"", desc);
            String pols = DirectoryPolicyImpl.availablePolicies().stream().map(DirectoryPolicy.MarkerPolicy::getOptionName).collect(Collectors.joining(", "));
            BucketInfo.println(out, "\tAvailable Policies: %s", pols);
            this.printOption(out, "\tAuthoritative paths", "fs.s3a.authoritative.path", "");
            DirectoryPolicy.MarkerPolicy mp = markerPolicy.getMarkerPolicy();
            String desiredMarker = marker == null ? "" : marker.trim();
            String optionName = mp.getOptionName();
            if (!desiredMarker.isEmpty()) {
                if (MARKERS_AWARE.equalsIgnoreCase(desiredMarker)) {
                    BucketInfo.println(out, IS_MARKER_AWARE, new Object[0]);
                } else if (!optionName.equalsIgnoreCase(desiredMarker)) {
                    throw BucketInfo.badState("Bucket %s: required marker policy is \"%s\" but actual policy is \"%s\"", fs.getUri(), desiredMarker, optionName);
                }
            }
        }

        private String printOption(PrintStream out, String description, String key, String defVal) {
            String t = this.getFilesystem().getConf().getTrimmed(key, defVal);
            BucketInfo.println(out, "%s: %s=%s", description, key, t);
            return t;
        }
    }

    static class Prune
    extends S3GuardTool {
        public static final String NAME = "prune";
        public static final String PURPOSE = "truncate older metadata from repository (all data in S3 is preserved)";
        public static final String TOMBSTONE = "tombstone";
        private static final String USAGE = "prune [OPTIONS] [s3a://BUCKET]\n\ttruncate older metadata from repository (all data in S3 is preserved)\n\nCommon options:\n  -meta URL - Metadata repository details (implementation-specific)\n[-tombstone]\nAge options. Any combination of these integer-valued options:\n[-days <days>] [-hours <hours>] [-minutes <minutes>] [-seconds <seconds>]\nAmazon DynamoDB-specific options:\n  -region REGION - Service region for connections\n\n  URLs for Amazon DynamoDB are of the form dynamodb://TABLE_NAME.\n  Specifying both the -region option and an S3A path\n  is not supported.";

        Prune(Configuration conf) {
            super(conf, TOMBSTONE);
            this.addAgeOptions();
        }

        @VisibleForTesting
        void setMetadataStore(MetadataStore ms) {
            Preconditions.checkNotNull((Object)ms);
            this.setStore(ms);
        }

        @Override
        public String getName() {
            return NAME;
        }

        @Override
        public String getUsage() {
            return USAGE;
        }

        @Override
        public int run(String[] args, PrintStream out) throws InterruptedException, IOException {
            List<String> paths = this.parseArgs(args);
            try {
                this.checkBucketNameOrDDBTableNameProvided(paths);
                this.parseDynamoDBRegion(paths);
            }
            catch (ExitUtil.ExitException e) {
                Prune.errorln(USAGE);
                throw e;
            }
            this.maybeInitFilesystem(paths);
            this.initMetadataStore(false);
            Configuration conf = this.getConf();
            long confDelta = conf.getLong("fs.s3a.s3guard.cli.prune.age", 0L);
            long cliDelta = this.ageOptionsToMsec();
            if (confDelta <= 0L && cliDelta <= 0L) {
                Prune.errorln("You must specify a positive age for metadata to prune.");
            }
            long delta = confDelta;
            if (cliDelta > 0L) {
                delta = cliDelta;
            }
            long now = System.currentTimeMillis();
            long divide = now - delta;
            String keyPrefix = "/";
            if (paths.size() > 0) {
                Path path = new Path(paths.get(0));
                keyPrefix = PathMetadataDynamoDBTranslation.pathToParentKey(path);
            }
            MetadataStore.PruneMode mode = MetadataStore.PruneMode.ALL_BY_MODTIME;
            if (this.getCommandFormat().getOpt(TOMBSTONE)) {
                mode = MetadataStore.PruneMode.TOMBSTONES_BY_LASTUPDATED;
            }
            try {
                this.getStore().prune(mode, divide, keyPrefix);
            }
            catch (UnsupportedOperationException e) {
                Prune.errorln("Prune operation not supported in metadata store.");
            }
            out.flush();
            return 0;
        }
    }

    static class Diff
    extends S3GuardTool {
        public static final String NAME = "diff";
        public static final String PURPOSE = "report on delta between S3 and repository";
        private static final String USAGE = "diff [OPTIONS] s3a://BUCKET\n\treport on delta between S3 and repository\n\nCommon options:\n  -meta URL - Metadata repository details (implementation-specific)\n\nAmazon DynamoDB-specific options:\n  -region REGION - Service region for connections\n\n  URLs for Amazon DynamoDB are of the form dynamodb://TABLE_NAME.\n  Specifying both the -region option and an S3A path\n  is not supported.";
        private static final String SEP = "\t";
        static final String S3_PREFIX = "S3";
        static final String MS_PREFIX = "MS";

        Diff(Configuration conf) {
            super(conf, new String[0]);
        }

        @Override
        public String getName() {
            return NAME;
        }

        @Override
        public String getUsage() {
            return USAGE;
        }

        private static String formatFileStatus(FileStatus status) {
            return String.format("%s%s%d%s%s", status.isDirectory() ? "D" : "F", SEP, status.getLen(), SEP, status.getPath().toString());
        }

        private static boolean differ(FileStatus thisOne, FileStatus thatOne) {
            Preconditions.checkArgument((thisOne != null || thatOne != null ? 1 : 0) != 0);
            return thisOne == null || thatOne == null || thisOne.getLen() != thatOne.getLen() || thisOne.isDirectory() != thatOne.isDirectory() || !thisOne.isDirectory() && thisOne.getModificationTime() != thatOne.getModificationTime();
        }

        private static void printDiff(FileStatus msStatus, FileStatus s3Status, PrintStream out) {
            Preconditions.checkArgument((msStatus != null || s3Status != null ? 1 : 0) != 0);
            if (msStatus != null && s3Status != null) {
                Preconditions.checkArgument((boolean)msStatus.getPath().equals((Object)s3Status.getPath()), (Object)String.format("The path from metadata store and s3 are different: ms=%s s3=%s", msStatus.getPath(), s3Status.getPath()));
            }
            if (Diff.differ(msStatus, s3Status)) {
                if (s3Status != null) {
                    Diff.println(out, "%s%s%s", S3_PREFIX, SEP, Diff.formatFileStatus(s3Status));
                }
                if (msStatus != null) {
                    Diff.println(out, "%s%s%s", MS_PREFIX, SEP, Diff.formatFileStatus(msStatus));
                }
            }
        }

        private void compareDir(FileStatus msDir, FileStatus s3Dir, PrintStream out) throws IOException {
            DirListingMetadata dirMeta;
            Preconditions.checkArgument((msDir != null || s3Dir != null ? 1 : 0) != 0, (Object)"The path does not exist in metadata store and on s3.");
            if (msDir != null && s3Dir != null) {
                Preconditions.checkArgument((boolean)msDir.getPath().equals((Object)s3Dir.getPath()), (Object)String.format("The path from metadata store and s3 are different: ms=%s s3=%s", msDir.getPath(), s3Dir.getPath()));
            }
            HashMap<Path, FileStatus> s3Children = new HashMap<Path, FileStatus>();
            if (s3Dir != null && s3Dir.isDirectory()) {
                for (FileStatus status : this.getFilesystem().listStatus(s3Dir.getPath())) {
                    s3Children.put(status.getPath(), status);
                }
            }
            HashMap<Path, S3AFileStatus> msChildren = new HashMap<Path, S3AFileStatus>();
            if (msDir != null && msDir.isDirectory() && (dirMeta = this.getStore().listChildren(msDir.getPath())) != null) {
                for (PathMetadata meta : dirMeta.getListing()) {
                    S3AFileStatus status = meta.getFileStatus();
                    msChildren.put(status.getPath(), status);
                }
            }
            HashSet allPaths = new HashSet(s3Children.keySet());
            allPaths.addAll(msChildren.keySet());
            for (Path path : allPaths) {
                FileStatus s3Status = (FileStatus)s3Children.get(path);
                FileStatus msStatus = (FileStatus)msChildren.get(path);
                Diff.printDiff(msStatus, s3Status, out);
                if ((s3Status == null || !s3Status.isDirectory()) && (msStatus == null || !msStatus.isDirectory())) continue;
                this.compareDir(msStatus, s3Status, out);
            }
            out.flush();
        }

        private void compareRoot(Path path, PrintStream out) throws IOException {
            Path qualified = this.getFilesystem().makeQualified(path);
            FileStatus s3Status = null;
            try {
                s3Status = this.getFilesystem().getFileStatus(qualified);
            }
            catch (FileNotFoundException fileNotFoundException) {
                // empty catch block
            }
            PathMetadata meta = this.getStore().get(qualified);
            S3AFileStatus msStatus = meta != null && !meta.isDeleted() ? meta.getFileStatus() : null;
            this.compareDir(msStatus, s3Status, out);
        }

        @Override
        @VisibleForTesting
        public int run(String[] args, PrintStream out) throws IOException {
            List<String> paths = this.parseArgs(args);
            if (paths.isEmpty()) {
                out.println(USAGE);
                throw Diff.invalidArgs("no arguments", new Object[0]);
            }
            String s3Path = paths.get(0);
            this.initS3AFileSystem(s3Path);
            this.initMetadataStore(false);
            URI uri = Diff.toUri(s3Path);
            Path root = uri.getPath().isEmpty() ? new Path("/") : new Path(uri.getPath());
            root = this.getFilesystem().makeQualified(root);
            this.compareRoot(root, out);
            out.flush();
            return 0;
        }
    }

    static class Import
    extends S3GuardTool {
        public static final String NAME = "import";
        public static final String PURPOSE = "import metadata from existing S3 data";
        public static final String AUTH_FLAG = "authoritative";
        private static final String USAGE = "import [OPTIONS] [s3a://PATH]\n\timport metadata from existing S3 data\n\nCommon options:\n  -authoritative - Mark imported directory data as authoritative.\n  -verbose - Verbose Output.\n  -meta URL - Metadata repository details (implementation-specific)\n\nAmazon DynamoDB-specific options:\n  -region REGION - Service region for connections\n\n  URLs for Amazon DynamoDB are of the form dynamodb://TABLE_NAME.\n  Specifying both the -region option and an S3A path\n  is not supported.";

        Import(Configuration conf) {
            super(conf, AUTH_FLAG, S3GuardTool.VERBOSE);
        }

        @Override
        public String getName() {
            return NAME;
        }

        @Override
        public String getUsage() {
            return USAGE;
        }

        @Override
        public int run(String[] args, PrintStream out) throws Exception {
            List<String> paths = this.parseArgs(args);
            if (paths.isEmpty()) {
                Import.errorln(this.getUsage());
                throw Import.invalidArgs("no arguments", new Object[0]);
            }
            String s3Path = paths.get(0);
            this.initS3AFileSystem(s3Path);
            URI uri = Import.toUri(s3Path);
            String filePath = uri.getPath();
            if (filePath.isEmpty()) {
                filePath = "/";
            }
            Path path = new Path(filePath);
            S3AFileStatus status = (S3AFileStatus)this.getFilesystem().getFileStatus(path);
            try {
                this.initMetadataStore(false);
            }
            catch (FileNotFoundException e) {
                throw Import.storeNotFound(e);
            }
            CommandFormat commandFormat = this.getCommandFormat();
            boolean verbose = commandFormat.getOpt(S3GuardTool.VERBOSE);
            ImportOperation importer = new ImportOperation(this.getFilesystem(), this.getStore(), status, commandFormat.getOpt(AUTH_FLAG), verbose);
            long items = importer.execute();
            Import.println(out, "Inserted %d items into Metadata Store", items);
            if (verbose) {
                this.dumpFileSystemStatistics(out);
            }
            return 0;
        }
    }

    static class Destroy
    extends S3GuardTool {
        public static final String NAME = "destroy";
        public static final String PURPOSE = "destroy the Metadata Store including its contents(all data in S3 is preserved)";
        private static final String USAGE = "destroy [OPTIONS] [s3a://BUCKET]\n\tdestroy the Metadata Store including its contents(all data in S3 is preserved)\n\nCommon options:\n  -meta URL - Metadata repository details (implementation-specific)\n\nAmazon DynamoDB-specific options:\n  -region REGION - Service region for connections\n\n  URLs for Amazon DynamoDB are of the form dynamodb://TABLE_NAME.\n  Specifying both the -region option and an S3A path\n  is not supported.";

        Destroy(Configuration conf) {
            super(conf, new String[0]);
        }

        @Override
        public String getName() {
            return NAME;
        }

        @Override
        public String getUsage() {
            return USAGE;
        }

        @Override
        public int run(String[] args, PrintStream out) throws Exception {
            List<String> paths = this.parseArgs(args);
            try {
                this.checkBucketNameOrDDBTableNameProvided(paths);
                this.checkIfS3BucketIsGuarded(paths);
                this.parseDynamoDBRegion(paths);
                this.maybeInitFilesystem(paths);
            }
            catch (ExitUtil.ExitException e) {
                Destroy.errorln(USAGE);
                throw e;
            }
            try {
                this.initMetadataStore(false);
            }
            catch (FileNotFoundException e) {
                Destroy.println(out, "Metadata Store does not exist.", new Object[0]);
                LOG.debug("Failed to bind to store to be destroyed", (Throwable)e);
                return 0;
            }
            Preconditions.checkState((this.getStore() != null ? 1 : 0) != 0, (Object)"Metadata Store is not initialized");
            try {
                this.getStore().destroy();
            }
            catch (TableDeleteTimeoutException e) {
                LOG.warn("The table is been deleted but it is still (briefly) listed as present in AWS");
                LOG.debug("Timeout waiting for table disappearing", (Throwable)((Object)e));
            }
            Destroy.println(out, "Metadata store is deleted.", new Object[0]);
            return 0;
        }
    }

    static class SetCapacity
    extends S3GuardTool {
        public static final String NAME = "set-capacity";
        public static final String PURPOSE = "Alter metadata store IO capacity";
        public static final String READ_CAP_INVALID = "Read capacity must have value greater than or equal to 1.";
        public static final String WRITE_CAP_INVALID = "Write capacity must have value greater than or equal to 1.";
        private static final String USAGE = "set-capacity [OPTIONS] [s3a://BUCKET]\n\tAlter metadata store IO capacity\n\nCommon options:\n  -meta URL - Metadata repository details (implementation-specific)\n\nAmazon DynamoDB-specific options:\n  -read UNIT - Provisioned read throughput units\n  -write UNIT - Provisioned write through put units\n\n  URLs for Amazon DynamoDB are of the form dynamodb://TABLE_NAME.\n  Specifying both the -region option and an S3A path\n  is not supported.";

        SetCapacity(Configuration conf) {
            super(conf, new String[0]);
            this.getCommandFormat().addOptionWithValue(S3GuardTool.READ_FLAG);
            this.getCommandFormat().addOptionWithValue(S3GuardTool.WRITE_FLAG);
        }

        @Override
        public String getName() {
            return NAME;
        }

        @Override
        public String getUsage() {
            return USAGE;
        }

        @Override
        public int run(String[] args, PrintStream out) throws Exception {
            String writeCap;
            List<String> paths = this.parseArgs(args);
            if (paths.isEmpty()) {
                SetCapacity.errorln(this.getUsage());
                throw SetCapacity.invalidArgs("no arguments", new Object[0]);
            }
            HashMap<String, String> options = new HashMap<String, String>();
            this.checkIfS3BucketIsGuarded(paths);
            String readCap = this.getCommandFormat().getOptValue(S3GuardTool.READ_FLAG);
            if (StringUtils.isNotEmpty((CharSequence)readCap)) {
                Preconditions.checkArgument((Integer.parseInt(readCap) > 0 ? 1 : 0) != 0, (Object)READ_CAP_INVALID);
                S3GuardTool.println(out, "Read capacity set to %s", readCap);
                options.put("fs.s3a.s3guard.ddb.table.capacity.read", readCap);
            }
            if (StringUtils.isNotEmpty((CharSequence)(writeCap = this.getCommandFormat().getOptValue(S3GuardTool.WRITE_FLAG)))) {
                Preconditions.checkArgument((Integer.parseInt(writeCap) > 0 ? 1 : 0) != 0, (Object)WRITE_CAP_INVALID);
                S3GuardTool.println(out, "Write capacity set to %s", writeCap);
                options.put("fs.s3a.s3guard.ddb.table.capacity.write", writeCap);
            }
            try {
                this.parseDynamoDBRegion(paths);
                this.maybeInitFilesystem(paths);
            }
            catch (ExitUtil.ExitException e) {
                SetCapacity.errorln(USAGE);
                throw e;
            }
            MetadataStore store = this.initMetadataStore(false);
            store.updateParameters(options);
            SetCapacity.printStoreDiagnostics(out, store);
            return 0;
        }
    }

    static class Init
    extends S3GuardTool {
        public static final String NAME = "init";
        public static final String PURPOSE = "initialize metadata repository";
        private static final String USAGE = "init [OPTIONS] [s3a://BUCKET]\n\tinitialize metadata repository\n\nCommon options:\n  -meta URL - Metadata repository details (implementation-specific)\n\nAmazon DynamoDB-specific options:\n  -region REGION - Service region for connections\n  -read UNIT - Provisioned read throughput units\n  -write UNIT - Provisioned write through put units\n  -sse - Enable server side encryption\n  -cmk KEY - Customer managed CMK\n  -tag key=value; list of tags to tag dynamo table\n\n  URLs for Amazon DynamoDB are of the form dynamodb://TABLE_NAME.\n  Specifying both the -region option and an S3A path\n  is not supported.\nTo create a table with per-request billing, set the read and write\ncapacities to 0";

        Init(Configuration conf) {
            super(conf, S3GuardTool.SSE_FLAG);
            this.getCommandFormat().addOptionWithValue(S3GuardTool.READ_FLAG);
            this.getCommandFormat().addOptionWithValue(S3GuardTool.WRITE_FLAG);
            this.getCommandFormat().addOptionWithValue(S3GuardTool.CMK_FLAG);
            this.getCommandFormat().addOptionWithValue(S3GuardTool.TAG_FLAG);
        }

        @Override
        public String getName() {
            return NAME;
        }

        @Override
        public String getUsage() {
            return USAGE;
        }

        @Override
        public int run(String[] args, PrintStream out) throws Exception {
            String tags;
            String writeCap;
            List<String> paths = this.parseArgs(args);
            try {
                this.checkBucketNameOrDDBTableNameProvided(paths);
            }
            catch (ExitUtil.ExitException e) {
                Init.errorln(USAGE);
                throw e;
            }
            CommandFormat commands = this.getCommandFormat();
            String readCap = commands.getOptValue(S3GuardTool.READ_FLAG);
            if (readCap != null && !readCap.isEmpty()) {
                int readCapacity = Integer.parseInt(readCap);
                this.getConf().setInt("fs.s3a.s3guard.ddb.table.capacity.read", readCapacity);
            }
            if ((writeCap = commands.getOptValue(S3GuardTool.WRITE_FLAG)) != null && !writeCap.isEmpty()) {
                int writeCapacity = Integer.parseInt(writeCap);
                this.getConf().setInt("fs.s3a.s3guard.ddb.table.capacity.write", writeCapacity);
            }
            if (!paths.isEmpty()) {
                String s3path = paths.get(0);
                URI fsURI = new URI(s3path);
                Configuration bucketConf = S3AUtils.propagateBucketOptions(this.getConf(), fsURI.getHost());
                this.setConf(bucketConf);
            }
            String cmk = commands.getOptValue(S3GuardTool.CMK_FLAG);
            if (commands.getOpt(S3GuardTool.SSE_FLAG)) {
                this.getConf().setBoolean("fs.s3a.s3guard.ddb.table.sse.enabled", true);
                LOG.debug("SSE flag is passed to command {}", (Object)this.getName());
                if (!StringUtils.isEmpty((CharSequence)cmk)) {
                    if ("alias/aws/dynamodb".equals(cmk)) {
                        LOG.warn("Ignoring default DynamoDB table KMS Master Key alias/aws/dynamodb in configuration");
                    } else {
                        LOG.debug("Setting customer managed CMK {}", (Object)cmk);
                        this.getConf().set("fs.s3a.s3guard.ddb.table.sse.cmk", cmk);
                    }
                }
            } else if (!StringUtils.isEmpty((CharSequence)cmk)) {
                throw Init.invalidArgs("Option %s can only be used with option %s", S3GuardTool.CMK_FLAG, S3GuardTool.SSE_FLAG);
            }
            if ((tags = commands.getOptValue(S3GuardTool.TAG_FLAG)) != null && !tags.isEmpty()) {
                String[] stringList = tags.split(";");
                HashMap<String, String> tagsKV = new HashMap<String, String>();
                for (String kv : stringList) {
                    if (kv.isEmpty() || !kv.contains("=")) continue;
                    String[] kvSplit = kv.split("=");
                    tagsKV.put(kvSplit[0], kvSplit[1]);
                }
                for (Map.Entry entry : tagsKV.entrySet()) {
                    this.getConf().set("fs.s3a.s3guard.ddb.table.tag." + (String)entry.getKey(), (String)entry.getValue());
                }
            }
            try {
                this.parseDynamoDBRegion(paths);
            }
            catch (ExitUtil.ExitException e) {
                Init.errorln(USAGE);
                throw e;
            }
            MetadataStore store = this.initMetadataStore(true);
            Init.printStoreDiagnostics(out, store);
            return 0;
        }
    }
}

