/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.io;

import java.io.Closeable;
import java.io.EOFException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import org.apache.hadoop.HadoopIllegalArgumentException;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.LocalFileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.io.DataInputBuffer;
import org.apache.hadoop.io.DataOutputBuffer;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.io.LongWritable;
import org.apache.hadoop.io.SequenceFile;
import org.apache.hadoop.io.Writable;
import org.apache.hadoop.io.WritableComparable;
import org.apache.hadoop.io.WritableComparator;
import org.apache.hadoop.io.compress.CompressionCodec;
import org.apache.hadoop.util.Options;
import org.apache.hadoop.util.Progressable;
import org.apache.hadoop.util.ReflectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Public
@InterfaceStability.Stable
public class MapFile {
    private static final Logger LOG = LoggerFactory.getLogger(MapFile.class);
    public static final String INDEX_FILE_NAME = "index";
    public static final String DATA_FILE_NAME = "data";

    protected MapFile() {
    }

    public static void rename(FileSystem fs, String oldName, String newName) throws IOException {
        Path oldDir = new Path(oldName);
        Path newDir = new Path(newName);
        if (!fs.rename(oldDir, newDir)) {
            throw new IOException("Could not rename " + oldDir + " to " + newDir);
        }
    }

    public static void delete(FileSystem fs, String name) throws IOException {
        Path dir = new Path(name);
        Path data = new Path(dir, DATA_FILE_NAME);
        Path index = new Path(dir, INDEX_FILE_NAME);
        fs.delete(data, true);
        fs.delete(index, true);
        fs.delete(dir, true);
    }

    public static long fix(FileSystem fs, Path dir, Class<? extends Writable> keyClass, Class<? extends Writable> valueClass, boolean dryrun, Configuration conf) throws Exception {
        String dr = dryrun ? "[DRY RUN ] " : "";
        Path data = new Path(dir, DATA_FILE_NAME);
        Path index = new Path(dir, INDEX_FILE_NAME);
        int indexInterval = conf.getInt("io.map.index.interval", 128);
        if (!fs.exists(data)) {
            throw new Exception(dr + "Missing data file in " + dir + ", impossible to fix this.");
        }
        if (fs.exists(index)) {
            return -1L;
        }
        SequenceFile.Reader dataReader = new SequenceFile.Reader(conf, SequenceFile.Reader.file(data));
        if (!dataReader.getKeyClass().equals(keyClass)) {
            throw new Exception(dr + "Wrong key class in " + dir + ", expected" + keyClass.getName() + ", got " + dataReader.getKeyClass().getName());
        }
        if (!dataReader.getValueClass().equals(valueClass)) {
            throw new Exception(dr + "Wrong value class in " + dir + ", expected" + valueClass.getName() + ", got " + dataReader.getValueClass().getName());
        }
        long cnt = 0L;
        Writable key = ReflectionUtils.newInstance(keyClass, conf);
        Writable value = ReflectionUtils.newInstance(valueClass, conf);
        SequenceFile.Writer indexWriter = null;
        if (!dryrun) {
            indexWriter = SequenceFile.createWriter(conf, SequenceFile.Writer.file(index), SequenceFile.Writer.keyClass(keyClass), SequenceFile.Writer.valueClass(LongWritable.class));
        }
        try {
            long pos = 0L;
            LongWritable position = new LongWritable();
            while (dataReader.next(key, value)) {
                if (++cnt % (long)indexInterval == 0L) {
                    position.set(pos);
                    if (!dryrun) {
                        indexWriter.append(key, position);
                    }
                }
                pos = dataReader.getPosition();
            }
        }
        catch (Throwable throwable) {
            // empty catch block
        }
        dataReader.close();
        if (!dryrun) {
            indexWriter.close();
        }
        return cnt;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void main(String[] args) throws Exception {
        String usage = "Usage: MapFile inFile outFile";
        if (args.length != 2) {
            System.err.println(usage);
            System.exit(-1);
        }
        String in = args[0];
        String out = args[1];
        Configuration conf = new Configuration();
        LocalFileSystem fs = FileSystem.getLocal(conf);
        Reader reader = null;
        try {
            reader = new Reader(fs, in, conf);
            WritableComparable key = ReflectionUtils.newInstance(reader.getKeyClass().asSubclass(WritableComparable.class), conf);
            Writable value = ReflectionUtils.newInstance(reader.getValueClass().asSubclass(Writable.class), conf);
            try (Writer writer = new Writer(conf, (FileSystem)fs, out, reader.getKeyClass().asSubclass(WritableComparable.class), reader.getValueClass());){
                while (reader.next(key, value)) {
                    writer.append(key, value);
                }
            }
        }
        catch (Throwable throwable) {
            IOUtils.cleanupWithLogger(LOG, reader);
            throw throwable;
        }
        IOUtils.cleanupWithLogger(LOG, reader);
    }

    public static class Merger {
        private Configuration conf;
        private WritableComparator comparator = null;
        private Reader[] inReaders;
        private Writer outWriter;
        private Class<Writable> valueClass = null;
        private Class<WritableComparable> keyClass = null;

        public Merger(Configuration conf) throws IOException {
            this.conf = conf;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void merge(Path[] inMapFiles, boolean deleteInputs, Path outMapFile) throws IOException {
            try {
                this.open(inMapFiles, outMapFile);
                this.mergePass();
            }
            finally {
                this.close();
            }
            if (deleteInputs) {
                for (int i = 0; i < inMapFiles.length; ++i) {
                    Path path = inMapFiles[i];
                    MapFile.delete(path.getFileSystem(this.conf), path.toString());
                }
            }
        }

        private void open(Path[] inMapFiles, Path outMapFile) throws IOException {
            this.inReaders = new Reader[inMapFiles.length];
            for (int i = 0; i < inMapFiles.length; ++i) {
                Reader reader = new Reader(inMapFiles[i], this.conf, new SequenceFile.Reader.Option[0]);
                if (this.keyClass == null || this.valueClass == null) {
                    this.keyClass = reader.getKeyClass();
                    this.valueClass = reader.getValueClass();
                } else if (this.keyClass != reader.getKeyClass() || this.valueClass != reader.getValueClass()) {
                    throw new HadoopIllegalArgumentException("Input files cannot be merged as they have different Key and Value classes");
                }
                this.inReaders[i] = reader;
            }
            if (this.comparator == null) {
                Class<WritableComparable> cls = this.keyClass.asSubclass(WritableComparable.class);
                this.comparator = WritableComparator.get(cls, this.conf);
            } else if (this.comparator.getKeyClass() != this.keyClass) {
                throw new HadoopIllegalArgumentException("Input files cannot be merged as they have different Key class compared to specified comparator");
            }
            this.outWriter = new Writer(this.conf, outMapFile, Writer.keyClass(this.keyClass), Writer.valueClass(this.valueClass));
        }

        private void mergePass() throws IOException {
            WritableComparable[] keys = new WritableComparable[this.inReaders.length];
            Writable[] values = new Writable[this.inReaders.length];
            for (int i = 0; i < this.inReaders.length; ++i) {
                keys[i] = ReflectionUtils.newInstance(this.keyClass, null);
                values[i] = ReflectionUtils.newInstance(this.valueClass, null);
                if (this.inReaders[i].next(keys[i], values[i])) continue;
                keys[i] = null;
                values[i] = null;
            }
            while (true) {
                int currentEntry = -1;
                WritableComparable currentKey = null;
                Writable currentValue = null;
                for (int i = 0; i < keys.length; ++i) {
                    if (keys[i] == null || currentKey != null && this.comparator.compare(currentKey, keys[i]) <= 0) continue;
                    currentEntry = i;
                    currentKey = keys[i];
                    currentValue = values[i];
                }
                if (currentKey == null) break;
                this.outWriter.append(currentKey, currentValue);
                if (this.inReaders[currentEntry].next(keys[currentEntry], values[currentEntry])) continue;
                keys[currentEntry] = null;
                values[currentEntry] = null;
            }
        }

        private void close() throws IOException {
            for (int i = 0; i < this.inReaders.length; ++i) {
                IOUtils.closeStream(this.inReaders[i]);
                this.inReaders[i] = null;
            }
            if (this.outWriter != null) {
                this.outWriter.close();
                this.outWriter = null;
            }
        }
    }

    public static class Reader
    implements Closeable {
        private int INDEX_SKIP = 0;
        private WritableComparator comparator;
        private WritableComparable nextKey;
        private long seekPosition = -1L;
        private int seekIndex = -1;
        private long firstPosition;
        private SequenceFile.Reader data;
        private SequenceFile.Reader index;
        private boolean indexClosed = false;
        private int count = -1;
        private WritableComparable[] keys;
        private long[] positions;

        public Class<?> getKeyClass() {
            return this.data.getKeyClass();
        }

        public Class<?> getValueClass() {
            return this.data.getValueClass();
        }

        public static Option comparator(WritableComparator value) {
            return new ComparatorOption(value);
        }

        public Reader(Path dir, Configuration conf, SequenceFile.Reader.Option ... opts) throws IOException {
            ComparatorOption comparatorOption = Options.getOption(ComparatorOption.class, opts);
            WritableComparator comparator = comparatorOption == null ? null : comparatorOption.getValue();
            this.INDEX_SKIP = conf.getInt("io.map.index.skip", 0);
            this.open(dir, comparator, conf, opts);
        }

        @Deprecated
        public Reader(FileSystem fs, String dirName, Configuration conf) throws IOException {
            this(new Path(dirName), conf, new SequenceFile.Reader.Option[0]);
        }

        @Deprecated
        public Reader(FileSystem fs, String dirName, WritableComparator comparator, Configuration conf) throws IOException {
            this(new Path(dirName), conf, Reader.comparator(comparator));
        }

        protected synchronized void open(Path dir, WritableComparator comparator, Configuration conf, SequenceFile.Reader.Option ... options) throws IOException {
            Path dataFile = new Path(dir, MapFile.DATA_FILE_NAME);
            Path indexFile = new Path(dir, MapFile.INDEX_FILE_NAME);
            this.data = this.createDataFileReader(dataFile, conf, options);
            this.firstPosition = this.data.getPosition();
            if (comparator == null) {
                Class<WritableComparable> cls = this.data.getKeyClass().asSubclass(WritableComparable.class);
                this.comparator = WritableComparator.get(cls, conf);
            } else {
                this.comparator = comparator;
            }
            SequenceFile.Reader.Option[] indexOptions = Options.prependOptions(options, SequenceFile.Reader.file(indexFile));
            this.index = new SequenceFile.Reader(conf, indexOptions);
        }

        protected SequenceFile.Reader createDataFileReader(Path dataFile, Configuration conf, SequenceFile.Reader.Option ... options) throws IOException {
            SequenceFile.Reader.Option[] newOptions = Options.prependOptions(options, SequenceFile.Reader.file(dataFile));
            return new SequenceFile.Reader(conf, newOptions);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void readIndex() throws IOException {
            if (this.keys != null) {
                return;
            }
            this.count = 0;
            this.positions = new long[1024];
            try {
                WritableComparable k;
                int skip = this.INDEX_SKIP;
                LongWritable position = new LongWritable();
                WritableComparable lastKey = null;
                long lastIndex = -1L;
                ArrayList<WritableComparable> keyBuilder = new ArrayList<WritableComparable>(1024);
                while (this.index.next(k = this.comparator.newKey(), position)) {
                    if (lastKey != null && this.comparator.compare(lastKey, k) > 0) {
                        throw new IOException("key out of order: " + k + " after " + lastKey);
                    }
                    lastKey = k;
                    if (skip > 0) {
                        --skip;
                        continue;
                    }
                    skip = this.INDEX_SKIP;
                    if (position.get() == lastIndex) continue;
                    if (this.count == this.positions.length) {
                        this.positions = Arrays.copyOf(this.positions, this.positions.length * 2);
                    }
                    keyBuilder.add(k);
                    this.positions[this.count] = position.get();
                    ++this.count;
                }
                this.keys = keyBuilder.toArray(new WritableComparable[this.count]);
                this.positions = Arrays.copyOf(this.positions, this.count);
            }
            catch (EOFException e) {
                LOG.warn("Unexpected EOF reading " + this.index + " at entry #" + this.count + ".  Ignoring.");
            }
            finally {
                this.indexClosed = true;
                this.index.close();
            }
        }

        public synchronized void reset() throws IOException {
            this.data.seek(this.firstPosition);
        }

        public synchronized WritableComparable midKey() throws IOException {
            this.readIndex();
            if (this.count == 0) {
                return null;
            }
            return this.keys[(this.count - 1) / 2];
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public synchronized void finalKey(WritableComparable key) throws IOException {
            long originalPosition = this.data.getPosition();
            try {
                this.readIndex();
                if (this.count > 0) {
                    this.data.seek(this.positions[this.count - 1]);
                } else {
                    this.reset();
                }
                while (this.data.next(key)) {
                }
            }
            finally {
                this.data.seek(originalPosition);
            }
        }

        public synchronized boolean seek(WritableComparable key) throws IOException {
            return this.seekInternal(key) == 0;
        }

        private synchronized int seekInternal(WritableComparable key) throws IOException {
            return this.seekInternal(key, false);
        }

        private synchronized int seekInternal(WritableComparable key, boolean before) throws IOException {
            this.readIndex();
            if (this.seekIndex == -1 || this.seekIndex + 1 >= this.count || this.comparator.compare(key, this.keys[this.seekIndex + 1]) >= 0 || this.comparator.compare(key, this.nextKey) < 0) {
                this.seekIndex = this.binarySearch(key);
                if (this.seekIndex < 0) {
                    this.seekIndex = -this.seekIndex - 2;
                }
                this.seekPosition = this.seekIndex == -1 ? this.firstPosition : this.positions[this.seekIndex];
            }
            this.data.seek(this.seekPosition);
            if (this.nextKey == null) {
                this.nextKey = this.comparator.newKey();
            }
            long prevPosition = -1L;
            long curPosition = this.seekPosition;
            while (this.data.next(this.nextKey)) {
                int c = this.comparator.compare(key, this.nextKey);
                if (c <= 0) {
                    if (before && c != 0) {
                        if (prevPosition == -1L) {
                            this.data.seek(curPosition);
                        } else {
                            this.data.seek(prevPosition);
                            this.data.next(this.nextKey);
                            return 1;
                        }
                    }
                    return c;
                }
                if (!before) continue;
                prevPosition = curPosition;
                curPosition = this.data.getPosition();
            }
            return 1;
        }

        private int binarySearch(WritableComparable key) {
            int low = 0;
            int high = this.count - 1;
            while (low <= high) {
                int mid = low + high >>> 1;
                WritableComparable midVal = this.keys[mid];
                int cmp = this.comparator.compare(midVal, key);
                if (cmp < 0) {
                    low = mid + 1;
                    continue;
                }
                if (cmp > 0) {
                    high = mid - 1;
                    continue;
                }
                return mid;
            }
            return -(low + 1);
        }

        public synchronized boolean next(WritableComparable key, Writable val) throws IOException {
            return this.data.next(key, val);
        }

        public synchronized Writable get(WritableComparable key, Writable val) throws IOException {
            if (this.seek(key)) {
                this.data.getCurrentValue(val);
                return val;
            }
            return null;
        }

        public synchronized WritableComparable getClosest(WritableComparable key, Writable val) throws IOException {
            return this.getClosest(key, val, false);
        }

        public synchronized WritableComparable getClosest(WritableComparable key, Writable val, boolean before) throws IOException {
            int c = this.seekInternal(key, before);
            if (!before && c > 0 || before && c < 0) {
                return null;
            }
            this.data.getCurrentValue(val);
            return this.nextKey;
        }

        @Override
        public synchronized void close() throws IOException {
            if (!this.indexClosed) {
                this.index.close();
            }
            this.data.close();
        }

        static class ComparatorOption
        implements Option {
            private final WritableComparator value;

            ComparatorOption(WritableComparator value) {
                this.value = value;
            }

            WritableComparator getValue() {
                return this.value;
            }
        }

        public static interface Option
        extends SequenceFile.Reader.Option {
        }
    }

    public static class Writer
    implements Closeable {
        private SequenceFile.Writer data;
        private SequenceFile.Writer index;
        private static final String INDEX_INTERVAL = "io.map.index.interval";
        private int indexInterval = 128;
        private long size;
        private LongWritable position = new LongWritable();
        private WritableComparator comparator;
        private DataInputBuffer inBuf = new DataInputBuffer();
        private DataOutputBuffer outBuf = new DataOutputBuffer();
        private WritableComparable lastKey;
        private long lastIndexPos = -1L;
        private long lastIndexKeyCount = Long.MIN_VALUE;

        @Deprecated
        public Writer(Configuration conf, FileSystem fs, String dirName, Class<? extends WritableComparable> keyClass, Class valClass) throws IOException {
            this(conf, new Path(dirName), Writer.keyClass(keyClass), Writer.valueClass(valClass));
        }

        @Deprecated
        public Writer(Configuration conf, FileSystem fs, String dirName, Class<? extends WritableComparable> keyClass, Class valClass, SequenceFile.CompressionType compress, Progressable progress) throws IOException {
            this(conf, new Path(dirName), Writer.keyClass(keyClass), Writer.valueClass(valClass), Writer.compression(compress), Writer.progressable(progress));
        }

        @Deprecated
        public Writer(Configuration conf, FileSystem fs, String dirName, Class<? extends WritableComparable> keyClass, Class valClass, SequenceFile.CompressionType compress, CompressionCodec codec, Progressable progress) throws IOException {
            this(conf, new Path(dirName), Writer.keyClass(keyClass), Writer.valueClass(valClass), Writer.compression(compress, codec), Writer.progressable(progress));
        }

        @Deprecated
        public Writer(Configuration conf, FileSystem fs, String dirName, Class<? extends WritableComparable> keyClass, Class valClass, SequenceFile.CompressionType compress) throws IOException {
            this(conf, new Path(dirName), Writer.keyClass(keyClass), Writer.valueClass(valClass), Writer.compression(compress));
        }

        @Deprecated
        public Writer(Configuration conf, FileSystem fs, String dirName, WritableComparator comparator, Class valClass) throws IOException {
            this(conf, new Path(dirName), Writer.comparator(comparator), Writer.valueClass(valClass));
        }

        @Deprecated
        public Writer(Configuration conf, FileSystem fs, String dirName, WritableComparator comparator, Class valClass, SequenceFile.CompressionType compress) throws IOException {
            this(conf, new Path(dirName), Writer.comparator(comparator), Writer.valueClass(valClass), Writer.compression(compress));
        }

        @Deprecated
        public Writer(Configuration conf, FileSystem fs, String dirName, WritableComparator comparator, Class valClass, SequenceFile.CompressionType compress, Progressable progress) throws IOException {
            this(conf, new Path(dirName), Writer.comparator(comparator), Writer.valueClass(valClass), Writer.compression(compress), Writer.progressable(progress));
        }

        @Deprecated
        public Writer(Configuration conf, FileSystem fs, String dirName, WritableComparator comparator, Class valClass, SequenceFile.CompressionType compress, CompressionCodec codec, Progressable progress) throws IOException {
            this(conf, new Path(dirName), Writer.comparator(comparator), Writer.valueClass(valClass), Writer.compression(compress, codec), Writer.progressable(progress));
        }

        public static Option keyClass(Class<? extends WritableComparable> value) {
            return new KeyClassOption(value);
        }

        public static Option comparator(WritableComparator value) {
            return new ComparatorOption(value);
        }

        public static SequenceFile.Writer.Option valueClass(Class<?> value) {
            return SequenceFile.Writer.valueClass(value);
        }

        public static SequenceFile.Writer.Option compression(SequenceFile.CompressionType type) {
            return SequenceFile.Writer.compression(type);
        }

        public static SequenceFile.Writer.Option compression(SequenceFile.CompressionType type, CompressionCodec codec) {
            return SequenceFile.Writer.compression(type, codec);
        }

        public static SequenceFile.Writer.Option progressable(Progressable value) {
            return SequenceFile.Writer.progressable(value);
        }

        public Writer(Configuration conf, Path dirName, SequenceFile.Writer.Option ... opts) throws IOException {
            Class<Object> keyClass;
            KeyClassOption keyClassOption = Options.getOption(KeyClassOption.class, opts);
            ComparatorOption comparatorOption = Options.getOption(ComparatorOption.class, opts);
            if (keyClassOption == null == (comparatorOption == null)) {
                throw new IllegalArgumentException("key class or comparator option must be set");
            }
            this.indexInterval = conf.getInt(INDEX_INTERVAL, this.indexInterval);
            if (keyClassOption == null) {
                this.comparator = comparatorOption.getValue();
                keyClass = this.comparator.getKeyClass();
            } else {
                keyClass = keyClassOption.getValue();
                this.comparator = WritableComparator.get(keyClass, conf);
            }
            this.lastKey = this.comparator.newKey();
            FileSystem fs = dirName.getFileSystem(conf);
            if (!fs.mkdirs(dirName)) {
                throw new IOException("Mkdirs failed to create directory " + dirName);
            }
            Path dataFile = new Path(dirName, MapFile.DATA_FILE_NAME);
            Path indexFile = new Path(dirName, MapFile.INDEX_FILE_NAME);
            SequenceFile.Writer.Option[] dataOptions = Options.prependOptions(opts, SequenceFile.Writer.file(dataFile), SequenceFile.Writer.keyClass(keyClass));
            this.data = SequenceFile.createWriter(conf, dataOptions);
            SequenceFile.Writer.Option[] indexOptions = Options.prependOptions(opts, SequenceFile.Writer.file(indexFile), SequenceFile.Writer.keyClass(keyClass), SequenceFile.Writer.valueClass(LongWritable.class), SequenceFile.Writer.compression(SequenceFile.CompressionType.BLOCK));
            this.index = SequenceFile.createWriter(conf, indexOptions);
        }

        public int getIndexInterval() {
            return this.indexInterval;
        }

        public void setIndexInterval(int interval) {
            this.indexInterval = interval;
        }

        public static void setIndexInterval(Configuration conf, int interval) {
            conf.setInt(INDEX_INTERVAL, interval);
        }

        @Override
        public synchronized void close() throws IOException {
            this.data.close();
            this.index.close();
        }

        public synchronized void append(WritableComparable key, Writable val) throws IOException {
            this.checkKey(key);
            long pos = this.data.getLength();
            if (this.size >= this.lastIndexKeyCount + (long)this.indexInterval && pos > this.lastIndexPos) {
                this.position.set(pos);
                this.index.append(key, this.position);
                this.lastIndexPos = pos;
                this.lastIndexKeyCount = this.size;
            }
            this.data.append(key, val);
            ++this.size;
        }

        private void checkKey(WritableComparable key) throws IOException {
            if (this.size != 0L && this.comparator.compare(this.lastKey, key) > 0) {
                throw new IOException("key out of order: " + key + " after " + this.lastKey);
            }
            this.outBuf.reset();
            key.write(this.outBuf);
            this.inBuf.reset(this.outBuf.getData(), this.outBuf.getLength());
            this.lastKey.readFields(this.inBuf);
        }

        private static class ComparatorOption
        implements Option {
            private final WritableComparator value;

            ComparatorOption(WritableComparator value) {
                this.value = value;
            }

            WritableComparator getValue() {
                return this.value;
            }
        }

        private static class KeyClassOption
        extends Options.ClassOption
        implements Option {
            KeyClassOption(Class<?> value) {
                super(value);
            }
        }

        public static interface Option
        extends SequenceFile.Writer.Option {
        }
    }
}

