/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.crypto.key.kms;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.ConnectException;
import java.net.SocketException;
import java.net.URI;
import java.security.GeneralSecurityException;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import javax.net.ssl.SSLException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.crypto.key.KeyProvider;
import org.apache.hadoop.crypto.key.KeyProviderCryptoExtension;
import org.apache.hadoop.crypto.key.KeyProviderDelegationTokenExtension;
import org.apache.hadoop.crypto.key.kms.KMSClientProvider;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.io.retry.RetryPolicies;
import org.apache.hadoop.io.retry.RetryPolicy;
import org.apache.hadoop.security.AccessControlException;
import org.apache.hadoop.security.Credentials;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.security.token.TokenIdentifier;
import org.apache.hadoop.util.KMSUtil;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LoadBalancingKMSClientProvider
extends KeyProvider
implements KeyProviderCryptoExtension.CryptoExtension,
KeyProviderDelegationTokenExtension.DelegationTokenExtension {
    public static Logger LOG = LoggerFactory.getLogger(LoadBalancingKMSClientProvider.class);
    private final KMSClientProvider[] providers;
    private final AtomicInteger currentIdx;
    private final Text dtService;
    private final Text canonicalService;
    private RetryPolicy retryPolicy = null;

    public LoadBalancingKMSClientProvider(URI providerUri, KMSClientProvider[] providers, Configuration conf) {
        this(providerUri, providers, Time.monotonicNow(), conf);
    }

    @VisibleForTesting
    LoadBalancingKMSClientProvider(KMSClientProvider[] providers, long seed, Configuration conf) {
        this(URI.create("kms://testing"), providers, seed, conf);
    }

    private LoadBalancingKMSClientProvider(URI uri, KMSClientProvider[] providers, long seed, Configuration conf) {
        super(conf);
        this.dtService = KMSClientProvider.getDtService(uri);
        this.canonicalService = KMSUtil.getKeyProviderUri(conf) == null ? this.dtService : new Text(providers[0].getCanonicalServiceName());
        this.providers = seed != 0L ? LoadBalancingKMSClientProvider.shuffle(providers) : providers;
        for (KMSClientProvider provider : providers) {
            provider.setClientTokenProvider(this);
        }
        this.currentIdx = new AtomicInteger((int)(seed % (long)providers.length));
        int maxNumRetries = conf.getInt("hadoop.security.kms.client.failover.max.retries", providers.length);
        int sleepBaseMillis = conf.getInt("hadoop.security.kms.client.failover.sleep.base.millis", 100);
        int sleepMaxMillis = conf.getInt("hadoop.security.kms.client.failover.sleep.max.millis", 2000);
        Preconditions.checkState((maxNumRetries >= 0 ? 1 : 0) != 0);
        Preconditions.checkState((sleepBaseMillis >= 0 ? 1 : 0) != 0);
        Preconditions.checkState((sleepMaxMillis >= 0 ? 1 : 0) != 0);
        this.retryPolicy = RetryPolicies.failoverOnNetworkException(RetryPolicies.TRY_ONCE_THEN_FAIL, maxNumRetries, 0, sleepBaseMillis, sleepMaxMillis);
        LOG.debug("Created LoadBalancingKMSClientProvider for KMS url: {} with {} providers. delegation token service: {}, canonical service: {}", new Object[]{uri, providers.length, this.dtService, this.canonicalService});
    }

    @VisibleForTesting
    public KMSClientProvider[] getProviders() {
        return this.providers;
    }

    public Token<? extends TokenIdentifier> selectDelegationToken(Credentials creds) {
        Token<?> token = KMSClientProvider.selectDelegationToken(creds, this.canonicalService);
        if (token == null) {
            token = KMSClientProvider.selectDelegationToken(creds, this.dtService);
        }
        if (token == null) {
            KMSClientProvider provider;
            KMSClientProvider[] kMSClientProviderArray = this.getProviders();
            int n = kMSClientProviderArray.length;
            for (int i = 0; i < n && (token = (provider = kMSClientProviderArray[i]).selectDelegationToken(creds)) == null; ++i) {
            }
        }
        return token;
    }

    private <T> T doOp(ProviderCallable<T> op, int currPos, boolean isIdempotent) throws IOException {
        if (this.providers.length == 0) {
            throw new IOException("No providers configured !");
        }
        int numFailovers = 0;
        int i = 0;
        while (true) {
            block14: {
                KMSClientProvider provider = this.providers[(currPos + i) % this.providers.length];
                try {
                    return op.call(provider);
                }
                catch (AccessControlException ace) {
                    throw ace;
                }
                catch (IOException ioe2) {
                    ConnectException ioe2;
                    LOG.warn("KMS provider at [{}] threw an IOException: ", (Object)provider.getKMSUrl(), (Object)ioe2);
                    if (ioe2 instanceof SSLException || ioe2 instanceof SocketException) {
                        IOException cause = ioe2;
                        ioe2 = new ConnectException("SSLHandshakeException: " + cause.getMessage());
                        ioe2.initCause(cause);
                    }
                    RetryPolicy.RetryAction action = null;
                    try {
                        action = this.retryPolicy.shouldRetry(ioe2, 0, numFailovers, isIdempotent);
                    }
                    catch (Exception e) {
                        if (e instanceof IOException) {
                            throw (IOException)e;
                        }
                        throw new IOException(e);
                    }
                    if (action.action == RetryPolicy.RetryAction.RetryDecision.FAIL && numFailovers >= this.providers.length - 1) {
                        LOG.error("Aborting since the Request has failed with all KMS providers(depending on {}={} setting and numProviders={}) in the group OR the exception is not recoverable", new Object[]{"hadoop.security.kms.client.failover.max.retries", this.getConf().getInt("hadoop.security.kms.client.failover.max.retries", this.providers.length), this.providers.length});
                        throw ioe2;
                    }
                    if ((numFailovers + 1) % this.providers.length != 0) break block14;
                    try {
                        Thread.sleep(action.delayMillis);
                    }
                    catch (InterruptedException e) {
                        throw new InterruptedIOException("Thread Interrupted");
                    }
                }
                catch (Exception e) {
                    if (e instanceof RuntimeException) {
                        throw (RuntimeException)e;
                    }
                    throw new WrapperException(e);
                }
            }
            ++i;
            ++numFailovers;
        }
    }

    private int nextIdx() {
        int next;
        int current;
        while (!this.currentIdx.compareAndSet(current = this.currentIdx.get(), next = (current + 1) % this.providers.length)) {
        }
        return current;
    }

    @Override
    public String getCanonicalServiceName() {
        return this.canonicalService.toString();
    }

    @Override
    public Token<?> getDelegationToken(final String renewer) throws IOException {
        return (Token)this.doOp(new ProviderCallable<Token<?>>(){

            @Override
            public Token<?> call(KMSClientProvider provider) throws IOException {
                Token<?> token = provider.getDelegationToken(renewer);
                token.setService(LoadBalancingKMSClientProvider.this.dtService);
                LOG.debug("New token service set. Token: ({})", token);
                return token;
            }
        }, this.nextIdx(), false);
    }

    @Override
    public long renewDelegationToken(final Token<?> token) throws IOException {
        return this.doOp(new ProviderCallable<Long>(){

            @Override
            public Long call(KMSClientProvider provider) throws IOException {
                return provider.renewDelegationToken(token);
            }
        }, this.nextIdx(), false);
    }

    @Override
    public Void cancelDelegationToken(final Token<?> token) throws IOException {
        return this.doOp(new ProviderCallable<Void>(){

            @Override
            public Void call(KMSClientProvider provider) throws IOException {
                provider.cancelDelegationToken(token);
                return null;
            }
        }, this.nextIdx(), false);
    }

    @Override
    public void warmUpEncryptedKeys(String ... keyNames) throws IOException {
        Preconditions.checkArgument((this.providers.length > 0 ? 1 : 0) != 0, (Object)"No providers are configured");
        boolean success = false;
        IOException e = null;
        for (KMSClientProvider provider : this.providers) {
            try {
                provider.warmUpEncryptedKeys(keyNames);
                success = true;
            }
            catch (IOException ioe) {
                e = ioe;
                LOG.error("Error warming up keys for provider with url[" + provider.getKMSUrl() + "]", (Throwable)ioe);
            }
        }
        if (!success && e != null) {
            throw e;
        }
    }

    @Override
    public void drain(String keyName) {
        for (KMSClientProvider provider : this.providers) {
            provider.drain(keyName);
        }
    }

    @Override
    public void invalidateCache(String keyName) throws IOException {
        for (KMSClientProvider provider : this.providers) {
            provider.invalidateCache(keyName);
        }
    }

    @Override
    public KeyProviderCryptoExtension.EncryptedKeyVersion generateEncryptedKey(final String encryptionKeyName) throws IOException, GeneralSecurityException {
        try {
            return this.doOp(new ProviderCallable<KeyProviderCryptoExtension.EncryptedKeyVersion>(){

                @Override
                public KeyProviderCryptoExtension.EncryptedKeyVersion call(KMSClientProvider provider) throws IOException, GeneralSecurityException {
                    return provider.generateEncryptedKey(encryptionKeyName);
                }
            }, this.nextIdx(), true);
        }
        catch (WrapperException we) {
            if (we.getCause() instanceof GeneralSecurityException) {
                throw (GeneralSecurityException)we.getCause();
            }
            throw new IOException(we.getCause());
        }
    }

    @Override
    public KeyProvider.KeyVersion decryptEncryptedKey(final KeyProviderCryptoExtension.EncryptedKeyVersion encryptedKeyVersion) throws IOException, GeneralSecurityException {
        try {
            return this.doOp(new ProviderCallable<KeyProvider.KeyVersion>(){

                @Override
                public KeyProvider.KeyVersion call(KMSClientProvider provider) throws IOException, GeneralSecurityException {
                    return provider.decryptEncryptedKey(encryptedKeyVersion);
                }
            }, this.nextIdx(), true);
        }
        catch (WrapperException we) {
            if (we.getCause() instanceof GeneralSecurityException) {
                throw (GeneralSecurityException)we.getCause();
            }
            throw new IOException(we.getCause());
        }
    }

    @Override
    public KeyProviderCryptoExtension.EncryptedKeyVersion reencryptEncryptedKey(final KeyProviderCryptoExtension.EncryptedKeyVersion ekv) throws IOException, GeneralSecurityException {
        try {
            return this.doOp(new ProviderCallable<KeyProviderCryptoExtension.EncryptedKeyVersion>(){

                @Override
                public KeyProviderCryptoExtension.EncryptedKeyVersion call(KMSClientProvider provider) throws IOException, GeneralSecurityException {
                    return provider.reencryptEncryptedKey(ekv);
                }
            }, this.nextIdx(), true);
        }
        catch (WrapperException we) {
            if (we.getCause() instanceof GeneralSecurityException) {
                throw (GeneralSecurityException)we.getCause();
            }
            throw new IOException(we.getCause());
        }
    }

    @Override
    public void reencryptEncryptedKeys(final List<KeyProviderCryptoExtension.EncryptedKeyVersion> ekvs) throws IOException, GeneralSecurityException {
        try {
            this.doOp(new ProviderCallable<Void>(){

                @Override
                public Void call(KMSClientProvider provider) throws IOException, GeneralSecurityException {
                    provider.reencryptEncryptedKeys(ekvs);
                    return null;
                }
            }, this.nextIdx(), true);
        }
        catch (WrapperException we) {
            if (we.getCause() instanceof GeneralSecurityException) {
                throw (GeneralSecurityException)we.getCause();
            }
            throw new IOException(we.getCause());
        }
    }

    @Override
    public KeyProvider.KeyVersion getKeyVersion(final String versionName) throws IOException {
        return this.doOp(new ProviderCallable<KeyProvider.KeyVersion>(){

            @Override
            public KeyProvider.KeyVersion call(KMSClientProvider provider) throws IOException {
                return provider.getKeyVersion(versionName);
            }
        }, this.nextIdx(), true);
    }

    @Override
    public List<String> getKeys() throws IOException {
        return this.doOp(new ProviderCallable<List<String>>(){

            @Override
            public List<String> call(KMSClientProvider provider) throws IOException {
                return provider.getKeys();
            }
        }, this.nextIdx(), true);
    }

    @Override
    public KeyProvider.Metadata[] getKeysMetadata(final String ... names) throws IOException {
        return this.doOp(new ProviderCallable<KeyProvider.Metadata[]>(){

            @Override
            public KeyProvider.Metadata[] call(KMSClientProvider provider) throws IOException {
                return provider.getKeysMetadata(names);
            }
        }, this.nextIdx(), true);
    }

    @Override
    public List<KeyProvider.KeyVersion> getKeyVersions(final String name) throws IOException {
        return this.doOp(new ProviderCallable<List<KeyProvider.KeyVersion>>(){

            @Override
            public List<KeyProvider.KeyVersion> call(KMSClientProvider provider) throws IOException {
                return provider.getKeyVersions(name);
            }
        }, this.nextIdx(), true);
    }

    @Override
    public KeyProvider.KeyVersion getCurrentKey(final String name) throws IOException {
        return this.doOp(new ProviderCallable<KeyProvider.KeyVersion>(){

            @Override
            public KeyProvider.KeyVersion call(KMSClientProvider provider) throws IOException {
                return provider.getCurrentKey(name);
            }
        }, this.nextIdx(), true);
    }

    @Override
    public KeyProvider.Metadata getMetadata(final String name) throws IOException {
        return this.doOp(new ProviderCallable<KeyProvider.Metadata>(){

            @Override
            public KeyProvider.Metadata call(KMSClientProvider provider) throws IOException {
                return provider.getMetadata(name);
            }
        }, this.nextIdx(), true);
    }

    @Override
    public KeyProvider.KeyVersion createKey(final String name, final byte[] material, final KeyProvider.Options options) throws IOException {
        return this.doOp(new ProviderCallable<KeyProvider.KeyVersion>(){

            @Override
            public KeyProvider.KeyVersion call(KMSClientProvider provider) throws IOException {
                return provider.createKey(name, material, options);
            }
        }, this.nextIdx(), false);
    }

    @Override
    public KeyProvider.KeyVersion createKey(final String name, final KeyProvider.Options options) throws NoSuchAlgorithmException, IOException {
        try {
            return this.doOp(new ProviderCallable<KeyProvider.KeyVersion>(){

                @Override
                public KeyProvider.KeyVersion call(KMSClientProvider provider) throws IOException, NoSuchAlgorithmException {
                    return provider.createKey(name, options);
                }
            }, this.nextIdx(), false);
        }
        catch (WrapperException e) {
            if (e.getCause() instanceof GeneralSecurityException) {
                throw (NoSuchAlgorithmException)e.getCause();
            }
            throw new IOException(e.getCause());
        }
    }

    @Override
    public void deleteKey(final String name) throws IOException {
        this.doOp(new ProviderCallable<Void>(){

            @Override
            public Void call(KMSClientProvider provider) throws IOException {
                provider.deleteKey(name);
                return null;
            }
        }, this.nextIdx(), false);
        this.invalidateCache(name);
    }

    @Override
    public KeyProvider.KeyVersion rollNewVersion(final String name, final byte[] material) throws IOException {
        KeyProvider.KeyVersion newVersion = this.doOp(new ProviderCallable<KeyProvider.KeyVersion>(){

            @Override
            public KeyProvider.KeyVersion call(KMSClientProvider provider) throws IOException {
                return provider.rollNewVersion(name, material);
            }
        }, this.nextIdx(), false);
        this.invalidateCache(name);
        return newVersion;
    }

    @Override
    public KeyProvider.KeyVersion rollNewVersion(final String name) throws NoSuchAlgorithmException, IOException {
        try {
            KeyProvider.KeyVersion newVersion = this.doOp(new ProviderCallable<KeyProvider.KeyVersion>(){

                @Override
                public KeyProvider.KeyVersion call(KMSClientProvider provider) throws IOException, NoSuchAlgorithmException {
                    return provider.rollNewVersion(name);
                }
            }, this.nextIdx(), false);
            this.invalidateCache(name);
            return newVersion;
        }
        catch (WrapperException e) {
            if (e.getCause() instanceof GeneralSecurityException) {
                throw (NoSuchAlgorithmException)e.getCause();
            }
            throw new IOException(e.getCause());
        }
    }

    @Override
    public void close() throws IOException {
        for (KMSClientProvider provider : this.providers) {
            try {
                provider.close();
            }
            catch (IOException ioe) {
                LOG.error("Error closing provider with url[" + provider.getKMSUrl() + "]");
            }
        }
    }

    @Override
    public void flush() throws IOException {
        for (KMSClientProvider provider : this.providers) {
            try {
                provider.flush();
            }
            catch (IOException ioe) {
                LOG.error("Error flushing provider with url[" + provider.getKMSUrl() + "]");
            }
        }
    }

    private static KMSClientProvider[] shuffle(KMSClientProvider[] providers) {
        List<KMSClientProvider> list = Arrays.asList(providers);
        Collections.shuffle(list);
        return list.toArray(providers);
    }

    static class WrapperException
    extends RuntimeException {
        public WrapperException(Throwable cause) {
            super(cause);
        }
    }

    static interface ProviderCallable<T> {
        public T call(KMSClientProvider var1) throws IOException, Exception;
    }
}

