/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.backup.impl;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.backup.BackupInfo;
import org.apache.hadoop.hbase.backup.BackupType;
import org.apache.hadoop.hbase.backup.HBackupFileSystem;
import org.apache.hadoop.hbase.backup.impl.BackupException;
import org.apache.hadoop.hbase.backup.impl.BackupManifest;
import org.apache.hadoop.hbase.backup.impl.BackupSystemTable;
import org.apache.hadoop.hbase.backup.master.BackupController;
import org.apache.hadoop.hbase.backup.master.BackupLogCleaner;
import org.apache.hadoop.hbase.backup.master.LogRollMasterProcedureManager;
import org.apache.hadoop.hbase.backup.regionserver.LogRollRegionServerProcedureManager;
import org.apache.hadoop.hbase.classification.InterfaceAudience;
import org.apache.hadoop.hbase.classification.InterfaceStability;
import org.apache.hadoop.hbase.client.Admin;
import org.apache.hadoop.hbase.client.Connection;
import org.apache.hadoop.hbase.client.ConnectionFactory;
import org.apache.hadoop.hbase.util.Pair;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public class BackupManager
implements Closeable {
    private static final Log LOG = LogFactory.getLog(BackupManager.class);
    private Configuration conf = null;
    private BackupInfo backupInfo = null;
    private ExecutorService pool = null;
    private boolean backupComplete = false;
    private BackupSystemTable systemTable;
    private final Connection conn;

    public BackupManager(Configuration conf) throws IOException {
        if (!conf.getBoolean("hbase.backup.enable", false)) {
            throw new BackupException("HBase backup is not enabled. Check your hbase.backup.enable setting.");
        }
        this.conf = conf;
        this.conn = ConnectionFactory.createConnection((Configuration)conf);
        this.systemTable = new BackupSystemTable(this.conn);
    }

    protected BackupInfo getBackupContext() {
        return this.backupInfo;
    }

    public static void decorateMasterConfiguration(Configuration conf) {
        String cleanerClass;
        if (!BackupManager.isBackupEnabled(conf)) {
            return;
        }
        String plugins = conf.get("hbase.master.logcleaner.plugins");
        if (!plugins.contains(cleanerClass = BackupLogCleaner.class.getCanonicalName())) {
            conf.set("hbase.master.logcleaner.plugins", plugins + "," + cleanerClass);
        }
        String classes = conf.get("hbase.procedure.master.classes");
        String masterProcedureClass = LogRollMasterProcedureManager.class.getName();
        if (classes == null) {
            conf.set("hbase.procedure.master.classes", masterProcedureClass);
        } else if (!classes.contains(masterProcedureClass)) {
            conf.set("hbase.procedure.master.classes", classes + "," + masterProcedureClass);
        }
        classes = conf.get("hbase.coprocessor.master.classes");
        String observerClass = BackupController.class.getName();
        if (classes == null) {
            conf.set("hbase.coprocessor.master.classes", observerClass);
        } else if (!classes.contains(observerClass)) {
            conf.set("hbase.coprocessor.master.classes", classes + "," + observerClass);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Added log cleaner: " + cleanerClass));
            LOG.debug((Object)("Added master procedure manager: " + masterProcedureClass));
            LOG.debug((Object)("Added master observer: " + observerClass));
        }
    }

    public static void decorateRSConfiguration(Configuration conf) {
        if (!BackupManager.isBackupEnabled(conf)) {
            return;
        }
        String classes = conf.get("hbase.procedure.regionserver.classes");
        String regionProcedureClass = LogRollRegionServerProcedureManager.class.getName();
        if (classes == null) {
            conf.set("hbase.procedure.regionserver.classes", regionProcedureClass);
        } else if (!classes.contains(regionProcedureClass)) {
            conf.set("hbase.procedure.regionserver.classes", classes + "," + regionProcedureClass);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Added region procedure manager: " + regionProcedureClass));
        }
    }

    public static boolean isBackupEnabled(Configuration conf) {
        return conf.getBoolean("hbase.backup.enable", false);
    }

    Configuration getConf() {
        return this.conf;
    }

    @Override
    public void close() {
        if (this.pool != null) {
            this.pool.shutdownNow();
        }
        if (this.systemTable != null) {
            try {
                this.systemTable.close();
            }
            catch (Exception e) {
                LOG.error((Object)e);
            }
        }
        if (this.conn != null) {
            try {
                this.conn.close();
            }
            catch (IOException e) {
                LOG.error((Object)e);
            }
        }
    }

    public BackupInfo createBackupInfo(String backupId, BackupType type, List<TableName> tableList, String targetRootDir, int workers, long bandwidth) throws BackupException {
        if (targetRootDir == null) {
            throw new BackupException("Wrong backup request parameter: target backup root directory");
        }
        if (type == BackupType.FULL && (tableList == null || tableList.isEmpty())) {
            HTableDescriptor[] htds = null;
            try (Admin hbadmin = this.conn.getAdmin();){
                htds = hbadmin.listTables();
            }
            catch (Exception e) {
                throw new BackupException((Throwable)e);
            }
            if (htds == null) {
                throw new BackupException("No table exists for full backup of all tables.");
            }
            tableList = new ArrayList<TableName>();
            for (HTableDescriptor hTableDescriptor : htds) {
                tableList.add(hTableDescriptor.getTableName());
            }
            LOG.info((Object)("Full backup all the tables available in the cluster: " + tableList));
        }
        this.backupInfo = new BackupInfo(backupId, type, tableList.toArray(new TableName[tableList.size()]), targetRootDir);
        this.backupInfo.setBandwidth(bandwidth);
        this.backupInfo.setWorkers(workers);
        return this.backupInfo;
    }

    private String getOngoingBackupId() throws IOException {
        ArrayList sessions = this.systemTable.getBackupContexts(BackupInfo.BackupState.RUNNING);
        if (sessions.size() == 0) {
            return null;
        }
        return ((BackupInfo)sessions.get(0)).getBackupId();
    }

    public void initialize() throws IOException {
        String ongoingBackupId = this.getOngoingBackupId();
        if (ongoingBackupId != null) {
            LOG.info((Object)("There is a ongoing backup " + ongoingBackupId + ". Can not launch new backup until no ongoing backup remains."));
            throw new BackupException("There is ongoing backup.");
        }
        int nrThreads = this.conf.getInt("hbase.backup.threads.max", 1);
        ThreadFactoryBuilder builder = new ThreadFactoryBuilder();
        builder.setNameFormat("BackupHandler-%1$d");
        this.pool = new ThreadPoolExecutor(nrThreads, nrThreads, 60L, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(), builder.build());
        ((ThreadPoolExecutor)this.pool).allowCoreThreadTimeOut(true);
    }

    public void setBackupInfo(BackupInfo backupInfo) {
        this.backupInfo = backupInfo;
    }

    public ArrayList<BackupManifest.BackupImage> getAncestors(BackupInfo backupCtx) throws IOException, BackupException {
        LOG.debug((Object)("Getting the direct ancestors of the current backup " + backupCtx.getBackupId()));
        ArrayList<BackupManifest.BackupImage> ancestors = new ArrayList<BackupManifest.BackupImage>();
        if (backupCtx.getType() == BackupType.FULL) {
            LOG.debug((Object)"Current backup is a full backup, no direct ancestor for it.");
            return ancestors;
        }
        ArrayList<BackupInfo> allHistoryList = this.getBackupHistory(true);
        for (BackupInfo backup : allHistoryList) {
            BackupManifest.BackupImage image = new BackupManifest.BackupImage(backup.getBackupId(), backup.getType(), backup.getTargetRootDir(), backup.getTableNames(), backup.getStartTs(), backup.getEndTs());
            if (backup.getType().equals((Object)BackupType.FULL)) {
                if (BackupManifest.canCoverImage(ancestors, image)) continue;
                ancestors.add(image);
                continue;
            }
            if (BackupManifest.canCoverImage(ancestors, image)) {
                LOG.debug((Object)"Met the backup boundary of the current table set. The root full backup images for the current backup scope:");
                for (BackupManifest.BackupImage image1 : ancestors) {
                    LOG.debug((Object)("  BackupId: " + image1.getBackupId() + ", Backup directory: " + image1.getRootDir()));
                }
                continue;
            }
            Path logBackupPath = HBackupFileSystem.getLogBackupPath(backup.getTargetRootDir(), backup.getBackupId());
            LOG.debug((Object)("Current backup has an incremental backup ancestor, touching its image manifest in " + logBackupPath.toString() + " to construct the dependency."));
            BackupManifest lastIncrImgManifest = new BackupManifest(this.conf, logBackupPath);
            BackupManifest.BackupImage lastIncrImage = lastIncrImgManifest.getBackupImage();
            ancestors.add(lastIncrImage);
            LOG.debug((Object)"Last dependent incremental backup image information:");
            LOG.debug((Object)("  Token: " + lastIncrImage.getBackupId()));
            LOG.debug((Object)("  Backup directory: " + lastIncrImage.getRootDir()));
        }
        LOG.debug((Object)("Got " + ancestors.size() + " ancestors for the current backup."));
        return ancestors;
    }

    public ArrayList<BackupManifest.BackupImage> getAncestors(BackupInfo backupContext, TableName table) throws BackupException, IOException {
        ArrayList<BackupManifest.BackupImage> ancestors = this.getAncestors(backupContext);
        ArrayList<BackupManifest.BackupImage> tableAncestors = new ArrayList<BackupManifest.BackupImage>();
        for (BackupManifest.BackupImage image : ancestors) {
            if (!image.hasTable(table)) continue;
            tableAncestors.add(image);
            if (image.getType() != BackupType.FULL) continue;
            break;
        }
        return tableAncestors;
    }

    public void updateBackupInfo(BackupInfo context) throws IOException {
        this.systemTable.updateBackupInfo(context);
    }

    public String readBackupStartCode() throws IOException {
        return this.systemTable.readBackupStartCode(this.backupInfo.getTargetRootDir());
    }

    public void writeBackupStartCode(Long startCode) throws IOException {
        this.systemTable.writeBackupStartCode(startCode, this.backupInfo.getTargetRootDir());
    }

    public Pair<Map<TableName, Map<String, Map<String, List<Pair<String, Boolean>>>>>, List<byte[]>> readOrigBulkloadRows(List<TableName> tableList) throws IOException {
        return this.systemTable.readOrigBulkloadRows(tableList);
    }

    public void removeOrigBulkLoadedRows(List<TableName> lst, List<byte[]> rows) throws IOException {
        this.systemTable.removeOrigBulkLoadedRows(lst, rows);
    }

    public void writeBulkLoadedFiles(List<TableName> sTableList, Map<byte[], List<Path>>[] maps) throws IOException {
        this.systemTable.writeBulkLoadedFiles(sTableList, (Map[])maps, this.backupInfo.getBackupId());
    }

    public HashMap<String, Long> readRegionServerLastLogRollResult() throws IOException {
        return this.systemTable.readRegionServerLastLogRollResult(this.backupInfo.getTargetRootDir());
    }

    public ArrayList<BackupInfo> getBackupHistory() throws IOException {
        return this.systemTable.getBackupHistory();
    }

    public ArrayList<BackupInfo> getBackupHistory(boolean completed) throws IOException {
        return this.systemTable.getBackupHistory(completed);
    }

    public void writeRegionServerLogTimestamp(Set<TableName> tables, HashMap<String, Long> newTimestamps) throws IOException {
        this.systemTable.writeRegionServerLogTimestamp(tables, newTimestamps, this.backupInfo.getTargetRootDir());
    }

    public HashMap<TableName, HashMap<String, Long>> readLogTimestampMap() throws IOException {
        return this.systemTable.readLogTimestampMap(this.backupInfo.getTargetRootDir());
    }

    public Set<TableName> getIncrementalBackupTableSet() throws IOException {
        return this.systemTable.getIncrementalBackupTableSet(this.backupInfo.getTargetRootDir());
    }

    public void addIncrementalBackupTableSet(Set<TableName> tables) throws IOException {
        this.systemTable.addIncrementalBackupTableSet(tables, this.backupInfo.getTargetRootDir());
    }

    public void recordWALFiles(List<String> files) throws IOException {
        this.systemTable.addWALFiles(files, this.backupInfo.getBackupId(), this.backupInfo.getTargetRootDir());
    }

    public Iterator<BackupSystemTable.WALItem> getWALFilesFromBackupSystem() throws IOException {
        return this.systemTable.getWALFilesIterator(this.backupInfo.getTargetRootDir());
    }

    public Connection getConnection() {
        return this.conn;
    }
}

