/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.util;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.GnuParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.NamespaceDescriptor;
import org.apache.hadoop.hbase.io.FileLink;
import org.apache.hadoop.hbase.io.HFileLink;
import org.apache.hadoop.hbase.regionserver.StoreFileInfo;
import org.apache.hadoop.hbase.util.FSTableDescriptors;
import org.apache.hadoop.hbase.util.FSUtils;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;

public class HFileV1Detector
extends Configured
implements Tool {
    private FileSystem fs;
    private static final Log LOG = LogFactory.getLog(HFileV1Detector.class);
    private static final int DEFAULT_NUM_OF_THREADS = 10;
    private static final String PRE_NS_DOT_ARCHIVE = ".archive";
    private static final String PRE_NS_DOT_TMP = ".tmp";
    private int numOfThreads;
    private Path targetDirPath;
    private ExecutorService exec;
    private final Set<Path> processedTables = new HashSet<Path>();
    private final Set<Path> corruptedHFiles = Collections.newSetFromMap(new ConcurrentHashMap());
    private final Set<Path> hFileV1Set = Collections.newSetFromMap(new ConcurrentHashMap());
    private Options options = new Options();
    private Path defaultNamespace;

    public HFileV1Detector() {
        Option pathOption = new Option("p", "path", true, "Path to a table, or hbase installation");
        pathOption.setRequired(false);
        this.options.addOption(pathOption);
        Option threadOption = new Option("n", "numberOfThreads", true, "Number of threads to use while processing HFiles.");
        threadOption.setRequired(false);
        this.options.addOption(threadOption);
        this.options.addOption("h", "help", false, "Help");
    }

    private boolean parseOption(String[] args) throws ParseException, IOException {
        if (args.length == 0) {
            return true;
        }
        GnuParser parser = new GnuParser();
        CommandLine cmd = parser.parse(this.options, args);
        if (cmd.hasOption("h")) {
            HelpFormatter formatter = new HelpFormatter();
            formatter.printHelp("HFileV1Detector", this.options, true);
            System.out.println("In case no option is provided, it processes hbase.rootdir using 10 threads.");
            System.out.println("Example:");
            System.out.println(" To detect any HFileV1 in a given hbase installation '/myhbase':");
            System.out.println(" $ $HBASE_HOME/bin/hbase " + ((Object)((Object)this)).getClass().getName() + " -p /myhbase");
            System.out.println();
            return false;
        }
        if (cmd.hasOption("p")) {
            this.targetDirPath = new Path(FSUtils.getRootDir(this.getConf()), cmd.getOptionValue("p"));
        }
        try {
            if (cmd.hasOption("n")) {
                int n = Integer.parseInt(cmd.getOptionValue("n"));
                if (n < 0 || n > 100) {
                    LOG.warn((Object)"Please use a positive number <= 100 for number of threads. Continuing with default value 10");
                    return true;
                }
                this.numOfThreads = n;
            }
        }
        catch (NumberFormatException nfe) {
            LOG.error((Object)"Please select a valid number for threads");
            return false;
        }
        return true;
    }

    public int run(String[] args) throws IOException, ParseException {
        FSUtils.setFsDefault(this.getConf(), new Path(FSUtils.getRootDir(this.getConf()).toUri()));
        this.fs = FileSystem.get((Configuration)this.getConf());
        this.numOfThreads = 10;
        this.targetDirPath = FSUtils.getRootDir(this.getConf());
        if (!this.parseOption(args)) {
            System.exit(-1);
        }
        this.exec = Executors.newFixedThreadPool(this.numOfThreads);
        try {
            int n = this.processResult(this.checkForV1Files(this.targetDirPath));
            return n;
        }
        catch (Exception e) {
            LOG.error((Object)e);
        }
        finally {
            this.exec.shutdown();
            this.fs.close();
        }
        return -1;
    }

    private void setDefaultNamespaceDir() throws IOException {
        Path dataDir = new Path(FSUtils.getRootDir(this.getConf()), "data");
        this.defaultNamespace = new Path(dataDir, NamespaceDescriptor.DEFAULT_NAMESPACE_NAME_STR);
    }

    private int processResult(Set<Path> regionsWithHFileV1) {
        LOG.info((Object)"Result: \n");
        this.printSet(this.processedTables, "Tables Processed: ");
        int count = this.hFileV1Set.size();
        LOG.info((Object)("Count of HFileV1: " + count));
        if (count > 0) {
            this.printSet(this.hFileV1Set, "HFileV1:");
        }
        count = this.corruptedHFiles.size();
        LOG.info((Object)("Count of corrupted files: " + count));
        if (count > 0) {
            this.printSet(this.corruptedHFiles, "Corrupted Files: ");
        }
        count = regionsWithHFileV1.size();
        LOG.info((Object)("Count of Regions with HFileV1: " + count));
        if (count > 0) {
            this.printSet(regionsWithHFileV1, "Regions to Major Compact: ");
        }
        return this.hFileV1Set.isEmpty() && this.corruptedHFiles.isEmpty() ? 0 : 1;
    }

    private void printSet(Set<Path> result, String msg) {
        LOG.info((Object)msg);
        for (Path p : result) {
            LOG.info((Object)p);
        }
    }

    private Set<Path> checkForV1Files(Path targetDir) throws IOException {
        FileStatus[] fsStats;
        LOG.info((Object)("Target dir is: " + targetDir));
        if (!this.fs.exists(targetDir)) {
            throw new IOException("The given path does not exist: " + targetDir);
        }
        if (HFileV1Detector.isTableDir(this.fs, targetDir)) {
            this.processedTables.add(targetDir);
            return this.processTable(targetDir);
        }
        HashSet<Path> regionsWithHFileV1 = new HashSet<Path>();
        for (FileStatus fsStat : fsStats = this.fs.listStatus(targetDir)) {
            if (HFileV1Detector.isTableDir(this.fs, fsStat.getPath()) && !this.isRootTable(fsStat.getPath())) {
                this.processedTables.add(fsStat.getPath());
                regionsWithHFileV1.addAll(this.processTable(fsStat.getPath()));
                continue;
            }
            LOG.info((Object)("Ignoring path: " + fsStat.getPath()));
        }
        return regionsWithHFileV1;
    }

    private boolean isRootTable(Path path) {
        return path != null && path.toString().endsWith("-ROOT-");
    }

    private Set<Path> processTable(Path tableDir) throws IOException {
        LOG.debug((Object)("processing table: " + tableDir));
        ArrayList<Future<Path>> regionLevelResults = new ArrayList<Future<Path>>();
        HashSet<Path> regionsWithHFileV1 = new HashSet<Path>();
        FileStatus[] fsStats = this.fs.listStatus(tableDir);
        for (FileStatus fsStat : fsStats) {
            if (!HFileV1Detector.isRegionDir(this.fs, fsStat.getPath())) continue;
            regionLevelResults.add(this.processRegion(fsStat.getPath()));
        }
        for (Future future : regionLevelResults) {
            try {
                if (future.get() == null) continue;
                regionsWithHFileV1.add((Path)future.get());
            }
            catch (InterruptedException e) {
                LOG.error((Object)e);
            }
            catch (ExecutionException e) {
                LOG.error((Object)e);
            }
        }
        return regionsWithHFileV1;
    }

    private Future<Path> processRegion(final Path regionDir) {
        LOG.debug((Object)("processing region: " + regionDir));
        Callable<Path> regionCallable = new Callable<Path>(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public Path call() throws Exception {
                for (Path familyDir : FSUtils.getFamilyDirs(HFileV1Detector.this.fs, regionDir)) {
                    FileStatus[] storeFiles = FSUtils.listStatus(HFileV1Detector.this.fs, familyDir);
                    if (storeFiles == null || storeFiles.length == 0) continue;
                    for (FileStatus storeFile : storeFiles) {
                        Path storeFilePath = storeFile.getPath();
                        long lenToRead = 0L;
                        try (FSDataInputStream fsdis = null;){
                            if (StoreFileInfo.isReference(storeFilePath)) continue;
                            if (HFileLink.isHFileLink(storeFilePath)) {
                                FileLink fLink = HFileV1Detector.this.getFileLinkWithPreNSPath(storeFilePath);
                                fsdis = fLink.open(HFileV1Detector.this.fs);
                                lenToRead = fLink.getFileStatus(HFileV1Detector.this.fs).getLen();
                            } else {
                                fsdis = HFileV1Detector.this.fs.open(storeFilePath);
                                lenToRead = storeFile.getLen();
                            }
                            int majorVersion = this.computeMajorVersion(fsdis, lenToRead);
                            if (majorVersion == 1) {
                                HFileV1Detector.this.hFileV1Set.add(storeFilePath);
                                Path path = regionDir;
                                return path;
                            }
                            if (majorVersion <= 2 && majorVersion >= 1) continue;
                            throw new IllegalArgumentException("Incorrect major version: " + majorVersion);
                        }
                    }
                }
                return null;
            }

            private int computeMajorVersion(FSDataInputStream istream, long fileSize) throws IOException {
                long seekPoint = fileSize - 4L;
                if (seekPoint < 0L) {
                    throw new IllegalArgumentException("File too small, no major version found");
                }
                istream.seek(seekPoint);
                int version = istream.readInt();
                return version & 0xFFFFFF;
            }
        };
        Future<Path> f = this.exec.submit(regionCallable);
        return f;
    }

    public FileLink getFileLinkWithPreNSPath(Path storeFilePath) throws IOException {
        HFileLink link = HFileLink.buildFromHFileLinkPattern(this.getConf(), storeFilePath);
        List<Path> pathsToProcess = this.getPreNSPathsForHFileLink(link);
        pathsToProcess.addAll(Arrays.asList(link.getLocations()));
        return new FileLink(pathsToProcess);
    }

    private List<Path> getPreNSPathsForHFileLink(HFileLink fileLink) throws IOException {
        if (this.defaultNamespace == null) {
            this.setDefaultNamespaceDir();
        }
        ArrayList<Path> p = new ArrayList<Path>();
        String relativeTablePath = this.removeDefaultNSPath(fileLink.getOriginPath());
        p.add(this.getPreNSPath(PRE_NS_DOT_ARCHIVE, relativeTablePath));
        p.add(this.getPreNSPath(PRE_NS_DOT_TMP, relativeTablePath));
        p.add(this.getPreNSPath(null, relativeTablePath));
        return p;
    }

    private String removeDefaultNSPath(Path originalPath) {
        String pathStr = originalPath.toString();
        if (!pathStr.startsWith(this.defaultNamespace.toString())) {
            return pathStr;
        }
        return pathStr.substring(this.defaultNamespace.toString().length() + 1);
    }

    private Path getPreNSPath(String prefix, String relativeTablePath) throws IOException {
        String relativePath = prefix == null ? relativeTablePath : prefix + "/" + relativeTablePath;
        return new Path(FSUtils.getRootDir(this.getConf()), relativePath);
    }

    private static boolean isTableDir(FileSystem fs, Path path) throws IOException {
        if (fs.isFile(path)) {
            return false;
        }
        return FSTableDescriptors.getTableInfoPath(fs, path) != null || FSTableDescriptors.getCurrentTableInfoStatus(fs, path, false) != null || path.toString().endsWith(".META.");
    }

    private static boolean isRegionDir(FileSystem fs, Path path) throws IOException {
        if (fs.isFile(path)) {
            return false;
        }
        Path regionInfo = new Path(path, ".regioninfo");
        return fs.exists(regionInfo);
    }

    public static void main(String[] args) throws Exception {
        System.exit(ToolRunner.run((Configuration)HBaseConfiguration.create(), (Tool)new HFileV1Detector(), (String[])args));
    }

    static /* synthetic */ Set access$200(HFileV1Detector x0) {
        return x0.corruptedHFiles;
    }

    static /* synthetic */ Log access$300() {
        return LOG;
    }
}

