/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.io.hfile;

import edu.umd.cs.findbugs.annotations.SuppressWarnings;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.Future;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.io.hfile.PrefetchProtoUtils;
import org.apache.hadoop.hbase.shaded.protobuf.generated.PersistentPrefetchProtos;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public final class PrefetchExecutor {
    private static final Logger LOG = LoggerFactory.getLogger(PrefetchExecutor.class);
    private static final Map<Path, Future<?>> prefetchFutures = new ConcurrentSkipListMap();
    @SuppressWarnings(value={"MS_SHOULD_BE_FINAL"})
    private static HashMap<String, Boolean> prefetchCompleted = new HashMap();
    private static final ScheduledExecutorService prefetchExecutorPool;
    private static final int prefetchDelayMillis;
    private static final float prefetchDelayVariation;
    static String prefetchedFileListPath;
    private static final Pattern prefetchPathExclude;

    public static void request(Path path, Runnable runnable) {
        if (prefetchCompleted != null && PrefetchExecutor.isFilePrefetched(path.getName())) {
            LOG.info("File has already been prefetched before the restart, so skipping prefetch : " + path);
            return;
        }
        if (!prefetchPathExclude.matcher(path.toString()).find()) {
            long delay = prefetchDelayMillis > 0 ? (long)((float)prefetchDelayMillis * (1.0f - prefetchDelayVariation / 2.0f) + (float)prefetchDelayMillis * (prefetchDelayVariation / 2.0f) * ThreadLocalRandom.current().nextFloat()) : 0L;
            try {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Prefetch requested for " + path + ", delay=" + delay + " ms");
                }
                prefetchFutures.put(path, prefetchExecutorPool.schedule(runnable, delay, TimeUnit.MILLISECONDS));
            }
            catch (RejectedExecutionException e) {
                prefetchFutures.remove(path);
                LOG.warn("Prefetch request rejected for " + path);
            }
        }
    }

    public static void complete(Path path) {
        prefetchFutures.remove(path);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Prefetch completed for " + path);
        }
        prefetchCompleted.put(path.getName(), true);
    }

    public static void cancel(Path path) {
        Future<?> future = prefetchFutures.get(path);
        if (future != null) {
            future.cancel(true);
            prefetchFutures.remove(path);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Prefetch cancelled for " + path);
            }
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Removing filename from the prefetched persistence list: {}", (Object)path.getName());
        }
        PrefetchExecutor.removePrefetchedFileWhileEvict(path.getName());
    }

    public static boolean isCompleted(Path path) {
        Future<?> future = prefetchFutures.get(path);
        if (future != null) {
            return future.isDone();
        }
        return true;
    }

    public static void persistToFile(String path) throws IOException {
        prefetchedFileListPath = path;
        if (prefetchedFileListPath == null) {
            LOG.info("Exception while persisting prefetch!");
            throw new IOException("Error persisting prefetched HFiles set!");
        }
        if (!prefetchCompleted.isEmpty()) {
            try (FileOutputStream fos = new FileOutputStream(prefetchedFileListPath, false);){
                PrefetchProtoUtils.toPB(prefetchCompleted).writeDelimitedTo((OutputStream)fos);
            }
        }
    }

    public static void retrieveFromFile(String path) throws IOException {
        prefetchedFileListPath = path;
        File prefetchPersistenceFile = new File(prefetchedFileListPath);
        if (!prefetchPersistenceFile.exists()) {
            LOG.warn("Prefetch persistence file does not exist!");
            return;
        }
        LOG.info("Retrieving from prefetch persistence file " + path);
        assert (prefetchedFileListPath != null);
        try (FileInputStream fis = PrefetchExecutor.deleteFileOnClose(prefetchPersistenceFile);){
            PersistentPrefetchProtos.PrefetchedHfileName proto = PersistentPrefetchProtos.PrefetchedHfileName.parseDelimitedFrom((InputStream)fis);
            Map protoPrefetchedFilesMap = proto.getPrefetchedFilesMap();
            prefetchCompleted.putAll(protoPrefetchedFilesMap);
        }
    }

    private static FileInputStream deleteFileOnClose(File file) throws IOException {
        return (new FileInputStream(file){
            private File myFile;

            private FileInputStream init(File file) {
                this.myFile = file;
                return this;
            }

            @Override
            public void close() throws IOException {
                if (this.myFile == null) {
                    return;
                }
                super.close();
                if (!this.myFile.delete()) {
                    throw new IOException("Failed deleting persistence file " + this.myFile.getAbsolutePath());
                }
                this.myFile = null;
            }
        }).init(file);
    }

    public static void removePrefetchedFileWhileEvict(String hfileName) {
        prefetchCompleted.remove(hfileName);
    }

    public static boolean isFilePrefetched(String hfileName) {
        return prefetchCompleted.containsKey(hfileName);
    }

    private PrefetchExecutor() {
    }

    static {
        Configuration conf = HBaseConfiguration.create();
        prefetchDelayMillis = conf.getInt("hbase.hfile.prefetch.delay", 1000);
        prefetchDelayVariation = conf.getFloat("hbase.hfile.prefetch.delay.variation", 0.2f);
        int prefetchThreads = conf.getInt("hbase.hfile.thread.prefetch", 4);
        prefetchExecutorPool = new ScheduledThreadPoolExecutor(prefetchThreads, new ThreadFactory(){

            @Override
            public Thread newThread(Runnable r) {
                String name = "hfile-prefetch-" + System.currentTimeMillis();
                Thread t = new Thread(r, name);
                t.setDaemon(true);
                return t;
            }
        });
        prefetchPathExclude = Pattern.compile("(/" + ".tmp".replace(".", "\\.") + '/' + ")|(" + '/' + "compaction.dir".replace(".", "\\.") + '/' + ")");
    }
}

