/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hive.jdbc.saml;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.awt.Desktop;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.net.InetAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketTimeoutException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import org.apache.hive.jdbc.Utils;
import org.apache.hive.jdbc.saml.IJdbcBrowserClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HiveJdbcBrowserClient
implements IJdbcBrowserClient {
    private static final Logger LOG = LoggerFactory.getLogger(HiveJdbcBrowserClient.class);
    @VisibleForTesting
    public static final String TIMEOUT_ERROR_MSG = "Timed out while waiting for server response";
    private final ServerSocket serverSocket;
    private IJdbcBrowserClient.HiveJdbcBrowserServerResponse serverResponse;
    protected IJdbcBrowserClient.JdbcBrowserClientContext clientContext;
    private static final int DEFAULT_SOCKET_TIMEOUT_SECS = 120;
    private final ExecutorService serverResponseThread = Executors.newSingleThreadExecutor(new ThreadFactoryBuilder().setNameFormat("Hive-Jdbc-Browser-Client-%d").setDaemon(true).build());

    HiveJdbcBrowserClient(Utils.JdbcConnectionParams connectionParams) throws IJdbcBrowserClient.HiveJdbcBrowserException {
        this.serverSocket = this.getServerSocket(connectionParams.getSessionVars());
    }

    private ServerSocket getServerSocket(Map<String, String> sessionConf) throws IJdbcBrowserClient.HiveJdbcBrowserException {
        ServerSocket serverSocket;
        int port = Integer.parseInt(sessionConf.getOrDefault("browserResponsePort", "0"));
        int timeout = Integer.parseInt(sessionConf.getOrDefault("browserResponseTimeout", String.valueOf(120)));
        try {
            serverSocket = new ServerSocket(port, 0, InetAddress.getByName("127.0.0.1"));
            LOG.debug("Browser response timeout is set to {} seconds", (Object)timeout);
            serverSocket.setSoTimeout(timeout * 1000);
        }
        catch (IOException e) {
            throw new IJdbcBrowserClient.HiveJdbcBrowserException("Unable to bind to the localhost");
        }
        return serverSocket;
    }

    @Override
    public Integer getPort() {
        return this.serverSocket.getLocalPort();
    }

    @Override
    public void close() throws IOException {
        if (this.serverSocket != null) {
            this.serverSocket.close();
        }
    }

    @Override
    public void init(IJdbcBrowserClient.JdbcBrowserClientContext clientContext) {
        this.reset();
        this.clientContext = clientContext;
        LOG.trace("Initialized the JDBCBrowser client with URL {}", (Object)clientContext.getSsoUri());
    }

    private void reset() {
        this.serverResponse = null;
        this.clientContext = null;
    }

    @Override
    public void doBrowserSSO() throws IJdbcBrowserClient.HiveJdbcBrowserException {
        Future<Void> serverResponseHandle = this.waitAsyncForServerResponse();
        this.logDebugInfoUri(this.clientContext.getSsoUri());
        this.openBrowserWindow();
        try {
            serverResponseHandle.get();
        }
        catch (InterruptedException | ExecutionException e) {
            throw new IJdbcBrowserClient.HiveJdbcBrowserException(e);
        }
    }

    private void logDebugInfoUri(URI ssoURI) {
        Map<Object, Object> uriParams = new HashMap();
        try {
            uriParams = this.getQueryParams(ssoURI);
        }
        catch (IJdbcBrowserClient.HiveJdbcBrowserException e) {
            LOG.debug("Could get query params of the SSO URI", (Throwable)e);
        }
        LOG.debug("Initializing browser SSO request to URI. Relay state is {}", uriParams.get("RelayState"));
    }

    private Map<String, String> getQueryParams(URI ssoUri) throws IJdbcBrowserClient.HiveJdbcBrowserException {
        String[] params;
        String decodedUrl;
        try {
            decodedUrl = URLDecoder.decode(ssoUri.toString(), StandardCharsets.UTF_8.name());
        }
        catch (UnsupportedEncodingException e) {
            throw new IJdbcBrowserClient.HiveJdbcBrowserException(e);
        }
        try {
            params = new URI(decodedUrl).getQuery().split("&");
        }
        catch (URISyntaxException e) {
            throw new IJdbcBrowserClient.HiveJdbcBrowserException(e);
        }
        HashMap<String, String> paramMap = new HashMap<String, String>();
        for (String param : params) {
            String key = param.split("=")[0];
            String val = param.split("=")[1];
            paramMap.put(key, val);
        }
        return paramMap;
    }

    @VisibleForTesting
    protected void openBrowserWindow() throws IJdbcBrowserClient.HiveJdbcBrowserException {
        URI ssoUri = this.clientContext.getSsoUri();
        Preconditions.checkNotNull((Object)ssoUri, (Object)"SSO Url is null");
        try {
            if (Desktop.isDesktopSupported() && Desktop.getDesktop().isSupported(Desktop.Action.BROWSE)) {
                Desktop.getDesktop().browse(ssoUri);
            } else {
                LOG.debug("Desktop mode is not supported. Attempting to use OS commands to open the default browser");
                OsType os = this.getOperatingSystem();
                switch (os) {
                    case WINDOWS: {
                        Runtime.getRuntime().exec("rundll32 url.dll,FileProtocolHandler " + ssoUri.toString());
                        break;
                    }
                    case MAC: {
                        Runtime.getRuntime().exec("open " + ssoUri.toString());
                        break;
                    }
                    case LINUX: {
                        Runtime.getRuntime().exec("xdg-open " + ssoUri.toString());
                        break;
                    }
                    case UNKNOWN: {
                        throw new IJdbcBrowserClient.HiveJdbcBrowserException("Unknown operating system " + System.getProperty("os.name"));
                    }
                }
            }
        }
        catch (IOException e) {
            throw new IJdbcBrowserClient.HiveJdbcBrowserException("Unable to open browser to execute SSO", e);
        }
    }

    private Future<Void> waitAsyncForServerResponse() {
        return this.serverResponseThread.submit(() -> {
            Socket socket;
            try {
                LOG.debug("Waiting for a server response on port {} with a timeout of {} ms", (Object)this.serverSocket.getLocalPort(), (Object)this.serverSocket.getSoTimeout());
                socket = this.serverSocket.accept();
            }
            catch (SocketTimeoutException timeoutException) {
                throw new IJdbcBrowserClient.HiveJdbcBrowserException(TIMEOUT_ERROR_MSG, timeoutException);
            }
            catch (IOException e) {
                throw new IJdbcBrowserClient.HiveJdbcBrowserException("Unexpected error while listening on port " + this.serverSocket.getLocalPort() + " for server response", e);
            }
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(socket.getInputStream(), StandardCharsets.UTF_8));){
                String[] lines;
                char[] buffer = new char[16384];
                int len = reader.read(buffer);
                String response = String.valueOf(buffer, 0, len);
                for (String line : lines = response.split("\r\n")) {
                    if (Strings.isNullOrEmpty((String)line)) continue;
                    if (line.contains("token=")) {
                        this.serverResponse = new IJdbcBrowserClient.HiveJdbcBrowserServerResponse(line);
                        this.sendBrowserMsg(socket, this.serverResponse.isSuccessful());
                        continue;
                    }
                    LOG.trace("Skipping line {} from server response", (Object)line);
                }
                if (this.serverResponse == null) {
                    throw new IJdbcBrowserClient.HiveJdbcBrowserException("Could not parse the response from server.");
                }
            }
            catch (IOException e) {
                throw new IJdbcBrowserClient.HiveJdbcBrowserException("Unexpected exception while processing server response ", e);
            }
            return null;
        });
    }

    @Override
    public IJdbcBrowserClient.HiveJdbcBrowserServerResponse getServerResponse() {
        return this.serverResponse;
    }

    @Override
    public String getClientIdentifier() {
        if (this.clientContext == null) {
            return null;
        }
        return this.clientContext.getClientIdentifier();
    }

    private void sendBrowserMsg(Socket socket, boolean success) throws IOException {
        PrintWriter out = new PrintWriter(socket.getOutputStream(), true);
        ArrayList<String> content = new ArrayList<String>();
        content.add("HTTP/1.0 200 OK");
        content.add("Content-Type: text/html");
        String responseText = success ? "<!DOCTYPE html><html><head><meta charset=\"UTF-8\"/><title>SAML Response Received</title></head><body>Successfully authenticated. You may close this window.</body></html>" : "<!DOCTYPE html><html><head><meta charset=\"UTF-8\"/><title>SAML Response Received</title></head><body>Authentication failed. Please check server logs for details. You may close this window.</body></html>";
        content.add(String.format("Content-Length: %s", responseText.length()));
        content.add("");
        content.add(responseText);
        for (int i = 0; i < content.size(); ++i) {
            if (i > 0) {
                out.print("\r\n");
            }
            out.print((String)content.get(i));
        }
        out.flush();
    }

    public OsType getMatchingOs(String osName) {
        if ((osName = osName.toLowerCase()).contains("win")) {
            return OsType.WINDOWS;
        }
        if (osName.contains("mac")) {
            return OsType.MAC;
        }
        if (osName.contains("nix") || osName.contains("nux") || osName.contains("aix")) {
            return OsType.LINUX;
        }
        return OsType.UNKNOWN;
    }

    private OsType getOperatingSystem() {
        String osName = System.getProperty("os.name");
        Preconditions.checkNotNull((Object)osName, (Object)"os.name is null");
        return this.getMatchingOs(osName);
    }

    private static enum OsType {
        WINDOWS,
        MAC,
        LINUX,
        UNKNOWN;

    }
}

