/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hive.metastore;

import com.google.common.collect.Interner;
import com.google.common.collect.Interners;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.IOException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hive.metastore.CheckResult;
import org.apache.hadoop.hive.metastore.IMetaStoreClient;
import org.apache.hadoop.hive.metastore.PartitionIterable;
import org.apache.hadoop.hive.metastore.Warehouse;
import org.apache.hadoop.hive.metastore.api.FieldSchema;
import org.apache.hadoop.hive.metastore.api.MetaException;
import org.apache.hadoop.hive.metastore.api.Partition;
import org.apache.hadoop.hive.metastore.api.Table;
import org.apache.hadoop.hive.metastore.api.UnknownDBException;
import org.apache.hadoop.hive.metastore.conf.MetastoreConf;
import org.apache.hadoop.hive.metastore.utils.FileUtils;
import org.apache.hadoop.hive.metastore.utils.MetaStoreUtils;
import org.apache.hadoop.hive.metastore.utils.MetastoreException;
import org.apache.thrift.TException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HiveMetaStoreChecker {
    public static final Logger LOG = LoggerFactory.getLogger(HiveMetaStoreChecker.class);
    private final IMetaStoreClient msc;
    private final Configuration conf;
    private final long partitionExpirySeconds;
    private final Interner<Path> pathInterner = Interners.newStrongInterner();

    public HiveMetaStoreChecker(IMetaStoreClient msc, Configuration conf) {
        this(msc, conf, -1L);
    }

    public HiveMetaStoreChecker(IMetaStoreClient msc, Configuration conf, long partitionExpirySeconds) {
        this.msc = msc;
        this.conf = conf;
        this.partitionExpirySeconds = partitionExpirySeconds;
    }

    public IMetaStoreClient getMsc() {
        return this.msc;
    }

    public void checkMetastore(String catName, String dbName, String tableName, List<? extends Map<String, String>> partitions, CheckResult result) throws MetastoreException, IOException {
        if (dbName == null || "".equalsIgnoreCase(dbName)) {
            dbName = "default";
        }
        try {
            if (tableName == null || "".equals(tableName)) {
                ArrayList<String> tables = new ArrayList();
                try {
                    tables = this.getMsc().getTables(catName, dbName, ".*");
                }
                catch (UnknownDBException unknownDBException) {
                    // empty catch block
                }
                for (String currentTableName : tables) {
                    this.checkTable(catName, dbName, currentTableName, null, result);
                }
                this.findUnknownTables(catName, dbName, tables, result);
            } else if (partitions == null || partitions.isEmpty()) {
                this.checkTable(catName, dbName, tableName, null, result);
            } else {
                this.checkTable(catName, dbName, tableName, partitions, result);
            }
            LOG.info("Number of partitionsNotInMs=" + result.getPartitionsNotInMs() + ", partitionsNotOnFs=" + result.getPartitionsNotOnFs() + ", tablesNotInMs=" + result.getTablesNotInMs() + ", tablesNotOnFs=" + result.getTablesNotOnFs() + ", expiredPartitions=" + result.getExpiredPartitions());
        }
        catch (TException e) {
            throw new MetastoreException(e);
        }
    }

    void findUnknownTables(String catName, String dbName, List<String> tables, CheckResult result) throws IOException, MetaException, TException {
        HashSet<Path> dbPaths = new HashSet<Path>();
        HashSet<String> tableNames = new HashSet<String>(tables);
        for (String tableName : tables) {
            Path tablePath;
            Table table = this.getMsc().getTable(catName, dbName, tableName);
            String isExternal = table.getParameters().get("EXTERNAL");
            if ("TRUE".equalsIgnoreCase(isExternal) || (tablePath = MetaStoreUtils.getPath(table)) == null) continue;
            dbPaths.add(tablePath.getParent());
        }
        for (Path dbPath : dbPaths) {
            FileStatus[] statuses;
            FileSystem fs = dbPath.getFileSystem(this.conf);
            for (FileStatus status : statuses = fs.listStatus(dbPath, FileUtils.HIDDEN_FILES_PATH_FILTER)) {
                if (!status.isDir() || tableNames.contains(status.getPath().getName())) continue;
                result.getTablesNotInMs().add(status.getPath().getName());
            }
        }
    }

    void checkTable(String catName, String dbName, String tableName, List<? extends Map<String, String>> partitions, CheckResult result) throws MetaException, IOException, MetastoreException {
        PartitionIterable parts;
        Table table;
        try {
            table = this.getMsc().getTable(catName, dbName, tableName);
        }
        catch (TException e) {
            result.getTablesNotInMs().add(tableName);
            return;
        }
        boolean findUnknownPartitions = true;
        if (MetaStoreUtils.isPartitioned(table)) {
            if (partitions == null || partitions.isEmpty()) {
                int batchSize = MetastoreConf.getIntVar(this.conf, MetastoreConf.ConfVars.BATCH_RETRIEVE_MAX);
                if (batchSize > 0) {
                    parts = new PartitionIterable(this.getMsc(), table, batchSize);
                } else {
                    List<Partition> loadedPartitions = MetaStoreUtils.getAllPartitionsOf(this.getMsc(), table);
                    parts = new PartitionIterable(loadedPartitions);
                }
            } else {
                findUnknownPartitions = false;
                ArrayList<Partition> loadedPartitions = new ArrayList<Partition>();
                for (Map<String, String> map : partitions) {
                    Partition part = MetaStoreUtils.getPartition(this.getMsc(), table, map);
                    if (part == null) {
                        CheckResult.PartitionResult pr = new CheckResult.PartitionResult();
                        pr.setTableName(tableName);
                        pr.setPartitionName(Warehouse.makePartPath(map));
                        result.getPartitionsNotInMs().add(pr);
                        continue;
                    }
                    loadedPartitions.add(part);
                }
                parts = new PartitionIterable(loadedPartitions);
            }
        } else {
            parts = new PartitionIterable(Collections.emptyList());
        }
        this.checkTable(table, parts, findUnknownPartitions, result);
    }

    void checkTable(Table table, PartitionIterable parts, boolean findUnknownPartitions, CheckResult result) throws IOException, MetastoreException {
        Path tablePath = MetaStoreUtils.getPath(table);
        if (tablePath == null) {
            return;
        }
        FileSystem fs = tablePath.getFileSystem(this.conf);
        if (!fs.exists(tablePath)) {
            result.getTablesNotOnFs().add(table.getTableName());
            return;
        }
        HashSet<Path> partPaths = new HashSet<Path>();
        for (Partition partition : parts) {
            long createdTime;
            long currentEpochSecs;
            long partitionAgeSeconds;
            Path partPath;
            if (partition == null || (partPath = MetaStoreUtils.getDataLocation(table, partition)) == null) continue;
            fs = partPath.getFileSystem(this.conf);
            CheckResult.PartitionResult prFromMetastore = new CheckResult.PartitionResult();
            prFromMetastore.setPartitionName(MetaStoreUtils.getPartitionName(table, partition));
            prFromMetastore.setTableName(partition.getTableName());
            if (!fs.exists(partPath)) {
                result.getPartitionsNotOnFs().add(prFromMetastore);
            } else {
                result.getCorrectPartitions().add(prFromMetastore);
            }
            if (this.partitionExpirySeconds > 0L && (partitionAgeSeconds = (currentEpochSecs = Instant.now().getEpochSecond()) - (createdTime = (long)partition.getCreateTime())) > this.partitionExpirySeconds) {
                CheckResult.PartitionResult pr = new CheckResult.PartitionResult();
                pr.setPartitionName(MetaStoreUtils.getPartitionName(table, partition));
                pr.setTableName(partition.getTableName());
                result.getExpiredPartitions().add(pr);
                if (LOG.isDebugEnabled()) {
                    LOG.debug("{}.{}.{}.{} expired. createdAt: {} current: {} age: {}s expiry: {}s", new Object[]{partition.getCatName(), partition.getDbName(), partition.getTableName(), pr.getPartitionName(), createdTime, currentEpochSecs, partitionAgeSeconds, this.partitionExpirySeconds});
                }
            }
            for (int i = 0; i < MetaStoreUtils.getPartitionSpec(table, partition).size(); ++i) {
                Path qualifiedPath = partPath.makeQualified(fs);
                this.pathInterner.intern((Object)qualifiedPath);
                partPaths.add(qualifiedPath);
                partPath = partPath.getParent();
            }
        }
        if (findUnknownPartitions) {
            this.findUnknownPartitions(table, partPaths, result);
        }
    }

    void findUnknownPartitions(Table table, Set<Path> partPaths, CheckResult result) throws IOException, MetastoreException {
        Path tablePath = MetaStoreUtils.getPath(table);
        if (tablePath == null) {
            return;
        }
        HashSet<Path> allPartDirs = new HashSet<Path>();
        this.checkPartitionDirs(tablePath, allPartDirs, Collections.unmodifiableList(MetaStoreUtils.getPartColNames(table)));
        allPartDirs.remove(tablePath);
        allPartDirs.removeAll(partPaths);
        HashSet partColNames = Sets.newHashSet();
        for (FieldSchema fSchema : MetaStoreUtils.getPartCols(table)) {
            partColNames.add(fSchema.getName());
        }
        for (Path partPath : allPartDirs) {
            FileSystem fs = partPath.getFileSystem(this.conf);
            String partitionName = MetaStoreUtils.getPartitionName(fs.makeQualified(tablePath), partPath, partColNames);
            LOG.debug("PartitionName: " + partitionName);
            if (partitionName == null) continue;
            CheckResult.PartitionResult pr = new CheckResult.PartitionResult();
            pr.setPartitionName(partitionName);
            pr.setTableName(table.getTableName());
            pr.setPath(partPath);
            if (result.getCorrectPartitions().contains(pr)) {
                String msg = "The partition '" + pr.toString() + "' already exists for table" + table.getTableName();
                throw new MetastoreException(msg);
            }
            if (result.getPartitionsNotInMs().contains(pr)) {
                String msg = "Found two paths for same partition '" + pr.toString() + "' for table " + table.getTableName();
                throw new MetastoreException(msg);
            }
            result.getPartitionsNotInMs().add(pr);
            if (!result.getPartitionsNotOnFs().contains(pr)) continue;
            result.getPartitionsNotOnFs().remove(pr);
        }
        LOG.debug("Number of partitions not in metastore : " + result.getPartitionsNotInMs().size());
    }

    private void checkPartitionDirs(Path basePath, Set<Path> allDirs, List<String> partColNames) throws IOException, MetastoreException {
        Object executor;
        int poolSize = MetastoreConf.getIntVar(this.conf, MetastoreConf.ConfVars.FS_HANDLER_THREADS_COUNT);
        if (poolSize <= 1) {
            LOG.debug("Using single-threaded version of MSCK-GetPaths");
            executor = MoreExecutors.newDirectExecutorService();
        } else {
            LOG.debug("Using multi-threaded version of MSCK-GetPaths with number of threads " + poolSize);
            ThreadFactory threadFactory = new ThreadFactoryBuilder().setDaemon(true).setNameFormat("MSCK-GetPaths-%d").build();
            executor = Executors.newFixedThreadPool(poolSize, threadFactory);
        }
        this.checkPartitionDirs((ExecutorService)executor, basePath, allDirs, basePath.getFileSystem(this.conf), partColNames);
        executor.shutdown();
    }

    private void checkPartitionDirs(ExecutorService executor, Path basePath, Set<Path> result, FileSystem fs, List<String> partColNames) throws MetastoreException {
        try {
            LinkedList<Future<Path>> futures = new LinkedList<Future<Path>>();
            ConcurrentLinkedQueue<PathDepthInfo> nextLevel = new ConcurrentLinkedQueue<PathDepthInfo>();
            nextLevel.add(new PathDepthInfo(basePath, 0));
            while (!nextLevel.isEmpty()) {
                ConcurrentLinkedQueue tempQueue = new ConcurrentLinkedQueue();
                while (!nextLevel.isEmpty()) {
                    futures.add(executor.submit(new PathDepthInfoCallable((PathDepthInfo)nextLevel.poll(), partColNames, fs, tempQueue)));
                }
                while (!futures.isEmpty()) {
                    Path p = (Path)((Future)futures.poll()).get();
                    if (p == null) continue;
                    result.add(p);
                }
                nextLevel = tempQueue;
            }
        }
        catch (InterruptedException | ExecutionException e) {
            LOG.error(e.getMessage());
            executor.shutdownNow();
            throw new MetastoreException(e.getCause());
        }
    }

    private static class PathDepthInfo {
        private final Path p;
        private final int depth;

        PathDepthInfo(Path p, int depth) {
            this.p = p;
            this.depth = depth;
        }
    }

    private final class PathDepthInfoCallable
    implements Callable<Path> {
        private final List<String> partColNames;
        private final FileSystem fs;
        private final ConcurrentLinkedQueue<PathDepthInfo> pendingPaths;
        private final boolean throwException;
        private final PathDepthInfo pd;

        private PathDepthInfoCallable(PathDepthInfo pd, List<String> partColNames, FileSystem fs, ConcurrentLinkedQueue<PathDepthInfo> basePaths) {
            this.partColNames = partColNames;
            this.pd = pd;
            this.fs = fs;
            this.pendingPaths = basePaths;
            this.throwException = "throw".equals(MetastoreConf.getVar(HiveMetaStoreChecker.this.conf, MetastoreConf.ConfVars.MSCK_PATH_VALIDATION));
        }

        @Override
        public Path call() throws Exception {
            return this.processPathDepthInfo(this.pd);
        }

        private Path processPathDepthInfo(PathDepthInfo pd) throws IOException, MetastoreException {
            Path currentPath = pd.p;
            int currentDepth = pd.depth;
            FileStatus[] fileStatuses = this.fs.listStatus(currentPath, FileUtils.HIDDEN_FILES_PATH_FILTER);
            if (fileStatuses.length == 0 && currentDepth > 0 && currentDepth < this.partColNames.size()) {
                this.logOrThrowExceptionWithMsg("MSCK is missing partition columns under " + currentPath.toString());
            } else {
                for (FileStatus fileStatus : fileStatuses) {
                    if (!fileStatus.isDirectory() && currentDepth < this.partColNames.size()) {
                        this.logOrThrowExceptionWithMsg("MSCK finds a file rather than a directory when it searches for " + fileStatus.getPath().toString());
                        continue;
                    }
                    if (!fileStatus.isDirectory() || currentDepth >= this.partColNames.size()) continue;
                    Path nextPath = fileStatus.getPath();
                    String[] parts = nextPath.getName().split("=");
                    if (parts.length != 2) {
                        this.logOrThrowExceptionWithMsg("Invalid partition name " + nextPath);
                        continue;
                    }
                    if (!parts[0].equalsIgnoreCase(this.partColNames.get(currentDepth))) {
                        this.logOrThrowExceptionWithMsg("Unexpected partition key " + parts[0] + " found at " + nextPath);
                        continue;
                    }
                    this.pendingPaths.add(new PathDepthInfo(nextPath, currentDepth + 1));
                }
                if (currentDepth == this.partColNames.size()) {
                    return currentPath;
                }
            }
            return null;
        }

        private void logOrThrowExceptionWithMsg(String msg) throws MetastoreException {
            if (this.throwException) {
                throw new MetastoreException(msg);
            }
            LOG.warn(msg);
        }
    }
}

