/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hive.metastore.leader;

import java.net.InetAddress;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hive.common.TableName;
import org.apache.hadoop.hive.metastore.LockComponentBuilder;
import org.apache.hadoop.hive.metastore.api.CheckLockRequest;
import org.apache.hadoop.hive.metastore.api.DataOperationType;
import org.apache.hadoop.hive.metastore.api.HeartbeatRequest;
import org.apache.hadoop.hive.metastore.api.LockComponent;
import org.apache.hadoop.hive.metastore.api.LockRequest;
import org.apache.hadoop.hive.metastore.api.LockResponse;
import org.apache.hadoop.hive.metastore.api.LockState;
import org.apache.hadoop.hive.metastore.api.MetaException;
import org.apache.hadoop.hive.metastore.api.NoSuchLockException;
import org.apache.hadoop.hive.metastore.api.NoSuchTxnException;
import org.apache.hadoop.hive.metastore.api.TxnAbortedException;
import org.apache.hadoop.hive.metastore.api.TxnOpenException;
import org.apache.hadoop.hive.metastore.api.UnlockRequest;
import org.apache.hadoop.hive.metastore.conf.MetastoreConf;
import org.apache.hadoop.hive.metastore.leader.LeaderElection;
import org.apache.hadoop.hive.metastore.leader.LeaderException;
import org.apache.hadoop.hive.metastore.txn.TxnStore;
import org.apache.hadoop.hive.metastore.txn.TxnUtils;
import org.apache.hadoop.hive.metastore.utils.SecurityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LeaseLeaderElection
implements LeaderElection<TableName> {
    private static final Logger LOG = LoggerFactory.getLogger(LeaseLeaderElection.class);
    private static final AtomicLong ID = new AtomicLong();
    private volatile boolean isLeader;
    private TxnStore store;
    private long nextSleep = 50L;
    private LeaseWatcher heartbeater;
    private LeaseWatcher nonLeaderWatcher;
    private volatile long lockId = -1L;
    private List<LeaderElection.LeadershipStateListener> listeners = new ArrayList<LeaderElection.LeadershipStateListener>();
    public static final String METASTORE_RENEW_LEASE = "metastore.renew.leader.lease";
    private String name;

    private void doWork(LockResponse resp, Configuration conf, TableName tableName) throws LeaderException {
        this.lockId = resp.getLockid();
        assert (resp.getState() == LockState.ACQUIRED || resp.getState() == LockState.WAITING);
        this.shutdownWatcher();
        switch (resp.getState()) {
            case ACQUIRED: {
                boolean renewLease = conf.getBoolean(METASTORE_RENEW_LEASE, true);
                this.heartbeater = renewLease ? new Heartbeater(conf, tableName) : new ReleaseAndRequireWatcher(conf, tableName);
                this.heartbeater.perform();
                if (this.isLeader) break;
                this.isLeader = true;
                this.notifyListener();
                break;
            }
            case WAITING: {
                this.nonLeaderWatcher = new NonLeaderWatcher(conf, tableName);
                this.nonLeaderWatcher.perform();
                if (!this.isLeader) break;
                this.isLeader = false;
                this.notifyListener();
                break;
            }
            default: {
                throw new IllegalStateException("Unexpected lock state: " + (Object)((Object)resp.getState()));
            }
        }
    }

    private void notifyListener() {
        this.listeners.forEach(listener -> {
            try {
                if (this.isLeader) {
                    listener.takeLeadership(this);
                } else {
                    listener.lossLeadership(this);
                }
            }
            catch (Exception e) {
                LOG.error("Error notifying the listener: " + listener + ", leader: " + this.isLeader, (Throwable)e);
            }
        });
    }

    @Override
    public void tryBeLeader(Configuration conf, TableName table) throws LeaderException {
        String hostName;
        String user;
        Objects.requireNonNull(conf, "conf is null");
        Objects.requireNonNull(table, "table is null");
        try {
            user = SecurityUtils.getUser();
            hostName = InetAddress.getLocalHost().getHostName();
        }
        catch (Exception e) {
            throw new LeaderException("Error while getting the username", e);
        }
        if (this.store == null) {
            this.store = TxnUtils.getTxnStore(conf);
        }
        LockComponent component = new LockComponentBuilder().setDbName(table.getDb()).setTableName(table.getTable()).setExclWrite().setOperationType(DataOperationType.NO_TXN).build();
        ArrayList<LockComponent> components = new ArrayList<LockComponent>(1);
        components.add(component);
        boolean lockable = false;
        MetaException recentException = null;
        long start = System.currentTimeMillis();
        LockRequest req = new LockRequest(components, user, hostName);
        int numRetries = MetastoreConf.getIntVar(conf, MetastoreConf.ConfVars.LOCK_NUMRETRIES);
        long maxSleep = MetastoreConf.getTimeVar(conf, MetastoreConf.ConfVars.LOCK_SLEEP_BETWEEN_RETRIES, TimeUnit.MILLISECONDS);
        for (int i = 0; i < numRetries; ++i) {
            try {
                LockResponse res = this.store.lock(req);
                if (res.getState() == LockState.WAITING || res.getState() == LockState.ACQUIRED) {
                    lockable = true;
                    this.doWork(res, conf, table);
                    break;
                }
            }
            catch (NoSuchTxnException | TxnAbortedException e) {
                throw new AssertionError("This should not happen, we didn't open txn", e);
            }
            catch (MetaException e) {
                recentException = e;
                LOG.warn("Error while locking the table: {}, num retries: {}, max retries: {}, exception: {}", new Object[]{table, i, numRetries, e});
            }
            this.backoff(maxSleep);
        }
        if (!lockable) {
            throw new LeaderException("Error locking the table: " + table + " in " + numRetries + " retries, time spent: " + (System.currentTimeMillis() - start) + " ms", (Throwable)((Object)recentException));
        }
    }

    private void backoff(long maxSleep) {
        this.nextSleep *= 2L;
        if (this.nextSleep > maxSleep) {
            this.nextSleep = maxSleep;
        }
        try {
            Thread.sleep(this.nextSleep);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    private void shutdownWatcher() {
        if (this.heartbeater != null) {
            this.heartbeater.shutDown();
            this.heartbeater = null;
        }
        if (this.nonLeaderWatcher != null) {
            this.nonLeaderWatcher.shutDown();
            this.nonLeaderWatcher = null;
        }
    }

    @Override
    public void addStateListener(LeaderElection.LeadershipStateListener listener) {
        Objects.requireNonNull(listener, "listener is null");
        this.listeners.add(listener);
    }

    @Override
    public boolean isLeader() {
        return this.isLeader;
    }

    @Override
    public void close() {
        this.shutdownWatcher();
        if (this.isLeader) {
            this.isLeader = false;
            this.notifyListener();
        }
        if (this.lockId > 0L) {
            try {
                UnlockRequest request = new UnlockRequest(this.lockId);
                this.store.unlock(request);
            }
            catch (NoSuchLockException | TxnOpenException request) {
            }
            catch (Exception e) {
                LOG.error("Error while unlocking: " + this.lockId, (Throwable)e);
            }
        }
    }

    public long getLockId() {
        return this.lockId;
    }

    @Override
    public void setName(String name) {
        this.name = name;
    }

    @Override
    public String getName() {
        return this.name;
    }

    private class ReleaseAndRequireWatcher
    extends LeaseWatcher {
        long timeout;

        public ReleaseAndRequireWatcher(Configuration conf, TableName tableName) {
            super(conf, tableName);
            this.timeout = MetastoreConf.getTimeVar(conf, MetastoreConf.ConfVars.TXN_TIMEOUT, TimeUnit.MILLISECONDS) + 3000L;
        }

        @Override
        public void beforeRun() {
            try {
                Thread.sleep(this.timeout);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }

        @Override
        public void runInternal() {
            this.shutDown();
            try {
                Thread.sleep(5000L);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
            this.conf = new Configuration(this.conf);
            this.conf.setBoolean(LeaseLeaderElection.METASTORE_RENEW_LEASE, true);
            this.reclaim();
        }
    }

    private class Heartbeater
    extends LeaseWatcher {
        private HeartbeatRequest req;
        private long heartbeatInterval;

        Heartbeater(Configuration conf, TableName table) {
            super(conf, table);
            this.req = new HeartbeatRequest();
            this.req.setLockid(LeaseLeaderElection.this.lockId);
            long interval = MetastoreConf.getTimeVar(conf, MetastoreConf.ConfVars.TXN_TIMEOUT, TimeUnit.MILLISECONDS) / 2L;
            if (interval == 0L) {
                throw new RuntimeException((Object)((Object)MetastoreConf.ConfVars.TXN_TIMEOUT) + " not set, heartbeats won't be sent");
            }
            this.heartbeatInterval = interval;
        }

        @Override
        public void beforeRun() {
            long initialDelay = (long)Math.floor((double)this.heartbeatInterval * 0.75 * Math.random());
            try {
                Thread.sleep(initialDelay);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }

        @Override
        public void runInternal() {
            try {
                LeaseLeaderElection.this.store.heartbeat(this.req);
            }
            catch (NoSuchTxnException | TxnAbortedException e) {
                throw new AssertionError("This should not happen, we didn't open txn", e);
            }
            catch (NoSuchLockException e) {
                this.reclaim();
            }
            catch (Exception e) {
                LOG.warn("Heartbeat failed with exception: " + e.getMessage(), (Throwable)e);
            }
        }

        @Override
        public void afterRun() {
            try {
                Thread.sleep(this.heartbeatInterval);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }
    }

    private class NonLeaderWatcher
    extends LeaseWatcher {
        private long sleep;
        private int count;
        private CheckLockRequest request;

        NonLeaderWatcher(Configuration conf, TableName table) {
            super(conf, table);
            this.request = new CheckLockRequest(LeaseLeaderElection.this.lockId);
            this.sleep = MetastoreConf.getTimeVar(conf, MetastoreConf.ConfVars.LOCK_SLEEP_BETWEEN_RETRIES, TimeUnit.MILLISECONDS);
        }

        @Override
        public void runInternal() {
            try {
                if (this.count++ % 3 > 0) {
                    LockResponse res = LeaseLeaderElection.this.store.checkLock(this.request);
                    if (res.getState() == LockState.ACQUIRED) {
                        LeaseLeaderElection.this.doWork(res, this.conf, this.tableName);
                    } else if (res.getState() == LockState.ABORT) {
                        this.reclaim();
                    }
                } else {
                    LeaseLeaderElection.this.store.performTimeOuts();
                }
            }
            catch (NoSuchTxnException | TxnAbortedException e) {
                throw new AssertionError("This should not happen, we didn't open txn", e);
            }
            catch (NoSuchLockException e) {
                this.reclaim();
            }
            catch (Exception e) {
                LOG.warn("CheckLock failed with exception: " + e.getMessage(), (Throwable)e);
            }
        }

        @Override
        public void afterRun() {
            try {
                Thread.sleep(this.sleep);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }
    }

    private abstract class LeaseWatcher
    extends Thread {
        protected Configuration conf;
        protected TableName tableName;
        private volatile boolean stopped = false;

        LeaseWatcher(Configuration conf, TableName tableName) {
            this.conf = conf;
            this.tableName = tableName;
            this.setDaemon(true);
            StringBuilder builder = new StringBuilder("Leader-Watcher-").append(LeaseLeaderElection.this.name != null ? LeaseLeaderElection.this.name : "").append(ID.incrementAndGet());
            this.setName(builder.toString());
        }

        public void perform() {
            LOG.info("Starting a watcher: {} for {}", (Object)this.getClass().getName(), (Object)LeaseLeaderElection.this.name);
            this.start();
        }

        @Override
        public void run() {
            this.beforeRun();
            do {
                try {
                    this.runInternal();
                }
                finally {
                    if (!this.stopped) {
                        this.afterRun();
                    }
                }
            } while (!this.stopped);
        }

        public void shutDown() {
            this.stopped = true;
        }

        public void beforeRun() {
        }

        public void afterRun() {
        }

        public abstract void runInternal();

        public void reclaim() {
            try {
                LeaseLeaderElection.this.tryBeLeader(this.conf, this.tableName);
            }
            catch (Exception e) {
                LOG.error("Error reclaiming the leader, will retry in next cycle", (Throwable)e);
            }
        }
    }
}

