/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.oss.dsbulk.config;

import com.typesafe.config.Config;
import com.typesafe.config.ConfigException;
import com.typesafe.config.ConfigFactory;
import com.typesafe.config.ConfigList;
import com.typesafe.config.ConfigMergeable;
import com.typesafe.config.ConfigObject;
import com.typesafe.config.ConfigValue;
import com.typesafe.config.ConfigValueFactory;
import com.typesafe.config.ConfigValueType;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import java.io.Console;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.InvalidPathException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import java.util.stream.Collectors;

public class ConfigUtils {
    private static final Pattern THREADS_PATTERN;
    @Nullable
    private static final URL CURRENT_DIR;
    @Nullable
    private static final Path USER_HOME;
    private static final String TYPE_ANNOTATION = "@type";
    private static final String LEAF_ANNOTATION = "@leaf";
    private static final Pattern GENERIC_PATTERN;
    private static final Pattern WRONG_TYPE_PATTERN;
    private static final Pattern WRONG_VALUE_PATTERN;
    private static final Pattern NULL_PATTERN;
    private static final Pattern BAD_ENUM_PATTERN;
    private static final Pattern BAD_VALUE_PATTERN;

    public static Config standaloneDSBulkReference() {
        return ConfigFactory.parseResourcesAnySyntax((String)"dsbulk-reference").resolve();
    }

    public static Config standaloneDriverReference() {
        return ConfigFactory.parseResourcesAnySyntax((String)"driver-reference").resolve();
    }

    @NonNull
    public static Config createReferenceConfig() {
        return ConfigFactory.parseResourcesAnySyntax((String)"dsbulk-reference").withFallback((ConfigMergeable)ConfigFactory.parseResourcesAnySyntax((String)"driver-reference")).withFallback((ConfigMergeable)ConfigFactory.defaultReferenceUnresolved());
    }

    @NonNull
    public static Config createApplicationConfig(@Nullable Path appConfigPath) {
        try {
            if (appConfigPath != null) {
                System.setProperty("config.file", appConfigPath.toString());
            }
            Config referenceConfig = ConfigUtils.createReferenceConfig();
            return ConfigFactory.defaultOverrides().withFallback((ConfigMergeable)ConfigFactory.defaultApplication()).withFallback((ConfigMergeable)referenceConfig);
        }
        catch (ConfigException.Parse e) {
            throw new IllegalArgumentException(String.format("Error parsing configuration file %s at line %s. Please make sure its format is compliant with HOCON syntax. If you are using \\ (backslash) to define a path, escape it with \\\\ or use / (forward slash) instead.", e.origin().filename(), e.origin().lineNumber()), e);
        }
    }

    public static Path getPath(Config config, String path) {
        String setting = config.getString(path);
        try {
            return ConfigUtils.resolvePath(setting);
        }
        catch (InvalidPathException e) {
            throw new ConfigException.WrongType(config.origin(), String.format("%s: Expecting valid filepath, got '%s'", path, setting), (Throwable)e);
        }
    }

    public static URL getURL(Config config, String path) {
        String setting = config.getString(path);
        try {
            return ConfigUtils.resolveURL(setting);
        }
        catch (Exception e) {
            throw new ConfigException.WrongType(config.origin(), String.format("%s: Expecting valid filepath or URL, got '%s'", path, setting), (Throwable)e);
        }
    }

    public static int getThreads(Config config, String path) {
        String setting = config.getString(path);
        try {
            return ConfigUtils.resolveThreads(setting);
        }
        catch (Exception e) {
            throw new ConfigException.WrongType(config.origin(), String.format("%s: Expecting positive integer or string in 'nC' syntax, got '%s'", path, setting), (Throwable)e);
        }
    }

    public static char getChar(Config config, String path) {
        String setting = config.getString(path);
        if (setting.length() != 1) {
            throw new ConfigException.WrongType(config.origin(), String.format("%s: Expecting single char, got '%s'", path, setting));
        }
        return setting.charAt(0);
    }

    public static Charset getCharset(Config config, String path) {
        String setting = config.getString(path);
        try {
            return Charset.forName(setting);
        }
        catch (Exception e) {
            throw new ConfigException.WrongType(config.origin(), String.format("%s: Expecting valid charset name, got '%s'", path, setting), (Throwable)e);
        }
    }

    @NonNull
    public static Path resolvePath(@NonNull String path) throws InvalidPathException {
        Optional<Path> resolved = ConfigUtils.resolveUserHome(path);
        return resolved.orElseGet(() -> Paths.get(path, new String[0]).toAbsolutePath().normalize());
    }

    @NonNull
    public static URL resolveURL(@NonNull String url) throws MalformedURLException, InvalidPathException {
        if (url.equals("-")) {
            url = "std:/";
        }
        Optional<Path> resolved = ConfigUtils.resolveUserHome(url);
        try {
            URL u = resolved.isPresent() ? resolved.get().toUri().toURL() : (CURRENT_DIR == null ? new URL(url) : new URL(CURRENT_DIR, url));
            return u.toURI().normalize().toURL();
        }
        catch (Exception e) {
            try {
                return ConfigUtils.resolvePath(url).toUri().toURL();
            }
            catch (MalformedURLException | InvalidPathException e1) {
                e1.addSuppressed(e);
                throw e1;
            }
        }
    }

    @NonNull
    public static Optional<Path> resolveUserHome(@NonNull String path) {
        if (USER_HOME != null && path.startsWith("~")) {
            if (path.equals("~") || path.startsWith("~/")) {
                Path resolved = USER_HOME.resolve('.' + path.substring(1)).toAbsolutePath().normalize();
                return Optional.of(resolved);
            }
            throw new InvalidPathException(path, "Cannot resolve home directory", 1);
        }
        return Optional.empty();
    }

    public static int resolveThreads(@NonNull String threadsStr) {
        int threads;
        try {
            threads = Integer.parseInt(threadsStr);
        }
        catch (NumberFormatException e) {
            Matcher matcher = THREADS_PATTERN.matcher(threadsStr.trim());
            if (matcher.matches()) {
                int threads2 = (int)((float)Runtime.getRuntime().availableProcessors() * Float.parseFloat(matcher.group(1)));
                return Math.max(1, threads2);
            }
            PatternSyntaxException e1 = new PatternSyntaxException("Cannot parse input as N * <num_cores>", THREADS_PATTERN.pattern(), 0);
            e1.addSuppressed(e);
            throw e1;
        }
        if (threads < 1) {
            throw new IllegalArgumentException("Expecting positive number of threads, got " + threads);
        }
        return threads;
    }

    @NonNull
    public static Optional<String> getTypeString(@NonNull Config config, @NonNull String path) {
        ConfigValue value = ConfigUtils.getNullSafeValue(config, path);
        Optional<String> typeHint = ConfigUtils.getTypeHint(value);
        if (typeHint.isPresent()) {
            return typeHint;
        }
        ConfigValueType type = value.valueType();
        if (type == ConfigValueType.LIST) {
            ConfigList list = config.getList(path);
            if (list.isEmpty()) {
                return Optional.of("list");
            }
            ConfigValueType elementType = ((ConfigValue)list.get(0)).valueType();
            return ConfigUtils.getTypeString(elementType).map(str -> "list<" + str + ">");
        }
        if (type == ConfigValueType.OBJECT) {
            ConfigObject object = config.getObject(path);
            if (object.isEmpty()) {
                return Optional.of("map");
            }
            ConfigValueType valueType = ((ConfigValue)object.values().iterator().next()).valueType();
            return ConfigUtils.getTypeString(valueType).map(str -> "map<string," + str + ">");
        }
        return ConfigUtils.getTypeString(type);
    }

    @NonNull
    public static ConfigValueType getValueType(@NonNull Config config, @NonNull String path) {
        ConfigValue value = ConfigUtils.getNullSafeValue(config, path);
        Optional<String> typeHint = ConfigUtils.getTypeHint(value);
        if (typeHint.isPresent()) {
            String hint = typeHint.get();
            if (hint.equals("string")) {
                return ConfigValueType.STRING;
            }
            if (hint.equals("number")) {
                return ConfigValueType.NUMBER;
            }
            if (hint.equals("boolean")) {
                return ConfigValueType.BOOLEAN;
            }
            if (hint.startsWith("list")) {
                return ConfigValueType.LIST;
            }
            if (hint.startsWith("map")) {
                return ConfigValueType.OBJECT;
            }
        }
        return value.valueType();
    }

    @NonNull
    public static Optional<String> getTypeHint(@NonNull ConfigValue value) {
        return value.origin().comments().stream().filter(ConfigUtils::isTypeHint).map(line -> line.replace(TYPE_ANNOTATION, "")).map(String::trim).findFirst();
    }

    @NonNull
    public static String getComments(@NonNull ConfigValue value) {
        return value.origin().comments().stream().filter(line -> !ConfigUtils.isTypeHint(line)).filter(line -> !ConfigUtils.isLeaf(line)).map(String::trim).collect(Collectors.joining("\n"));
    }

    public static boolean isTypeHint(@NonNull String line) {
        return line.contains(TYPE_ANNOTATION);
    }

    @NonNull
    private static Optional<String> getTypeString(@NonNull ConfigValueType type) {
        switch (type) {
            case STRING: {
                return Optional.of("string");
            }
            case LIST: {
                return Optional.of("list");
            }
            case OBJECT: {
                return Optional.of("map");
            }
            case NUMBER: {
                return Optional.of("number");
            }
            case BOOLEAN: {
                return Optional.of("boolean");
            }
        }
        return Optional.empty();
    }

    public static boolean isLeaf(@NonNull ConfigValue value) {
        return !(value instanceof ConfigObject) || value.origin().comments().stream().anyMatch(ConfigUtils::isLeaf);
    }

    public static boolean isLeaf(@NonNull String line) {
        return line.contains(LEAF_ANNOTATION);
    }

    @NonNull
    public static ConfigValue getNullSafeValue(@NonNull Config config, @NonNull String path) {
        int dot = path.indexOf(46);
        if (dot == -1) {
            ConfigValue value = config.root().get((Object)path);
            if (value == null) {
                throw new ConfigException.Missing(path);
            }
            return value;
        }
        try {
            return ConfigUtils.getNullSafeValue(config.getConfig(path.substring(0, dot)), path.substring(dot + 1));
        }
        catch (ConfigException.Missing e) {
            throw new ConfigException.Missing(path);
        }
    }

    public static List<URL> getURLsFromFile(Path urlfile) throws IOException {
        ArrayList<URL> result = new ArrayList<URL>();
        List<String> paths = Files.readAllLines(urlfile);
        for (String path : paths) {
            try {
                if (path.startsWith("#")) continue;
                result.add(ConfigUtils.resolveURL(path.trim()));
            }
            catch (Exception e) {
                throw new IllegalArgumentException(String.format("%s: Expecting valid filepath or URL, got '%s'", urlfile, path), e);
            }
        }
        return result;
    }

    public static boolean isPathPresentAndNotEmpty(Config config, String path) {
        return config.hasPath(path) && !config.getString(path).isEmpty();
    }

    public static boolean hasReferenceValue(Config config, String path) {
        if (!config.hasPathOrNull(path)) {
            return false;
        }
        ConfigValue value = config.getIsNull(path) ? ConfigUtils.getNullSafeValue(config, path) : config.getValue(path);
        String resource = value.origin().resource();
        return resource != null && resource.endsWith("reference.conf");
    }

    public static boolean hasUserOverride(Config config, String path) {
        return config.hasPathOrNull(path) && !ConfigUtils.hasReferenceValue(config, path);
    }

    @NonNull
    public static IllegalArgumentException convertConfigException(@NonNull ConfigException e, @NonNull String basePath) {
        if (!basePath.isEmpty()) {
            basePath = basePath + '.';
        }
        try {
            Matcher matcher;
            String errorMsg;
            if (e instanceof ConfigException.WrongType) {
                errorMsg = e.getMessage();
                matcher = WRONG_TYPE_PATTERN.matcher(errorMsg);
                if (matcher.find()) {
                    errorMsg = matcher.replaceAll("Invalid value for " + basePath + "$3, expecting $5, got $4 [at: $1: $2]");
                    return new IllegalArgumentException(errorMsg);
                }
                matcher = WRONG_VALUE_PATTERN.matcher(errorMsg);
                if (matcher.find()) {
                    errorMsg = matcher.replaceAll("Invalid value for " + basePath + "$3, expecting $4, got $5 [at: $1: $2]");
                    return new IllegalArgumentException(errorMsg);
                }
            } else if (e instanceof ConfigException.BadValue) {
                errorMsg = e.getMessage();
                matcher = BAD_ENUM_PATTERN.matcher(errorMsg);
                if (matcher.find()) {
                    errorMsg = matcher.replaceAll("Invalid value for " + basePath + "$3, expecting one of $5, got: $4 [at: $1: $2]");
                    return new IllegalArgumentException(errorMsg);
                }
                matcher = BAD_VALUE_PATTERN.matcher(errorMsg);
                if (matcher.find()) {
                    errorMsg = matcher.replaceAll("Invalid value for " + basePath + "$3: $4 [at: $1: $2]");
                    return new IllegalArgumentException(errorMsg);
                }
            } else if (e instanceof ConfigException.Null && (matcher = NULL_PATTERN.matcher(errorMsg = e.getMessage())).find()) {
                errorMsg = matcher.replaceAll("Invalid value for " + basePath + "$3, expecting $4, got NULL [at: $1: $2]");
                return new IllegalArgumentException(errorMsg);
            }
            if ((matcher = GENERIC_PATTERN.matcher(errorMsg = e.getMessage())).find()) {
                errorMsg = matcher.replaceAll("$3 [at: $1: $2]");
                return new IllegalArgumentException(errorMsg);
            }
        }
        catch (RuntimeException runtimeException) {
            // empty catch block
        }
        return new IllegalArgumentException(e.getMessage());
    }

    @NonNull
    public static Config readPassword(@NonNull Config config, @NonNull String path, @NonNull Console console) {
        char[] password = console.readPassword("Please input value for setting %s: ", path);
        config = config.withValue(path, ConfigValueFactory.fromAnyRef((Object)new String(password), (String)"stdin"));
        Arrays.fill(password, ' ');
        return config;
    }

    static {
        Path userHome;
        URL currentDir;
        THREADS_PATTERN = Pattern.compile("(.+)\\s*C", 2);
        GENERIC_PATTERN = Pattern.compile("(.+): (\\d+): (.+)", 2);
        WRONG_TYPE_PATTERN = Pattern.compile("(.+): (\\d+): (.+) has type (.+) rather than (.+)", 2);
        WRONG_VALUE_PATTERN = Pattern.compile("(.+): (\\d+): (.+): Expecting (.+), got (.+)", 2);
        NULL_PATTERN = Pattern.compile("(.+): (\\d+): Configuration key '(.+)' is set to null but expected (.+)", 2);
        BAD_ENUM_PATTERN = Pattern.compile("(.+): (\\d+): Invalid value at '(.+)': The enum class .+ has no constant of the name ('.+') \\(should be one of \\[([^]]+)]\\.\\)", 2);
        BAD_VALUE_PATTERN = Pattern.compile("(.+): (\\d+): Invalid value at '(.+)': (.+)", 2);
        try {
            currentDir = Paths.get(System.getProperty("user.dir"), new String[0]).toAbsolutePath().toUri().toURL();
        }
        catch (Throwable t) {
            currentDir = null;
        }
        CURRENT_DIR = currentDir;
        try {
            userHome = Paths.get(System.getProperty("user.home"), new String[0]).toAbsolutePath();
        }
        catch (Throwable t) {
            userHome = null;
        }
        USER_HOME = userHome;
    }
}

