/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hive.metastore.utils;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.base.Objects;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URI;
import java.security.PrivilegedExceptionAction;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import javax.security.auth.login.LoginException;
import org.apache.commons.lang.ArrayUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FsShell;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.AclEntry;
import org.apache.hadoop.fs.permission.AclEntryScope;
import org.apache.hadoop.fs.permission.AclEntryType;
import org.apache.hadoop.fs.permission.AclStatus;
import org.apache.hadoop.fs.permission.FsAction;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.client.HdfsAdmin;
import org.apache.hadoop.hive.metastore.utils.SecurityUtils;
import org.apache.hadoop.security.AccessControlException;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.tools.DistCp;
import org.apache.hadoop.tools.DistCpOptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HdfsUtils {
    private static final Logger LOG = LoggerFactory.getLogger(HdfsUtils.class);
    private static final String DISTCP_OPTIONS_PREFIX = "distcp.options.";
    private static final String HDFS_ID_PATH_PREFIX = "/.reserved/.inodes/";

    public static void checkFileAccess(FileSystem fs, FileStatus stat, FsAction action) throws IOException, LoginException {
        HdfsUtils.checkFileAccess(fs, stat, action, SecurityUtils.getUGI());
    }

    @VisibleForTesting
    static void checkFileAccess(FileSystem fs, FileStatus stat, FsAction action, UserGroupInformation ugi) throws IOException {
        String superGroupName;
        String user = ugi.getShortUserName();
        String[] groups = ugi.getGroupNames();
        if (groups != null && HdfsUtils.arrayContains(groups, superGroupName = fs.getConf().get("dfs.permissions.supergroup", ""))) {
            LOG.debug("User \"" + user + "\" belongs to super-group \"" + superGroupName + "\". Permission granted for action: " + action + ".");
            return;
        }
        FsPermission dirPerms = stat.getPermission();
        if (user.equals(stat.getOwner()) ? dirPerms.getUserAction().implies(action) : (HdfsUtils.arrayContains(groups, stat.getGroup()) ? dirPerms.getGroupAction().implies(action) : dirPerms.getOtherAction().implies(action))) {
            return;
        }
        throw new AccessControlException("action " + action + " not permitted on path " + stat.getPath() + " for user " + user);
    }

    public static boolean isPathEncrypted(Configuration conf, URI fsUri, Path path) throws IOException {
        Path fullPath = path.isAbsolute() ? path : path.getFileSystem(conf).makeQualified(path);
        if (!"hdfs".equalsIgnoreCase(path.toUri().getScheme())) {
            return false;
        }
        try {
            HdfsAdmin hdfsAdmin = new HdfsAdmin(fsUri, conf);
            return hdfsAdmin.getEncryptionZoneForPath(fullPath) != null;
        }
        catch (FileNotFoundException fnfe) {
            LOG.debug("Failed to get EZ for non-existent path: " + fullPath, (Throwable)fnfe);
            return false;
        }
    }

    private static boolean arrayContains(String[] array, String value) {
        if (array == null) {
            return false;
        }
        for (String element : array) {
            if (!element.equals(value)) continue;
            return true;
        }
        return false;
    }

    public static boolean runDistCpAs(final List<Path> srcPaths, final Path dst, final Configuration conf, String doAsUser) throws IOException {
        UserGroupInformation proxyUser = UserGroupInformation.createProxyUser((String)doAsUser, (UserGroupInformation)UserGroupInformation.getLoginUser());
        try {
            return (Boolean)proxyUser.doAs((PrivilegedExceptionAction)new PrivilegedExceptionAction<Boolean>(){

                @Override
                public Boolean run() throws Exception {
                    return HdfsUtils.runDistCp(srcPaths, dst, conf);
                }
            });
        }
        catch (InterruptedException e) {
            throw new IOException(e);
        }
    }

    public static boolean runDistCp(List<Path> srcPaths, Path dst, Configuration conf) throws IOException {
        DistCpOptions options = new DistCpOptions.Builder(srcPaths, dst).withSyncFolder(true).withCRC(true).preserve(DistCpOptions.FileAttribute.BLOCKSIZE).build();
        List<String> params = HdfsUtils.constructDistCpParams(srcPaths, dst, conf);
        try {
            conf.setBoolean("mapred.mapper.new-api", true);
            DistCp distcp = new DistCp(conf, options);
            if (distcp.run(params.toArray(new String[params.size()])) == 0) {
                boolean bl = true;
                return bl;
            }
            boolean bl = false;
            return bl;
        }
        catch (Exception e) {
            throw new IOException("Cannot execute DistCp process: ", e);
        }
        finally {
            conf.setBoolean("mapred.mapper.new-api", false);
        }
    }

    private static List<String> constructDistCpParams(List<Path> srcPaths, Path dst, Configuration conf) {
        List<String> params = HdfsUtils.constructDistCpOptions(conf);
        if (params.size() == 0) {
            params.add("-update");
            params.add("-pbx");
        }
        for (Path src : srcPaths) {
            params.add(src.toString());
        }
        params.add(dst.toString());
        return params;
    }

    public static List<String> constructDistCpOptions(Configuration conf) {
        ArrayList<String> options = new ArrayList<String>();
        for (Map.Entry entry : conf.getPropsWithPrefix(DISTCP_OPTIONS_PREFIX).entrySet()) {
            String distCpOption = (String)entry.getKey();
            String distCpVal = (String)entry.getValue();
            options.add("-" + distCpOption);
            if (distCpVal == null || distCpVal.isEmpty()) continue;
            options.add(distCpVal);
        }
        return options;
    }

    public static Path getFileIdPath(FileSystem fileSystem, Path path, long fileId) {
        return fileSystem instanceof DistributedFileSystem ? new Path(HDFS_ID_PATH_PREFIX + fileId) : path;
    }

    public static long getFileId(FileSystem fs, String path) throws IOException {
        return HdfsUtils.ensureDfs(fs).getClient().getFileInfo(path).getFileId();
    }

    private static DistributedFileSystem ensureDfs(FileSystem fs) {
        if (!(fs instanceof DistributedFileSystem)) {
            throw new UnsupportedOperationException("Only supported for DFS; got " + fs.getClass());
        }
        return (DistributedFileSystem)fs;
    }

    public static void setFullFileStatus(Configuration conf, HadoopFileStatus sourceStatus, String targetGroup, FileSystem fs, Path target, boolean recursion) {
        HdfsUtils.setFullFileStatus(conf, sourceStatus, targetGroup, fs, target, recursion, recursion ? new FsShell() : null);
    }

    @VisibleForTesting
    static void setFullFileStatus(Configuration conf, HadoopFileStatus sourceStatus, String targetGroup, FileSystem fs, Path target, boolean recursion, FsShell fsShell) {
        try {
            FileStatus fStatus = sourceStatus.getFileStatus();
            String group = fStatus.getGroup();
            boolean aclEnabled = Objects.equal(conf.get("dfs.namenode.acls.enabled"), "true");
            FsPermission sourcePerm = fStatus.getPermission();
            ArrayList<AclEntry> aclEntries = null;
            if (aclEnabled && sourceStatus.getAclEntries() != null) {
                LOG.trace(sourceStatus.getAclStatus().toString());
                aclEntries = new ArrayList<AclEntry>(sourceStatus.getAclEntries());
                HdfsUtils.removeBaseAclEntries(aclEntries);
                aclEntries.add(HdfsUtils.newAclEntry(AclEntryScope.ACCESS, AclEntryType.USER, sourcePerm.getUserAction()));
                aclEntries.add(HdfsUtils.newAclEntry(AclEntryScope.ACCESS, AclEntryType.GROUP, sourcePerm.getGroupAction()));
                aclEntries.add(HdfsUtils.newAclEntry(AclEntryScope.ACCESS, AclEntryType.OTHER, sourcePerm.getOtherAction()));
            }
            if (recursion) {
                fsShell.setConf(conf);
                if (group != null && !group.isEmpty()) {
                    HdfsUtils.run(fsShell, new String[]{"-chgrp", "-R", group, target.toString()});
                }
                if (aclEnabled) {
                    if (null != aclEntries) {
                        try {
                            String aclEntry = Joiner.on(",").join(aclEntries);
                            HdfsUtils.run(fsShell, new String[]{"-setfacl", "-R", "--set", aclEntry, target.toString()});
                        }
                        catch (Exception e) {
                            LOG.info("Skipping ACL inheritance: File system for path " + target + " does not support ACLs but dfs.namenode.acls.enabled is set to true. ");
                            LOG.debug("The details are: " + e, (Throwable)e);
                        }
                    }
                } else {
                    String permission = Integer.toString(sourcePerm.toShort(), 8);
                    HdfsUtils.run(fsShell, new String[]{"-chmod", "-R", permission, target.toString()});
                }
            } else {
                if (!(group == null || group.isEmpty() || targetGroup != null && group.equals(targetGroup))) {
                    fs.setOwner(target, null, group);
                }
                if (aclEnabled) {
                    if (null != aclEntries) {
                        fs.setAcl(target, aclEntries);
                    }
                } else {
                    fs.setPermission(target, sourcePerm);
                }
            }
        }
        catch (Exception e) {
            LOG.warn("Unable to inherit permissions for file " + target + " from file " + sourceStatus.getFileStatus().getPath(), (Object)e.getMessage());
            LOG.debug("Exception while inheriting permissions", (Throwable)e);
        }
    }

    private static void removeBaseAclEntries(List<AclEntry> entries) {
        Iterables.removeIf(entries, new Predicate<AclEntry>(){

            @Override
            public boolean apply(AclEntry input) {
                return input.getName() == null;
            }
        });
    }

    private static AclEntry newAclEntry(AclEntryScope scope, AclEntryType type, FsAction permission) {
        return new AclEntry.Builder().setScope(scope).setType(type).setPermission(permission).build();
    }

    private static void run(FsShell shell, String[] command) throws Exception {
        LOG.debug(ArrayUtils.toString(command));
        int retval = shell.run(command);
        LOG.debug("Return value is :" + retval);
    }

    public static class HadoopFileStatus {
        private final FileStatus fileStatus;
        private final AclStatus aclStatus;

        public HadoopFileStatus(Configuration conf, FileSystem fs, Path file) throws IOException {
            FileStatus fileStatus = fs.getFileStatus(file);
            AclStatus aclStatus = null;
            if (Objects.equal(conf.get("dfs.namenode.acls.enabled"), "true")) {
                try {
                    aclStatus = fs.getAclStatus(file);
                }
                catch (Exception e) {
                    LOG.info("Skipping ACL inheritance: File system for path " + file + " does not support ACLs but dfs.namenode.acls.enabled is set to true. ");
                    LOG.debug("The details are: " + e, (Throwable)e);
                }
            }
            this.fileStatus = fileStatus;
            this.aclStatus = aclStatus;
        }

        public FileStatus getFileStatus() {
            return this.fileStatus;
        }

        List<AclEntry> getAclEntries() {
            return this.aclStatus == null ? null : Collections.unmodifiableList(this.aclStatus.getEntries());
        }

        @VisibleForTesting
        AclStatus getAclStatus() {
            return this.aclStatus;
        }
    }
}

