// Copyright (c) 2007-2023 Broadcom. All Rights Reserved. The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
//
// This software, the RabbitMQ Java client library, is triple-licensed under the
// Mozilla Public License 2.0 ("MPL"), the GNU General Public License version 2
// ("GPL") and the Apache License version 2 ("ASL"). For the MPL, please see
// LICENSE-MPL-RabbitMQ. For the GPL, please see LICENSE-GPL2.  For the ASL,
// please see LICENSE-APACHE2.
//
// This software is distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND,
// either express or implied. See the LICENSE file for specific language governing
// rights and limitations of this software.
//
// If you have any questions regarding licensing, please contact us at
// info@rabbitmq.com.


package com.rabbitmq.client.impl;

import java.io.DataOutputStream;
import java.io.IOException;
import java.util.Date;
import java.util.Map;

import com.rabbitmq.client.ContentHeader;
import com.rabbitmq.client.LongString;

/**
 * Generates an AMQP wire-protocol packet from a {@link ContentHeader}.
 * Methods on this object are usually called from autogenerated code.
 */
public class ContentHeaderPropertyWriter {
    /** Accumulates our output */
    private final ValueWriter out;

    /** Current flags word being accumulated */
    private int flagWord;

    /** Position within current flags word */
    private int bitCount;

    /**
     * Constructs a fresh ContentHeaderPropertyWriter.
     */
    public ContentHeaderPropertyWriter(DataOutputStream out) {
        this.out = new ValueWriter(out);
        this.flagWord = 0;
        this.bitCount = 0;
    }

    private void emitFlagWord(boolean continuationBit) throws IOException {
        out.writeShort(continuationBit ? (flagWord | 1) : flagWord);
        flagWord = 0;
        bitCount = 0;
    }

    public void writePresence(boolean present) throws IOException {
        if (bitCount == 15) {
            emitFlagWord(true);
        }

        if (present) {
            int bit = 15 - bitCount;
            flagWord = flagWord | (1 << bit);
        }
        bitCount++;
    }

    public void finishPresence() throws IOException {
        emitFlagWord(false);
    }

    public void writeShortstr(String str) throws IOException {
        out.writeShortstr(str);
    }

    public void writeLongstr(String str) throws IOException {
        out.writeLongstr(str);
    }

    public void writeLongstr(LongString str) throws IOException {
        out.writeLongstr(str);
    }

    public void writeShort(Integer s) throws IOException {
        out.writeShort(s);
    }

    public void writeLong(Integer l) throws IOException {
        out.writeLong(l);
    }

    public void writeLonglong(Long ll) throws IOException {
        out.writeLonglong(ll);
    }

    public void writeTable(Map<String, Object> table) throws IOException {
        out.writeTable(table);
    }

    public void writeOctet(Integer octet) throws IOException {
        out.writeOctet(octet);
    }

    public void writeOctet(int octet) throws IOException {
        out.writeOctet(octet);
    }

    public void writeTimestamp(Date timestamp) throws IOException {
        out.writeTimestamp(timestamp);
    }
}
