/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.util;

import java.io.IOException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.CompoundConfiguration;
import org.apache.hadoop.hbase.DoNotRetryIOException;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.ColumnFamilyDescriptor;
import org.apache.hadoop.hbase.client.TableDescriptor;
import org.apache.hadoop.hbase.conf.ConfigKey;
import org.apache.hadoop.hbase.fs.ErasureCodingUtils;
import org.apache.hadoop.hbase.regionserver.DataTieringType;
import org.apache.hadoop.hbase.regionserver.DateTieredStoreEngine;
import org.apache.hadoop.hbase.regionserver.DefaultStoreEngine;
import org.apache.hadoop.hbase.regionserver.HStore;
import org.apache.hadoop.hbase.regionserver.RegionCoprocessorHost;
import org.apache.hadoop.hbase.regionserver.RegionSplitPolicy;
import org.apache.hadoop.hbase.regionserver.StoreEngine;
import org.apache.hadoop.hbase.regionserver.compactions.ExploringCompactionPolicy;
import org.apache.hadoop.hbase.regionserver.compactions.FIFOCompactionPolicy;
import org.apache.hadoop.hbase.shaded.protobuf.generated.WALProtos;
import org.apache.hadoop.hbase.util.BloomFilterUtil;
import org.apache.hadoop.hbase.util.CompressionTest;
import org.apache.hadoop.hbase.util.EncryptionTest;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public final class TableDescriptorChecker {
    private static Logger LOG = LoggerFactory.getLogger(TableDescriptorChecker.class);
    public static final String TABLE_SANITY_CHECKS = "hbase.table.sanity.checks";
    public static final boolean DEFAULT_TABLE_SANITY_CHECKS = true;
    public static final String MASTER_CHECK_COMPRESSION = "hbase.master.check.compression";
    public static final boolean DEFAULT_MASTER_CHECK_COMPRESSION = true;
    public static final String MASTER_CHECK_ENCRYPTION = "hbase.master.check.encryption";
    public static final boolean DEFAULT_MASTER_CHECK_ENCRYPTION = true;

    private TableDescriptorChecker() {
    }

    private static boolean shouldSanityCheck(Configuration conf) {
        return conf.getBoolean(TABLE_SANITY_CHECKS, true);
    }

    public static void sanityCheck(Configuration c, TableDescriptor td) throws IOException {
        int regionReplicas;
        String message;
        long flushSize;
        long maxFileSize;
        CompoundConfiguration conf = new CompoundConfiguration().add(c).addBytesMap(td.getValues());
        boolean logWarn = !TableDescriptorChecker.shouldSanityCheck((Configuration)conf);
        TableDescriptorChecker.warnOrThrowExceptionForFailure(logWarn, () -> ConfigKey.validate((Configuration)conf));
        TableDescriptorChecker.warnOrThrowExceptionForFailure(logWarn, () -> {
            for (ColumnFamilyDescriptor cfd : td.getColumnFamilies()) {
                ConfigKey.validate((Configuration)new CompoundConfiguration().addStringMap(cfd.getConfiguration()).addBytesMap(cfd.getValues()));
            }
        });
        long maxFileSizeLowerLimit = 0x200000L;
        long l = maxFileSize = td.getValue("MAX_FILESIZE") == null ? conf.getLong(HConstants.HREGION_MAX_FILESIZE, maxFileSizeLowerLimit) : Long.parseLong(td.getValue("MAX_FILESIZE"));
        if (maxFileSize < conf.getLong("hbase.hregion.max.filesize.limit", maxFileSizeLowerLimit)) {
            String message2 = "MAX_FILESIZE for table descriptor or \"hbase.hregion.max.filesize\" (" + maxFileSize + ") is too small, which might cause over splitting into unmanageable number of regions.";
            TableDescriptorChecker.warnOrThrowExceptionForFailure(logWarn, message2, null);
        }
        long flushSizeLowerLimit = 0x100000L;
        long l2 = flushSize = td.getValue("MEMSTORE_FLUSHSIZE") == null ? conf.getLong(HConstants.HREGION_MEMSTORE_FLUSH_SIZE, flushSizeLowerLimit) : Long.parseLong(td.getValue("MEMSTORE_FLUSHSIZE"));
        if (flushSize < conf.getLong("hbase.hregion.memstore.flush.size.limit", flushSizeLowerLimit)) {
            message = "MEMSTORE_FLUSHSIZE for table descriptor or \"hbase.hregion.memstore.flush.size\" (" + flushSize + ") is too small, which might cause very frequent flushing.";
            TableDescriptorChecker.warnOrThrowExceptionForFailure(logWarn, message, null);
        }
        TableDescriptorChecker.checkClassLoading((Configuration)conf, td);
        if (conf.getBoolean(MASTER_CHECK_COMPRESSION, true)) {
            TableDescriptorChecker.checkCompression((Configuration)conf, td);
        }
        if (conf.getBoolean(MASTER_CHECK_ENCRYPTION, true)) {
            TableDescriptorChecker.checkEncryption((Configuration)conf, td);
        }
        TableDescriptorChecker.checkCompactionPolicy((Configuration)conf, td);
        if (td.getColumnFamilyCount() == 0) {
            message = "Table should have at least one column family.";
            TableDescriptorChecker.warnOrThrowExceptionForFailure(logWarn, message, null);
        }
        if ((regionReplicas = td.getRegionReplication()) < 1) {
            String message3 = "Table region replication should be at least one.";
            TableDescriptorChecker.warnOrThrowExceptionForFailure(logWarn, message3, null);
        }
        if (td.isReadOnly() && TableName.isMetaTableName((TableName)td.getTableName())) {
            TableDescriptorChecker.warnOrThrowExceptionForFailure(false, "Meta table can't be set as read only.", null);
        }
        TableDescriptorChecker.checkReplicationScope((Configuration)conf, td);
        TableDescriptorChecker.checkBloomFilterType((Configuration)conf, td);
        if (td.getErasureCodingPolicy() != null) {
            TableDescriptorChecker.warnOrThrowExceptionForFailure(logWarn, () -> ErasureCodingUtils.verifySupport((Configuration)conf, td.getErasureCodingPolicy()));
        }
        for (ColumnFamilyDescriptor hcd : td.getColumnFamilies()) {
            String message4;
            if (hcd.getTimeToLive() <= 0) {
                message4 = "TTL for column family " + hcd.getNameAsString() + " must be positive.";
                TableDescriptorChecker.warnOrThrowExceptionForFailure(logWarn, message4, null);
            }
            if (hcd.getBlocksize() < 1024 || hcd.getBlocksize() > 0x1000000) {
                message4 = "Block size for column family " + hcd.getNameAsString() + "  must be between 1K and 16MB.";
                TableDescriptorChecker.warnOrThrowExceptionForFailure(logWarn, message4, null);
            }
            if (hcd.getMinVersions() < 0) {
                message4 = "Min versions for column family " + hcd.getNameAsString() + "  must be positive.";
                TableDescriptorChecker.warnOrThrowExceptionForFailure(logWarn, message4, null);
            }
            if (hcd.getMinVersions() > hcd.getMaxVersions()) {
                message4 = "Min versions for column family " + hcd.getNameAsString() + " must be less than the Max versions.";
                TableDescriptorChecker.warnOrThrowExceptionForFailure(logWarn, message4, null);
            }
            if (hcd.getDFSReplication() < 0) {
                message4 = "HFile Replication for column family " + hcd.getNameAsString() + "  must be greater than zero.";
                TableDescriptorChecker.warnOrThrowExceptionForFailure(logWarn, message4, null);
            }
            TableDescriptorChecker.warnOrThrowExceptionForFailure(logWarn, () -> ((ColumnFamilyDescriptor)hcd).getInMemoryCompaction());
            TableDescriptorChecker.checkDateTieredCompactionForTimeRangeDataTiering((Configuration)conf, td);
        }
    }

    private static void checkReplicationScope(Configuration conf, TableDescriptor td) throws IOException {
        TableDescriptorChecker.warnOrThrowExceptionForFailure(conf, () -> {
            for (ColumnFamilyDescriptor cfd : td.getColumnFamilies()) {
                WALProtos.ScopeType scop = WALProtos.ScopeType.valueOf((int)cfd.getScope());
                if (scop != null) continue;
                String message = "Replication scope for column family " + cfd.getNameAsString() + " is " + cfd.getScope() + " which is invalid.";
                throw new DoNotRetryIOException(message);
            }
        });
    }

    private static void checkDateTieredCompactionForTimeRangeDataTiering(Configuration conf, TableDescriptor td) throws IOException {
        TableDescriptorChecker.checkDateTieredCompactionForTimeRangeDataTiering(conf);
        for (ColumnFamilyDescriptor cfd : td.getColumnFamilies()) {
            CompoundConfiguration cfdConf = new CompoundConfiguration().add(conf).addStringMap(cfd.getConfiguration());
            TableDescriptorChecker.checkDateTieredCompactionForTimeRangeDataTiering((Configuration)cfdConf);
        }
    }

    private static void checkDateTieredCompactionForTimeRangeDataTiering(Configuration conf) throws IOException {
        String errorMessage = "Time Range Data Tiering should be enabled with Date Tiered Compaction.";
        TableDescriptorChecker.warnOrThrowExceptionForFailure(false, () -> {
            if (DataTieringType.TIME_RANGE.name().equals(conf.get("hbase.hstore.datatiering.type")) && !DateTieredStoreEngine.DATE_TIERED_STORE_ENGINE.equals(conf.get(StoreEngine.STORE_ENGINE_CLASS_KEY))) {
                throw new IllegalArgumentException("Time Range Data Tiering should be enabled with Date Tiered Compaction.");
            }
        });
    }

    private static void checkCompactionPolicy(Configuration conf, TableDescriptor td) throws IOException {
        TableDescriptorChecker.warnOrThrowExceptionForFailure(false, () -> {
            String className = td.getValue(DefaultStoreEngine.DEFAULT_COMPACTION_POLICY_CLASS_KEY);
            if (className == null) {
                className = conf.get(DefaultStoreEngine.DEFAULT_COMPACTION_POLICY_CLASS_KEY, ExploringCompactionPolicy.class.getName());
            }
            int blockingFileCount = 16;
            String sv = td.getValue(HStore.BLOCKING_STOREFILES_KEY);
            blockingFileCount = sv != null ? Integer.parseInt(sv) : conf.getInt(HStore.BLOCKING_STOREFILES_KEY, blockingFileCount);
            for (ColumnFamilyDescriptor hcd : td.getColumnFamilies()) {
                String compactionPolicy = hcd.getConfigurationValue(DefaultStoreEngine.DEFAULT_COMPACTION_POLICY_CLASS_KEY);
                if (compactionPolicy == null) {
                    compactionPolicy = className;
                }
                if (!compactionPolicy.equals(FIFOCompactionPolicy.class.getName())) continue;
                String message = null;
                if (hcd.getTimeToLive() == Integer.MAX_VALUE) {
                    message = "Default TTL is not supported for FIFO compaction";
                    throw new IOException(message);
                }
                if (hcd.getMinVersions() > 0) {
                    message = "MIN_VERSION > 0 is not supported for FIFO compaction";
                    throw new IOException(message);
                }
                sv = hcd.getConfigurationValue(HStore.BLOCKING_STOREFILES_KEY);
                if (sv != null) {
                    blockingFileCount = Integer.parseInt(sv);
                }
                if (blockingFileCount >= 1000) continue;
                message = "Blocking file count '" + HStore.BLOCKING_STOREFILES_KEY + "' " + blockingFileCount + " is below recommended minimum of 1000 for column family " + hcd.getNameAsString();
                throw new IOException(message);
            }
        });
    }

    private static void checkBloomFilterType(Configuration conf, TableDescriptor td) throws IOException {
        TableDescriptorChecker.warnOrThrowExceptionForFailure(conf, () -> {
            for (ColumnFamilyDescriptor cfd : td.getColumnFamilies()) {
                CompoundConfiguration cfdConf = new CompoundConfiguration().addStringMap(cfd.getConfiguration());
                try {
                    BloomFilterUtil.getBloomFilterParam(cfd.getBloomFilterType(), (Configuration)cfdConf);
                }
                catch (IllegalArgumentException e) {
                    throw new DoNotRetryIOException("Failed to get bloom filter param", (Throwable)e);
                }
            }
        });
    }

    public static void checkCompression(Configuration conf, TableDescriptor td) throws IOException {
        TableDescriptorChecker.warnOrThrowExceptionForFailure(conf, () -> {
            for (ColumnFamilyDescriptor cfd : td.getColumnFamilies()) {
                CompressionTest.testCompression(cfd.getCompressionType());
                CompressionTest.testCompression(cfd.getCompactionCompressionType());
                CompressionTest.testCompression(cfd.getMajorCompactionCompressionType());
                CompressionTest.testCompression(cfd.getMinorCompactionCompressionType());
            }
        });
    }

    public static void checkEncryption(Configuration conf, TableDescriptor td) throws IOException {
        TableDescriptorChecker.warnOrThrowExceptionForFailure(conf, () -> {
            for (ColumnFamilyDescriptor cfd : td.getColumnFamilies()) {
                EncryptionTest.testEncryption(conf, cfd.getEncryptionType(), cfd.getEncryptionKey());
            }
        });
    }

    public static void checkClassLoading(Configuration conf, TableDescriptor td) throws IOException {
        TableDescriptorChecker.warnOrThrowExceptionForFailure(conf, () -> {
            RegionSplitPolicy.getSplitPolicyClass(td, conf);
            RegionCoprocessorHost.testTableCoprocessorAttrs(conf, td);
        });
    }

    private static void warnOrThrowExceptionForFailure(boolean logWarn, String message, Exception cause) throws IOException {
        if (!logWarn) {
            throw new DoNotRetryIOException(message + " Set " + TABLE_SANITY_CHECKS + " to false at conf or table descriptor if you want to bypass sanity checks", (Throwable)cause);
        }
        LOG.warn(message);
    }

    private static void warnOrThrowExceptionForFailure(Configuration conf, ThrowingRunnable runnable) throws IOException {
        boolean logWarn = !TableDescriptorChecker.shouldSanityCheck(conf);
        TableDescriptorChecker.warnOrThrowExceptionForFailure(logWarn, runnable);
    }

    private static void warnOrThrowExceptionForFailure(boolean logWarn, ThrowingRunnable runnable) throws IOException {
        try {
            runnable.run();
        }
        catch (Exception e) {
            TableDescriptorChecker.warnOrThrowExceptionForFailure(logWarn, e.getMessage(), e);
        }
    }

    @FunctionalInterface
    static interface ThrowingRunnable {
        public void run() throws Exception;
    }
}

