/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.common.table.timeline;

import java.io.IOException;
import java.text.ParseException;
import java.util.AbstractMap;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.hudi.avro.model.HoodieCleanMetadata;
import org.apache.hudi.avro.model.HoodieRestoreMetadata;
import org.apache.hudi.common.config.HoodieCommonConfig;
import org.apache.hudi.common.model.HoodieCommitMetadata;
import org.apache.hudi.common.model.HoodieReplaceCommitMetadata;
import org.apache.hudi.common.model.HoodieTableType;
import org.apache.hudi.common.model.WriteOperationType;
import org.apache.hudi.common.table.HoodieTableMetaClient;
import org.apache.hudi.common.table.timeline.CommitMetadataSerDe;
import org.apache.hudi.common.table.timeline.HoodieActiveTimeline;
import org.apache.hudi.common.table.timeline.HoodieArchivedTimeline;
import org.apache.hudi.common.table.timeline.HoodieInstant;
import org.apache.hudi.common.table.timeline.HoodieInstantTimeGenerator;
import org.apache.hudi.common.table.timeline.HoodieTimeline;
import org.apache.hudi.common.table.timeline.InstantComparison;
import org.apache.hudi.common.table.timeline.InstantFileNameGenerator;
import org.apache.hudi.common.table.timeline.InstantGenerator;
import org.apache.hudi.common.table.timeline.TimeGenerator;
import org.apache.hudi.common.table.timeline.TimelineLayout;
import org.apache.hudi.common.table.timeline.TimelineMetadataUtils;
import org.apache.hudi.common.util.CleanerUtils;
import org.apache.hudi.common.util.ClusteringUtils;
import org.apache.hudi.common.util.CollectionUtils;
import org.apache.hudi.common.util.Option;
import org.apache.hudi.common.util.StringUtils;
import org.apache.hudi.common.util.ValidationUtils;
import org.apache.hudi.common.util.collection.Pair;
import org.apache.hudi.exception.HoodieException;
import org.apache.hudi.exception.HoodieIOException;
import org.apache.hudi.exception.HoodieTimeTravelException;
import org.apache.hudi.storage.HoodieStorage;
import org.apache.hudi.storage.StoragePath;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TimelineUtils {
    public static final Set<String> NOT_PARSABLE_TIMESTAMPS = new HashSet<String>(3){
        {
            this.add("00000000000000");
            this.add("00000000000001");
            this.add("00000000000002");
        }
    };
    private static final Logger LOG = LoggerFactory.getLogger(TimelineUtils.class);

    public static List<String> getWrittenPartitions(HoodieTimeline timeline) {
        HoodieTimeline timelineToSync = timeline.getWriteTimeline();
        return TimelineUtils.getAffectedPartitions(timelineToSync);
    }

    public static List<String> getDroppedPartitions(HoodieTableMetaClient metaClient, Option<String> lastCommitTimeSynced, Option<String> lastCommitCompletionTimeSynced) {
        HoodieActiveTimeline timeline = lastCommitTimeSynced.isPresent() ? TimelineUtils.getCommitsTimelineAfter(metaClient, lastCommitTimeSynced.get(), lastCommitCompletionTimeSynced) : metaClient.getActiveTimeline();
        HoodieTimeline completedTimeline = timeline.getWriteTimeline().filterCompletedInstants();
        HoodieTimeline replaceCommitTimeline = completedTimeline.getCompletedReplaceTimeline();
        Map<String, String> partitionToLatestDeleteTimestamp = replaceCommitTimeline.getInstantsAsStream().map(instant -> {
            try {
                HoodieReplaceCommitMetadata commitMetadata = HoodieReplaceCommitMetadata.fromBytes(replaceCommitTimeline.getInstantDetails((HoodieInstant)instant).get(), HoodieReplaceCommitMetadata.class);
                return Pair.of(instant, commitMetadata);
            }
            catch (IOException e) {
                throw new HoodieIOException("Failed to get partitions modified at " + instant, e);
            }
        }).filter(pair -> TimelineUtils.isDeletePartition(((HoodieReplaceCommitMetadata)pair.getRight()).getOperationType())).flatMap(pair -> ((HoodieReplaceCommitMetadata)pair.getRight()).getPartitionToReplaceFileIds().keySet().stream().map(partition -> new AbstractMap.SimpleEntry<String, String>((String)partition, ((HoodieInstant)pair.getLeft()).requestedTime()))).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue, (existing, replace) -> replace));
        HoodieTimeline cleanerTimeline = metaClient.getActiveTimeline().getCleanerTimeline().filterCompletedInstants();
        cleanerTimeline.getInstantsAsStream().forEach(instant -> {
            try {
                HoodieCleanMetadata cleanMetadata = TimelineMetadataUtils.deserializeHoodieCleanMetadata(cleanerTimeline.getInstantDetails((HoodieInstant)instant).get());
                cleanMetadata.getPartitionMetadata().forEach((partition, partitionMetadata) -> {
                    if (partitionMetadata.getIsPartitionDeleted().booleanValue()) {
                        partitionToLatestDeleteTimestamp.put((String)partition, instant.requestedTime());
                    }
                });
            }
            catch (IOException e) {
                throw new HoodieIOException("Failed to get partitions cleaned at " + instant, e);
            }
        });
        if (partitionToLatestDeleteTimestamp.isEmpty()) {
            return Collections.emptyList();
        }
        String earliestDeleteTimestamp = partitionToLatestDeleteTimestamp.values().stream().reduce((left, right) -> InstantComparison.compareTimestamps(left, InstantComparison.LESSER_THAN, right) ? left : right).get();
        Map<String, String> partitionToLatestWriteTimestamp = completedTimeline.getInstantsAsStream().filter(instant -> InstantComparison.compareTimestamps(instant.requestedTime(), InstantComparison.GREATER_THAN_OR_EQUALS, earliestDeleteTimestamp)).flatMap(instant -> {
            try {
                HoodieCommitMetadata commitMetadata = TimelineUtils.getCommitMetadata(instant, completedTimeline);
                return commitMetadata.getWritePartitionPaths().stream().map(partition -> new AbstractMap.SimpleEntry<String, String>((String)partition, instant.requestedTime()));
            }
            catch (IOException e) {
                throw new HoodieIOException("Failed to get partitions writes at " + instant, e);
            }
        }).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue, (existing, replace) -> replace));
        return partitionToLatestDeleteTimestamp.entrySet().stream().filter(entry -> !partitionToLatestWriteTimestamp.containsKey(entry.getKey()) || InstantComparison.compareTimestamps((String)entry.getValue(), InstantComparison.GREATER_THAN, (String)partitionToLatestWriteTimestamp.get(entry.getKey()))).map(Map.Entry::getKey).filter(partition -> !partition.isEmpty()).collect(Collectors.toList());
    }

    public static List<String> getAffectedPartitions(HoodieTimeline timeline) {
        return timeline.filterCompletedInstants().getInstantsAsStream().flatMap(s -> {
            switch (s.getAction()) {
                case "commit": 
                case "deltacommit": {
                    try {
                        CommitMetadataSerDe metadataSerDe = TimelineLayout.fromVersion(timeline.getTimelineLayoutVersion()).getCommitMetadataSerDe();
                        HoodieCommitMetadata commitMetadata = metadataSerDe.deserialize((HoodieInstant)s, timeline.getInstantDetails((HoodieInstant)s).get(), HoodieCommitMetadata.class);
                        return commitMetadata.getPartitionToWriteStats().keySet().stream();
                    }
                    catch (IOException e) {
                        throw new HoodieIOException("Failed to get partitions written at " + s, e);
                    }
                }
                case "replacecommit": {
                    try {
                        HoodieReplaceCommitMetadata commitMetadata = HoodieReplaceCommitMetadata.fromBytes(timeline.getInstantDetails((HoodieInstant)s).get(), HoodieReplaceCommitMetadata.class);
                        HashSet<String> partitions = new HashSet<String>();
                        partitions.addAll(commitMetadata.getPartitionToReplaceFileIds().keySet());
                        partitions.addAll(commitMetadata.getPartitionToWriteStats().keySet());
                        return partitions.stream();
                    }
                    catch (IOException e) {
                        throw new HoodieIOException("Failed to get partitions modified at " + s, e);
                    }
                }
                case "clean": {
                    try {
                        HoodieCleanMetadata cleanMetadata = TimelineMetadataUtils.deserializeHoodieCleanMetadata(timeline.getInstantDetails((HoodieInstant)s).get());
                        return cleanMetadata.getPartitionMetadata().keySet().stream();
                    }
                    catch (IOException e) {
                        throw new HoodieIOException("Failed to get partitions cleaned at " + s, e);
                    }
                }
                case "rollback": {
                    try {
                        return TimelineMetadataUtils.deserializeHoodieRollbackMetadata(timeline.getInstantDetails((HoodieInstant)s).get()).getPartitionMetadata().keySet().stream();
                    }
                    catch (IOException e) {
                        throw new HoodieIOException("Failed to get partitions rolledback at " + s, e);
                    }
                }
                case "restore": {
                    try {
                        HoodieRestoreMetadata restoreMetadata = TimelineMetadataUtils.deserializeAvroMetadata(timeline.getInstantDetails((HoodieInstant)s).get(), HoodieRestoreMetadata.class);
                        return restoreMetadata.getHoodieRestoreMetadata().values().stream().flatMap(Collection::stream).flatMap(rollbackMetadata -> rollbackMetadata.getPartitionMetadata().keySet().stream());
                    }
                    catch (IOException e) {
                        throw new HoodieIOException("Failed to get partitions restored at " + s, e);
                    }
                }
                case "savepoint": {
                    try {
                        return TimelineMetadataUtils.deserializeHoodieSavepointMetadata(timeline.getInstantDetails((HoodieInstant)s).get()).getPartitionMetadata().keySet().stream();
                    }
                    catch (IOException e) {
                        throw new HoodieIOException("Failed to get partitions savepoint at " + s, e);
                    }
                }
                case "compaction": {
                    return Stream.empty();
                }
            }
            throw new HoodieIOException("unknown action in timeline " + s.getAction());
        }).distinct().filter(s -> !s.isEmpty()).collect(Collectors.toList());
    }

    public static Option<String> getExtraMetadataFromLatest(HoodieTableMetaClient metaClient, String extraMetadataKey) {
        return metaClient.getCommitsTimeline().filterCompletedInstants().getReverseOrderedInstants().filter(instant -> !TimelineUtils.isClusteringCommit(metaClient, instant)).findFirst().map(instant -> TimelineUtils.getMetadataValue(metaClient, extraMetadataKey, instant)).orElse(Option.empty());
    }

    public static Option<String> getExtraMetadataFromLatestIncludeClustering(HoodieTableMetaClient metaClient, String extraMetadataKey) {
        return metaClient.getCommitsTimeline().filterCompletedInstants().getReverseOrderedInstants().findFirst().map(instant -> TimelineUtils.getMetadataValue(metaClient, extraMetadataKey, instant)).orElse(Option.empty());
    }

    public static Map<String, Option<String>> getAllExtraMetadataForKey(HoodieTableMetaClient metaClient, String extraMetadataKey) {
        return metaClient.getCommitsTimeline().filterCompletedInstants().getReverseOrderedInstants().collect(Collectors.toMap(HoodieInstant::requestedTime, instant -> TimelineUtils.getMetadataValue(metaClient, extraMetadataKey, instant)));
    }

    private static Option<String> getMetadataValue(HoodieTableMetaClient metaClient, String extraMetadataKey, HoodieInstant instant) {
        try {
            LOG.info("reading checkpoint info for:" + instant + " key: " + extraMetadataKey);
            HoodieCommitMetadata commitMetadata = metaClient.getCommitMetadataSerDe().deserialize(instant, metaClient.getCommitsTimeline().getInstantDetails(instant).get(), HoodieCommitMetadata.class);
            return Option.ofNullable(commitMetadata.getExtraMetadata().get(extraMetadataKey));
        }
        catch (IOException e) {
            throw new HoodieIOException("Unable to parse instant metadata " + instant, e);
        }
    }

    public static boolean isClusteringCommit(HoodieTableMetaClient metaClient, HoodieInstant completedInstant) {
        ValidationUtils.checkArgument(completedInstant.isCompleted(), "The instant should be completed for this API");
        try {
            if ("replacecommit".equals(completedInstant.getAction())) {
                HoodieReplaceCommitMetadata replaceMetadata = HoodieReplaceCommitMetadata.fromBytes(metaClient.getActiveTimeline().getInstantDetails(completedInstant).get(), HoodieReplaceCommitMetadata.class);
                return WriteOperationType.CLUSTER.equals((Object)replaceMetadata.getOperationType());
            }
            return false;
        }
        catch (IOException e) {
            throw new HoodieIOException("Unable to read instant information: " + completedInstant + " for " + metaClient.getBasePath(), e);
        }
    }

    public static HoodieTimeline getTimeline(HoodieTableMetaClient metaClient, boolean includeArchivedTimeline) {
        HoodieActiveTimeline activeTimeline = metaClient.getActiveTimeline();
        if (includeArchivedTimeline) {
            HoodieArchivedTimeline archivedTimeline = metaClient.getArchivedTimeline();
            return archivedTimeline.mergeTimeline(activeTimeline);
        }
        return activeTimeline;
    }

    public static HoodieTimeline getCommitsTimelineAfter(HoodieTableMetaClient metaClient, String exclusiveStartInstantTime, Option<String> lastMaxCompletionTime) {
        HoodieTimeline hollowInstantsTimeline;
        HoodieTimeline writeTimeline = metaClient.getActiveTimeline().getWriteTimeline();
        HoodieTimeline timeline = writeTimeline.isBeforeTimelineStarts(exclusiveStartInstantTime) ? metaClient.getArchivedTimeline(exclusiveStartInstantTime).mergeTimeline(writeTimeline) : writeTimeline;
        HoodieTimeline timelineSinceLastSync = timeline.getCommitsTimeline().findInstantsAfter(exclusiveStartInstantTime, Integer.MAX_VALUE);
        if (lastMaxCompletionTime.isPresent() && !(hollowInstantsTimeline = timeline.getCommitsTimeline().filter(s -> InstantComparison.compareTimestamps(s.requestedTime(), InstantComparison.LESSER_THAN, exclusiveStartInstantTime)).filter(s -> InstantComparison.compareTimestamps(s.getCompletionTime(), InstantComparison.GREATER_THAN, (String)lastMaxCompletionTime.get()))).empty()) {
            return timelineSinceLastSync.mergeTimeline(hollowInstantsTimeline);
        }
        return timelineSinceLastSync;
    }

    public static HoodieCommitMetadata getCommitMetadata(HoodieInstant instant, HoodieTimeline timeline) throws IOException {
        byte[] data = timeline.getInstantDetails(instant).get();
        if (instant.getAction().equals("replacecommit") || instant.getAction().equals("clustering")) {
            return HoodieReplaceCommitMetadata.fromBytes(data, HoodieReplaceCommitMetadata.class);
        }
        CommitMetadataSerDe metadataSerDe = TimelineLayout.fromVersion(timeline.getTimelineLayoutVersion()).getCommitMetadataSerDe();
        return metadataSerDe.deserialize(instant, data, HoodieCommitMetadata.class);
    }

    public static Option<HoodieInstant> getEarliestInstantForMetadataArchival(HoodieActiveTimeline dataTableActiveTimeline, boolean shouldArchiveBeyondSavepoint) {
        Option<HoodieInstant> earliestCommit = shouldArchiveBeyondSavepoint ? dataTableActiveTimeline.getTimelineOfActions(CollectionUtils.createSet("commit", "deltacommit", "replacecommit", "clustering", "savepoint")).getFirstNonSavepointCommit() : dataTableActiveTimeline.getCommitsTimeline().firstInstant();
        Option<HoodieInstant> earliestInflight = dataTableActiveTimeline.filterInflightsAndRequested().firstInstant();
        if (earliestCommit.isPresent() && earliestInflight.isPresent()) {
            if (earliestCommit.get().compareTo(earliestInflight.get()) < 0) {
                return earliestCommit;
            }
            return earliestInflight;
        }
        if (earliestCommit.isPresent()) {
            return earliestCommit;
        }
        if (earliestInflight.isPresent()) {
            return earliestInflight;
        }
        return Option.empty();
    }

    public static void validateTimestampAsOf(HoodieTableMetaClient metaClient, String timestampAsOf) {
        String incompleteCommitTime;
        Option<HoodieInstant> firstIncompleteCommit = metaClient.getCommitsTimeline().filterInflightsAndRequested().filter(instant -> !ClusteringUtils.isClusteringInstant(metaClient.getActiveTimeline(), instant, metaClient.getInstantGenerator())).firstInstant();
        if (firstIncompleteCommit.isPresent() && InstantComparison.compareTimestamps(timestampAsOf, InstantComparison.GREATER_THAN_OR_EQUALS, incompleteCommitTime = firstIncompleteCommit.get().requestedTime())) {
            throw new HoodieTimeTravelException(String.format("Time travel's timestamp '%s' must be earlier than the first incomplete commit timestamp '%s'.", timestampAsOf, incompleteCommitTime));
        }
        Option<HoodieInstant> latestCleanOpt = metaClient.getActiveTimeline().getCleanerTimeline().filterCompletedInstants().lastInstant();
        if (latestCleanOpt.isPresent()) {
            try {
                HoodieCleanMetadata cleanMetadata = CleanerUtils.getCleanerMetadata(metaClient, latestCleanOpt.get());
                String earliestCommitToRetain = cleanMetadata.getEarliestCommitToRetain();
                if (!StringUtils.isNullOrEmpty(earliestCommitToRetain)) {
                    ValidationUtils.checkArgument(InstantComparison.compareTimestamps(earliestCommitToRetain, InstantComparison.LESSER_THAN_OR_EQUALS, timestampAsOf), "Cleaner cleaned up the timestamp of interest. Please ensure sufficient commits are retained with cleaner for Timestamp as of query to work");
                } else {
                    Option<HoodieInstant> firstCompletedInstant = metaClient.getActiveTimeline().getWriteTimeline().filterCompletedInstants().firstInstant();
                    if (firstCompletedInstant.isPresent()) {
                        ValidationUtils.checkArgument(InstantComparison.compareTimestamps(firstCompletedInstant.get().requestedTime(), InstantComparison.LESSER_THAN_OR_EQUALS, timestampAsOf), "Please ensure sufficient commits are retained (uncleaned and un-archived) for timestamp as of query to work.");
                    }
                }
            }
            catch (IOException e) {
                throw new HoodieTimeTravelException("Cleaner cleaned up the timestamp of interest. Please ensure sufficient commits are retained with cleaner for Timestamp as of query to work ");
            }
        }
    }

    public static HoodieTimeline handleHollowCommitIfNeeded(HoodieTimeline completedCommitTimeline, HoodieTableMetaClient metaClient, HollowCommitHandling handlingMode) {
        if (handlingMode == HollowCommitHandling.USE_TRANSITION_TIME) {
            return completedCommitTimeline;
        }
        Option<HoodieInstant> firstIncompleteCommit = metaClient.getCommitsTimeline().filterInflightsAndRequested().filter(instant -> !ClusteringUtils.isClusteringInstant(metaClient.getActiveTimeline(), instant, metaClient.getInstantGenerator())).firstInstant();
        boolean noHollowCommit = firstIncompleteCommit.map(i -> completedCommitTimeline.findInstantsAfter(i.requestedTime()).empty()).orElse(true);
        if (noHollowCommit) {
            return completedCommitTimeline;
        }
        String hollowCommitTimestamp = firstIncompleteCommit.get().requestedTime();
        switch (handlingMode) {
            case FAIL: {
                throw new HoodieException(String.format("Found hollow commit: '%s'. Adjust config `%s` accordingly if to avoid throwing this exception.", hollowCommitTimestamp, HoodieCommonConfig.INCREMENTAL_READ_HANDLE_HOLLOW_COMMIT.key()));
            }
            case BLOCK: {
                LOG.warn(String.format("Found hollow commit '%s'. Config `%s` was set to `%s`: no data will be returned beyond '%s' until it's completed.", new Object[]{hollowCommitTimestamp, HoodieCommonConfig.INCREMENTAL_READ_HANDLE_HOLLOW_COMMIT.key(), handlingMode, hollowCommitTimestamp}));
                return completedCommitTimeline.findInstantsBefore(hollowCommitTimestamp);
            }
        }
        throw new HoodieException("Unexpected handling mode: " + (Object)((Object)handlingMode));
    }

    public static Date parseDateFromInstantTime(String timestamp) throws ParseException {
        return HoodieInstantTimeGenerator.parseDateFromInstantTime(timestamp);
    }

    public static Option<Date> parseDateFromInstantTimeSafely(String timestamp) {
        Option<Date> parsedDate;
        try {
            parsedDate = Option.of(HoodieInstantTimeGenerator.parseDateFromInstantTime(timestamp));
        }
        catch (ParseException e) {
            if (NOT_PARSABLE_TIMESTAMPS.contains(timestamp)) {
                parsedDate = Option.of(new Date(Integer.parseInt(timestamp)));
            }
            LOG.warn("Failed to parse timestamp {}: {}", (Object)timestamp, (Object)e.getMessage());
            parsedDate = Option.empty();
        }
        return parsedDate;
    }

    public static String formatDate(Date timestamp) {
        return HoodieInstantTimeGenerator.formatDate(timestamp);
    }

    public static String generateInstantTime(boolean shouldLock, TimeGenerator timeGenerator) {
        return TimelineUtils.generateInstantTime(shouldLock, timeGenerator, 0L);
    }

    public static String generateInstantTime(boolean shouldLock, TimeGenerator timeGenerator, long milliseconds) {
        return HoodieInstantTimeGenerator.createNewInstantTime(shouldLock, timeGenerator, milliseconds);
    }

    public static void deleteInstantFile(HoodieStorage storage, StoragePath metaPath, HoodieInstant instant, InstantFileNameGenerator factory) {
        String filePath = factory.getFileName(instant);
        try {
            storage.deleteFile(new StoragePath(metaPath, filePath));
        }
        catch (IOException e) {
            throw new HoodieIOException("Could not delete instant file" + filePath, e);
        }
    }

    public static HoodieInstant getInflightInstant(HoodieInstant instant, HoodieTableMetaClient metaClient) {
        InstantGenerator factory = metaClient.getInstantGenerator();
        if (metaClient.getTableType() == HoodieTableType.MERGE_ON_READ) {
            HoodieActiveTimeline rawActiveTimeline;
            Option<HoodieInstant> logCompactionInstant;
            if (instant.getAction().equals("commit")) {
                return factory.createNewInstant(HoodieInstant.State.INFLIGHT, "compaction", instant.requestedTime());
            }
            if (instant.getAction().equals("deltacommit") && (logCompactionInstant = Option.fromJavaOptional((rawActiveTimeline = metaClient.getTimelineLayout().getTimelineFactory().createActiveTimeline(metaClient, false)).getInstantsAsStream().filter(hoodieInstant -> hoodieInstant.requestedTime().equals(instant.requestedTime()) && "logcompaction".equals(hoodieInstant.getAction())).findFirst())).isPresent()) {
                return factory.createNewInstant(HoodieInstant.State.INFLIGHT, "logcompaction", instant.requestedTime());
            }
        }
        return factory.createNewInstant(HoodieInstant.State.INFLIGHT, instant.getAction(), instant.requestedTime());
    }

    public static HoodieTimeline concatTimeline(HoodieTimeline timeline1, HoodieTimeline timeline2, HoodieTableMetaClient metaClient) {
        return metaClient.getTimelineLayout().getTimelineFactory().createDefaultTimeline(Stream.concat(timeline1.getInstantsAsStream(), timeline2.getInstantsAsStream()).sorted(), instant -> metaClient.getActiveTimeline().getInstantDetails((HoodieInstant)instant));
    }

    public static boolean isDeletePartition(WriteOperationType operation) {
        return operation == WriteOperationType.DELETE_PARTITION || operation == WriteOperationType.INSERT_OVERWRITE_TABLE || operation == WriteOperationType.INSERT_OVERWRITE;
    }

    public static enum HollowCommitHandling {
        FAIL,
        BLOCK,
        USE_TRANSITION_TIME;

    }
}

