/*
 * Decompiled with CFR 0.152.
 */
package org.kiwiproject.consul.cache;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.collect.ImmutableMap;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.math.BigInteger;
import java.time.Duration;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executors;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Function;
import org.kiwiproject.consul.async.ConsulResponseCallback;
import org.kiwiproject.consul.cache.CacheDescriptor;
import org.kiwiproject.consul.config.CacheConfig;
import org.kiwiproject.consul.model.ConsulResponse;
import org.kiwiproject.consul.monitoring.ClientEventHandler;
import org.kiwiproject.consul.option.ImmutableQueryOptions;
import org.kiwiproject.consul.option.Options;
import org.kiwiproject.consul.option.QueryOptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConsulCache<K, V>
implements AutoCloseable {
    private static final Logger LOG = LoggerFactory.getLogger(ConsulCache.class);
    private final AtomicReference<BigInteger> latestIndex = new AtomicReference<Object>(null);
    private final AtomicLong lastContact = new AtomicLong();
    private final AtomicBoolean isKnownLeader = new AtomicBoolean();
    private final AtomicReference<ConsulResponse.CacheResponseInfo> lastCacheInfo = new AtomicReference<Object>(null);
    private final AtomicReference<ImmutableMap<K, V>> lastResponse = new AtomicReference<Object>(null);
    private final AtomicReference<State> state = new AtomicReference<State>(State.LATENT);
    private final CountDownLatch initLatch = new CountDownLatch(1);
    private final Scheduler scheduler;
    private final CopyOnWriteArrayList<Listener<K, V>> listeners = new CopyOnWriteArrayList();
    private final ReentrantLock listenersStartingLock = new ReentrantLock();
    private final Stopwatch stopWatch = Stopwatch.createUnstarted();
    private final Function<V, K> keyConversion;
    private final CallbackConsumer<V> callBackConsumer;
    private final ConsulResponseCallback<List<V>> responseCallback;
    private final ClientEventHandler eventHandler;
    private final CacheDescriptor cacheDescriptor;

    protected ConsulCache(Function<V, K> keyConversion, CallbackConsumer<V> callbackConsumer, CacheConfig cacheConfig, ClientEventHandler eventHandler, CacheDescriptor cacheDescriptor) {
        this(keyConversion, callbackConsumer, cacheConfig, eventHandler, cacheDescriptor, ConsulCache.createDefault());
    }

    protected ConsulCache(Function<V, K> keyConversion, CallbackConsumer<V> callbackConsumer, CacheConfig cacheConfig, ClientEventHandler eventHandler, CacheDescriptor cacheDescriptor, ScheduledExecutorService callbackScheduleExecutorService) {
        this(keyConversion, callbackConsumer, cacheConfig, eventHandler, cacheDescriptor, new ExternalScheduler(callbackScheduleExecutorService));
    }

    protected ConsulCache(Function<V, K> keyConversion, CallbackConsumer<V> callbackConsumer, CacheConfig cacheConfig, ClientEventHandler eventHandler, CacheDescriptor cacheDescriptor, Scheduler callbackScheduler) {
        Preconditions.checkArgument(Objects.nonNull(keyConversion), "keyConversion must not be null");
        Preconditions.checkArgument(Objects.nonNull(callbackConsumer), "callbackConsumer must not be null");
        Preconditions.checkArgument(Objects.nonNull(cacheConfig), "cacheConfig must not be null");
        Preconditions.checkArgument(Objects.nonNull(eventHandler), "eventHandler must not be null");
        Preconditions.checkArgument(Objects.nonNull(cacheDescriptor), "cacheDescriptor must not be null");
        Preconditions.checkArgument(Objects.nonNull(callbackScheduler), "callbackScheduler must not be null");
        this.keyConversion = keyConversion;
        this.callBackConsumer = callbackConsumer;
        this.eventHandler = eventHandler;
        this.cacheDescriptor = cacheDescriptor;
        this.scheduler = callbackScheduler;
        this.responseCallback = new DefaultConsulResponseCallback(cacheConfig);
    }

    static long computeBackOffDelayMs(CacheConfig cacheConfig) {
        return cacheConfig.getMinimumBackOffDelay().toMillis() + Math.round(Math.random() * (double)cacheConfig.getMaximumBackOffDelay().minus(cacheConfig.getMinimumBackOffDelay()).toMillis());
    }

    public void start() {
        Preconditions.checkState(this.state.compareAndSet(State.LATENT, State.STARTING), "Cannot transition from state %s to %s", (Object)this.state.get(), (Object)State.STARTING);
        this.eventHandler.cacheStart(this.cacheDescriptor);
        this.runCallback();
    }

    public void stop() {
        try {
            this.eventHandler.cacheStop(this.cacheDescriptor);
        }
        catch (RejectedExecutionException ree) {
            LOG.error("Unable to propagate cache stop event. ", (Throwable)ree);
        }
        State previous = this.state.getAndSet(State.STOPPED);
        if (this.stopWatch.isRunning()) {
            this.stopWatch.stop();
        }
        if (previous != State.STOPPED) {
            this.scheduler.shutdownNow();
        }
    }

    @Override
    public void close() {
        this.stop();
    }

    private void runCallback() {
        if (this.isRunning()) {
            this.stopWatch.reset().start();
            this.callBackConsumer.consume(this.latestIndex.get(), this.responseCallback);
        }
    }

    private boolean isRunning() {
        return this.state.get() == State.STARTED || this.state.get() == State.STARTING;
    }

    public boolean awaitInitialized(long timeout2, TimeUnit unit) throws InterruptedException {
        return this.initLatch.await(timeout2, unit);
    }

    public ImmutableMap<K, V> getMap() {
        return this.lastResponse.get();
    }

    public ConsulResponse<ImmutableMap<K, V>> getMapWithMetadata() {
        return new ConsulResponse<ImmutableMap<K, V>>(this.lastResponse.get(), this.lastContact.get(), this.isKnownLeader.get(), this.latestIndex.get(), Optional.ofNullable(this.lastCacheInfo.get()));
    }

    @VisibleForTesting
    ImmutableMap<K, V> convertToMap(ConsulResponse<List<V>> response) {
        if (Objects.isNull(response) || Objects.isNull(response.getResponse()) || response.getResponse().isEmpty()) {
            return ImmutableMap.of();
        }
        ImmutableMap.Builder<K, V> builder = ImmutableMap.builder();
        HashSet<K> keySet = new HashSet<K>();
        for (V v : response.getResponse()) {
            K key = this.keyConversion.apply(v);
            if (!Objects.nonNull(key)) continue;
            if (keySet.contains(key)) {
                LOG.warn("Duplicate service encountered. May differ by tags. Try using more specific tags? {}", key);
                continue;
            }
            builder.put(key, v);
            keySet.add(key);
        }
        return builder.build();
    }

    protected static QueryOptions watchParams(BigInteger index, int blockSeconds, QueryOptions queryOptions) {
        Preconditions.checkArgument(queryOptions.getIndex().isEmpty() && queryOptions.getWait().isEmpty(), "Index and wait cannot be overridden");
        ImmutableQueryOptions.Builder builder = ImmutableQueryOptions.builder().from(ConsulCache.watchDefaultParams(index, blockSeconds)).token(queryOptions.getToken()).consistencyMode(queryOptions.getConsistencyMode()).near(queryOptions.getNear()).datacenter(queryOptions.getDatacenter());
        for (String tag : queryOptions.getTag()) {
            builder.addTag(tag);
        }
        return builder.build();
    }

    private static QueryOptions watchDefaultParams(BigInteger index, int blockSeconds) {
        if (Objects.isNull(index)) {
            return Options.BLANK_QUERY_OPTIONS;
        }
        return QueryOptions.blockSeconds(blockSeconds, index).build();
    }

    protected static Scheduler createDefault() {
        return new DefaultScheduler();
    }

    protected static Scheduler createExternal(ScheduledExecutorService executor) {
        return new ExternalScheduler(executor);
    }

    public boolean addListener(Listener<K, V> listener) {
        this.performListenerActionOptionallyLocking(() -> {
            this.listeners.add(listener);
            if (this.state.get() == State.STARTED) {
                try {
                    listener.notify(this.lastResponse.get());
                }
                catch (RuntimeException e) {
                    LOG.warn("ConsulCache Listener's notify method threw an exception.", (Throwable)e);
                }
            }
        });
        return true;
    }

    private void performListenerActionOptionallyLocking(Runnable action) {
        boolean locked = false;
        if (this.state.get() == State.STARTING) {
            this.listenersStartingLock.lock();
            locked = true;
        }
        try {
            action.run();
        }
        finally {
            if (locked) {
                this.listenersStartingLock.unlock();
            }
        }
    }

    public List<Listener<K, V>> getListeners() {
        return List.copyOf(this.listeners);
    }

    public boolean removeListener(Listener<K, V> listener) {
        return this.listeners.remove(listener);
    }

    public State getState() {
        return this.state.get();
    }

    protected static void checkWatch(int networkReadMillis, int cacheWatchSeconds) {
        if ((long)networkReadMillis <= TimeUnit.SECONDS.toMillis(cacheWatchSeconds)) {
            throw new IllegalArgumentException("Cache watchInterval=" + cacheWatchSeconds + "sec >= networkClientReadTimeout=" + networkReadMillis + "ms. It can cause issues");
        }
    }

    protected static class Scheduler {
        private final ScheduledExecutorService executor;

        public Scheduler(ScheduledExecutorService executor) {
            this.executor = executor;
        }

        void schedule(Runnable r, long delay, TimeUnit unit) {
            this.executor.schedule(r, delay, unit);
        }

        void shutdownNow() {
            this.executor.shutdownNow();
        }
    }

    protected static interface CallbackConsumer<V> {
        public void consume(BigInteger var1, ConsulResponseCallback<List<V>> var2);
    }

    private static class ExternalScheduler
    extends Scheduler {
        public ExternalScheduler(ScheduledExecutorService executor) {
            super(executor);
        }

        @Override
        public void shutdownNow() {
        }
    }

    public static enum State {
        LATENT,
        STARTING,
        STARTED,
        STOPPED;

    }

    class DefaultConsulResponseCallback
    implements ConsulResponseCallback<List<V>> {
        private final CacheConfig cacheConfig;

        public DefaultConsulResponseCallback(CacheConfig cacheConfig) {
            this.cacheConfig = Objects.requireNonNull(cacheConfig);
        }

        @Override
        public void onComplete(ConsulResponse<List<V>> consulResponse) {
            if (this.isNotRunning()) {
                return;
            }
            long elapsedTime = ConsulCache.this.stopWatch.elapsed(TimeUnit.MILLISECONDS);
            this.updateIndex(consulResponse);
            LOG.debug("Consul cache updated for {} (index={}), request duration: {} ms", new Object[]{ConsulCache.this.cacheDescriptor, ConsulCache.this.latestIndex, elapsedTime});
            ImmutableMap full = ConsulCache.this.convertToMap(consulResponse);
            boolean changed = !full.equals(ConsulCache.this.lastResponse.get());
            ConsulCache.this.eventHandler.cachePollingSuccess(ConsulCache.this.cacheDescriptor, changed, elapsedTime);
            if (changed) {
                ConsulCache.this.lastResponse.set(full);
                ConsulCache.this.lastContact.set(consulResponse.getLastContact());
                ConsulCache.this.isKnownLeader.set(consulResponse.isKnownLeader());
                ConsulCache.this.performListenerActionOptionallyLocking(() -> this.notifyListeners(full));
            }
            if (ConsulCache.this.state.compareAndSet(State.STARTING, State.STARTED)) {
                ConsulCache.this.initLatch.countDown();
            }
            Duration timeToWait = this.cacheConfig.getMinimumDurationBetweenRequests();
            Duration minimumDelayOnEmptyResult = this.cacheConfig.getMinimumDurationDelayOnEmptyResult();
            if (this.hasNullOrEmptyResponse(consulResponse) && this.isLongerThan(minimumDelayOnEmptyResult, timeToWait)) {
                timeToWait = minimumDelayOnEmptyResult;
            }
            timeToWait = timeToWait.minusMillis(elapsedTime);
            ConsulCache.this.scheduler.schedule(ConsulCache.this::runCallback, timeToWait.toMillis(), TimeUnit.MILLISECONDS);
        }

        private void updateIndex(ConsulResponse<List<V>> consulResponse) {
            if (Objects.nonNull(consulResponse) && Objects.nonNull(consulResponse.getIndex())) {
                ConsulCache.this.latestIndex.set(consulResponse.getIndex());
            }
        }

        private void notifyListeners(ImmutableMap<K, V> newValues) {
            for (Listener l : ConsulCache.this.listeners) {
                try {
                    l.notify(newValues);
                }
                catch (RuntimeException e) {
                    LOG.warn("ConsulCache Listener's notify method threw an exception.", (Throwable)e);
                }
            }
        }

        private boolean hasNullOrEmptyResponse(ConsulResponse<List<V>> consulResponse) {
            return Objects.isNull(consulResponse.getResponse()) || consulResponse.getResponse().isEmpty();
        }

        private boolean isLongerThan(Duration duration1, Duration duration2) {
            return duration1.compareTo(duration2) > 0;
        }

        @Override
        public void onFailure(Throwable throwable) {
            if (this.isNotRunning()) {
                return;
            }
            ConsulCache.this.eventHandler.cachePollingError(ConsulCache.this.cacheDescriptor, throwable);
            long delayMs = ConsulCache.computeBackOffDelayMs(this.cacheConfig);
            String message = String.format("Error getting response from consul for %s, will retry in %d %s", new Object[]{ConsulCache.this.cacheDescriptor, delayMs, TimeUnit.MILLISECONDS});
            this.cacheConfig.getRefreshErrorLoggingConsumer().accept(LOG, message, throwable);
            ConsulCache.this.scheduler.schedule(ConsulCache.this::runCallback, delayMs, TimeUnit.MILLISECONDS);
        }

        private boolean isNotRunning() {
            return !ConsulCache.this.isRunning();
        }
    }

    private static class DefaultScheduler
    extends Scheduler {
        public DefaultScheduler() {
            super(Executors.newSingleThreadScheduledExecutor(new ThreadFactoryBuilder().setNameFormat("consulCacheScheduledCallback-%d").setDaemon(true).build()));
        }
    }

    public static interface Listener<K, V> {
        public void notify(Map<K, V> var1);
    }
}

