/*
 * Decompiled with CFR 0.152.
 */
package org.openeuler.sun.security.ssl;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.security.CryptoPrimitive;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.security.SignatureException;
import java.security.interfaces.ECPublicKey;
import java.security.spec.ECParameterSpec;
import java.security.spec.ECPoint;
import java.security.spec.ECPublicKeySpec;
import java.security.spec.InvalidKeySpecException;
import java.text.MessageFormat;
import java.util.EnumSet;
import java.util.Locale;
import java.util.Map;
import org.openeuler.sun.misc.HexDumpEncoder;
import org.openeuler.sun.security.ssl.Alert;
import org.openeuler.sun.security.ssl.ClientHandshakeContext;
import org.openeuler.sun.security.ssl.ConnectionContext;
import org.openeuler.sun.security.ssl.ECDHKeyExchange;
import org.openeuler.sun.security.ssl.HandshakeContext;
import org.openeuler.sun.security.ssl.HandshakeOutStream;
import org.openeuler.sun.security.ssl.HandshakeProducer;
import org.openeuler.sun.security.ssl.JsseJce;
import org.openeuler.sun.security.ssl.RSASignature;
import org.openeuler.sun.security.ssl.Record;
import org.openeuler.sun.security.ssl.SSLConsumer;
import org.openeuler.sun.security.ssl.SSLCredentials;
import org.openeuler.sun.security.ssl.SSLHandshake;
import org.openeuler.sun.security.ssl.SSLLogger;
import org.openeuler.sun.security.ssl.SSLPossession;
import org.openeuler.sun.security.ssl.ServerHandshakeContext;
import org.openeuler.sun.security.ssl.SignatureScheme;
import org.openeuler.sun.security.ssl.SupportedGroupsExtension;
import org.openeuler.sun.security.ssl.Utilities;
import org.openeuler.sun.security.ssl.X509Authentication;

final class ECDHServerKeyExchange {
    static final SSLConsumer ecdheHandshakeConsumer = new ECDHServerKeyExchangeConsumer();
    static final HandshakeProducer ecdheHandshakeProducer = new ECDHServerKeyExchangeProducer();

    ECDHServerKeyExchange() {
    }

    private static final class ECDHServerKeyExchangeConsumer
    implements SSLConsumer {
        private ECDHServerKeyExchangeConsumer() {
        }

        @Override
        public void consume(ConnectionContext context, ByteBuffer message) throws IOException {
            ClientHandshakeContext chc = (ClientHandshakeContext)context;
            ECDHServerKeyExchangeMessage skem = new ECDHServerKeyExchangeMessage(chc, message);
            if (SSLLogger.isOn && SSLLogger.isOn("ssl,handshake")) {
                SSLLogger.fine("Consuming ECDH ServerKeyExchange handshake message", skem);
            }
            if (chc.algorithmConstraints != null && !chc.algorithmConstraints.permits(EnumSet.of(CryptoPrimitive.KEY_AGREEMENT), skem.publicKey)) {
                throw chc.conContext.fatal(Alert.INSUFFICIENT_SECURITY, "ECDH ServerKeyExchange does not comply to algorithm constraints");
            }
            chc.handshakeCredentials.add(new ECDHKeyExchange.ECDHECredentials(skem.publicKey, skem.namedGroup));
        }
    }

    private static final class ECDHServerKeyExchangeProducer
    implements HandshakeProducer {
        private ECDHServerKeyExchangeProducer() {
        }

        @Override
        public byte[] produce(ConnectionContext context, SSLHandshake.HandshakeMessage message) throws IOException {
            ServerHandshakeContext shc = (ServerHandshakeContext)context;
            ECDHServerKeyExchangeMessage skem = new ECDHServerKeyExchangeMessage(shc);
            if (SSLLogger.isOn && SSLLogger.isOn("ssl,handshake")) {
                SSLLogger.fine("Produced ECDH ServerKeyExchange handshake message", skem);
            }
            skem.write(shc.handshakeOutput);
            shc.handshakeOutput.flush();
            return null;
        }
    }

    private static final class ECDHServerKeyExchangeMessage
    extends SSLHandshake.HandshakeMessage {
        private static final byte CURVE_NAMED_CURVE = 3;
        private final SupportedGroupsExtension.NamedGroup namedGroup;
        private final byte[] publicPoint;
        private final byte[] paramsSignature;
        private final ECPublicKey publicKey;
        private final boolean useExplicitSigAlgorithm;
        private final SignatureScheme signatureScheme;

        ECDHServerKeyExchangeMessage(HandshakeContext handshakeContext) throws IOException {
            super(handshakeContext);
            ServerHandshakeContext shc = (ServerHandshakeContext)handshakeContext;
            ECDHKeyExchange.ECDHEPossession ecdhePossession = null;
            X509Authentication.X509Possession x509Possession = null;
            for (SSLPossession possession : shc.handshakePossessions) {
                if (possession instanceof ECDHKeyExchange.ECDHEPossession) {
                    ecdhePossession = (ECDHKeyExchange.ECDHEPossession)possession;
                    if (x509Possession == null) continue;
                    break;
                }
                if (!(possession instanceof X509Authentication.X509Possession)) continue;
                x509Possession = (X509Authentication.X509Possession)possession;
                if (ecdhePossession == null) continue;
                break;
            }
            if (ecdhePossession == null) {
                throw shc.conContext.fatal(Alert.ILLEGAL_PARAMETER, "No ECDHE credentials negotiated for server key exchange");
            }
            this.publicKey = ecdhePossession.publicKey;
            ECParameterSpec params = this.publicKey.getParams();
            ECPoint point = this.publicKey.getW();
            this.publicPoint = JsseJce.encodePoint(point, params.getCurve());
            this.namedGroup = SupportedGroupsExtension.NamedGroup.valueOf(params);
            if (this.namedGroup == null || this.namedGroup.oid == null) {
                throw shc.conContext.fatal(Alert.ILLEGAL_PARAMETER, "Unnamed EC parameter spec: " + params);
            }
            if (x509Possession == null) {
                this.paramsSignature = null;
                this.signatureScheme = null;
                this.useExplicitSigAlgorithm = false;
            } else {
                this.useExplicitSigAlgorithm = shc.negotiatedProtocol.useTLS12PlusSpec();
                Signature signer = null;
                if (this.useExplicitSigAlgorithm) {
                    Map.Entry<SignatureScheme, Signature> schemeAndSigner = SignatureScheme.getSignerOfPreferableAlgorithm(shc.peerRequestedSignatureSchemes, x509Possession, shc.negotiatedProtocol);
                    if (schemeAndSigner == null) {
                        throw shc.conContext.fatal(Alert.INTERNAL_ERROR, "No supported signature algorithm for " + x509Possession.popPrivateKey.getAlgorithm() + "  key");
                    }
                    this.signatureScheme = schemeAndSigner.getKey();
                    signer = schemeAndSigner.getValue();
                } else {
                    this.signatureScheme = null;
                    try {
                        signer = ECDHServerKeyExchangeMessage.getSignature(x509Possession.popPrivateKey.getAlgorithm(), x509Possession.popPrivateKey);
                    }
                    catch (InvalidKeyException | NoSuchAlgorithmException e) {
                        throw shc.conContext.fatal(Alert.INTERNAL_ERROR, "Unsupported signature algorithm: " + x509Possession.popPrivateKey.getAlgorithm(), e);
                    }
                }
                byte[] signature = null;
                try {
                    ECDHServerKeyExchangeMessage.updateSignature(signer, shc.clientHelloRandom.randomBytes, shc.serverHelloRandom.randomBytes, this.namedGroup.id, this.publicPoint);
                    signature = signer.sign();
                }
                catch (SignatureException ex) {
                    throw shc.conContext.fatal(Alert.INTERNAL_ERROR, "Failed to sign ecdhe parameters: " + x509Possession.popPrivateKey.getAlgorithm(), ex);
                }
                this.paramsSignature = signature;
            }
        }

        ECDHServerKeyExchangeMessage(HandshakeContext handshakeContext, ByteBuffer m) throws IOException {
            super(handshakeContext);
            Signature signer;
            ClientHandshakeContext chc = (ClientHandshakeContext)handshakeContext;
            byte curveType = (byte)Record.getInt8(m);
            if (curveType != 3) {
                throw chc.conContext.fatal(Alert.ILLEGAL_PARAMETER, "Unsupported ECCurveType: " + curveType);
            }
            int namedGroupId = Record.getInt16(m);
            this.namedGroup = SupportedGroupsExtension.NamedGroup.valueOf(namedGroupId);
            if (this.namedGroup == null) {
                throw chc.conContext.fatal(Alert.ILLEGAL_PARAMETER, "Unknown named group ID: " + namedGroupId);
            }
            if (!SupportedGroupsExtension.SupportedGroups.isSupported(this.namedGroup)) {
                throw chc.conContext.fatal(Alert.ILLEGAL_PARAMETER, "Unsupported named group: " + (Object)((Object)this.namedGroup));
            }
            if (this.namedGroup.oid == null) {
                throw chc.conContext.fatal(Alert.ILLEGAL_PARAMETER, "Unknown named EC curve: " + (Object)((Object)this.namedGroup));
            }
            ECParameterSpec parameters = JsseJce.getECParameterSpec(this.namedGroup.oid);
            if (parameters == null) {
                throw chc.conContext.fatal(Alert.ILLEGAL_PARAMETER, "No supported EC parameter: " + (Object)((Object)this.namedGroup));
            }
            this.publicPoint = Record.getBytes8(m);
            if (this.publicPoint.length == 0) {
                throw chc.conContext.fatal(Alert.ILLEGAL_PARAMETER, "Insufficient ECPoint data: " + (Object)((Object)this.namedGroup));
            }
            ECPublicKey ecPublicKey = null;
            try {
                ECPoint point = JsseJce.decodePoint(this.publicPoint, parameters.getCurve());
                KeyFactory factory2 = JsseJce.getKeyFactory("EC");
                ecPublicKey = (ECPublicKey)factory2.generatePublic(new ECPublicKeySpec(point, parameters));
            }
            catch (IOException | NoSuchAlgorithmException | InvalidKeySpecException ex) {
                throw chc.conContext.fatal(Alert.ILLEGAL_PARAMETER, "Invalid ECPoint: " + (Object)((Object)this.namedGroup), ex);
            }
            this.publicKey = ecPublicKey;
            X509Authentication.X509Credentials x509Credentials = null;
            for (SSLCredentials cd : chc.handshakeCredentials) {
                if (!(cd instanceof X509Authentication.X509Credentials)) continue;
                x509Credentials = (X509Authentication.X509Credentials)cd;
                break;
            }
            if (x509Credentials == null) {
                if (m.hasRemaining()) {
                    throw chc.conContext.fatal(Alert.HANDSHAKE_FAILURE, "Invalid DH ServerKeyExchange: unknown extra data");
                }
                this.signatureScheme = null;
                this.paramsSignature = null;
                this.useExplicitSigAlgorithm = false;
                return;
            }
            this.useExplicitSigAlgorithm = chc.negotiatedProtocol.useTLS12PlusSpec();
            if (this.useExplicitSigAlgorithm) {
                int ssid = Record.getInt16(m);
                this.signatureScheme = SignatureScheme.valueOf(ssid);
                if (this.signatureScheme == null) {
                    throw chc.conContext.fatal(Alert.HANDSHAKE_FAILURE, "Invalid signature algorithm (" + ssid + ") used in ECDH ServerKeyExchange handshake message");
                }
                if (!chc.localSupportedSignAlgs.contains((Object)this.signatureScheme)) {
                    throw chc.conContext.fatal(Alert.HANDSHAKE_FAILURE, "Unsupported signature algorithm (" + this.signatureScheme.name + ") used in ECDH ServerKeyExchange handshake message");
                }
            } else {
                this.signatureScheme = null;
            }
            this.paramsSignature = Record.getBytes16(m);
            if (this.useExplicitSigAlgorithm) {
                try {
                    signer = this.signatureScheme.getVerifier(x509Credentials.popPublicKey);
                }
                catch (InvalidAlgorithmParameterException | InvalidKeyException | NoSuchAlgorithmException nsae) {
                    throw chc.conContext.fatal(Alert.INTERNAL_ERROR, "Unsupported signature algorithm: " + this.signatureScheme.name, nsae);
                }
            }
            try {
                signer = ECDHServerKeyExchangeMessage.getSignature(x509Credentials.popPublicKey.getAlgorithm(), x509Credentials.popPublicKey);
            }
            catch (InvalidKeyException | NoSuchAlgorithmException e) {
                throw chc.conContext.fatal(Alert.INTERNAL_ERROR, "Unsupported signature algorithm: " + x509Credentials.popPublicKey.getAlgorithm(), e);
            }
            try {
                ECDHServerKeyExchangeMessage.updateSignature(signer, chc.clientHelloRandom.randomBytes, chc.serverHelloRandom.randomBytes, this.namedGroup.id, this.publicPoint);
                if (!signer.verify(this.paramsSignature)) {
                    throw chc.conContext.fatal(Alert.HANDSHAKE_FAILURE, "Invalid ECDH ServerKeyExchange signature");
                }
            }
            catch (SignatureException ex) {
                throw chc.conContext.fatal(Alert.HANDSHAKE_FAILURE, "Cannot verify ECDH ServerKeyExchange signature", ex);
            }
        }

        @Override
        public SSLHandshake handshakeType() {
            return SSLHandshake.SERVER_KEY_EXCHANGE;
        }

        @Override
        public int messageLength() {
            int sigLen = 0;
            if (this.paramsSignature != null) {
                sigLen = 2 + this.paramsSignature.length;
                if (this.useExplicitSigAlgorithm) {
                    sigLen += SignatureScheme.sizeInRecord();
                }
            }
            return 4 + this.publicPoint.length + sigLen;
        }

        @Override
        public void send(HandshakeOutStream hos) throws IOException {
            hos.putInt8(3);
            hos.putInt16(this.namedGroup.id);
            hos.putBytes8(this.publicPoint);
            if (this.paramsSignature != null) {
                if (this.useExplicitSigAlgorithm) {
                    hos.putInt16(this.signatureScheme.id);
                }
                hos.putBytes16(this.paramsSignature);
            }
        }

        public String toString() {
            if (this.useExplicitSigAlgorithm) {
                MessageFormat messageFormat = new MessageFormat("\"ECDH ServerKeyExchange\": '{'\n  \"parameters\": '{'\n    \"named group\": \"{0}\"\n    \"ecdh public\": '{'\n{1}\n    '}',\n  '}',\n  \"digital signature\":  '{'\n    \"signature algorithm\": \"{2}\"\n    \"signature\": '{'\n{3}\n    '}',\n  '}'\n'}'", Locale.ENGLISH);
                HexDumpEncoder hexEncoder = new HexDumpEncoder();
                Object[] messageFields = new Object[]{this.namedGroup.name, Utilities.indent(hexEncoder.encodeBuffer(this.publicPoint), "      "), this.signatureScheme.name, Utilities.indent(hexEncoder.encodeBuffer(this.paramsSignature), "      ")};
                return messageFormat.format(messageFields);
            }
            if (this.paramsSignature != null) {
                MessageFormat messageFormat = new MessageFormat("\"ECDH ServerKeyExchange\": '{'\n  \"parameters\":  '{'\n    \"named group\": \"{0}\"\n    \"ecdh public\": '{'\n{1}\n    '}',\n  '}',\n  \"signature\": '{'\n{2}\n  '}'\n'}'", Locale.ENGLISH);
                HexDumpEncoder hexEncoder = new HexDumpEncoder();
                Object[] messageFields = new Object[]{this.namedGroup.name, Utilities.indent(hexEncoder.encodeBuffer(this.publicPoint), "      "), Utilities.indent(hexEncoder.encodeBuffer(this.paramsSignature), "    ")};
                return messageFormat.format(messageFields);
            }
            MessageFormat messageFormat = new MessageFormat("\"ECDH ServerKeyExchange\": '{'\n  \"parameters\":  '{'\n    \"named group\": \"{0}\"\n    \"ecdh public\": '{'\n{1}\n    '}',\n  '}'\n'}'", Locale.ENGLISH);
            HexDumpEncoder hexEncoder = new HexDumpEncoder();
            Object[] messageFields = new Object[]{this.namedGroup.name, Utilities.indent(hexEncoder.encodeBuffer(this.publicPoint), "      ")};
            return messageFormat.format(messageFields);
        }

        private static Signature getSignature(String keyAlgorithm, Key key) throws NoSuchAlgorithmException, InvalidKeyException {
            Signature signer = null;
            switch (keyAlgorithm) {
                case "EC": {
                    signer = JsseJce.getSignature("SHA1withECDSA");
                    break;
                }
                case "RSA": {
                    signer = RSASignature.getInstance();
                    break;
                }
                default: {
                    throw new NoSuchAlgorithmException("neither an RSA or a EC key : " + keyAlgorithm);
                }
            }
            if (signer != null) {
                if (key instanceof PublicKey) {
                    signer.initVerify((PublicKey)key);
                } else {
                    signer.initSign((PrivateKey)key);
                }
            }
            return signer;
        }

        private static void updateSignature(Signature sig, byte[] clntNonce, byte[] svrNonce, int namedGroupId, byte[] publicPoint) throws SignatureException {
            sig.update(clntNonce);
            sig.update(svrNonce);
            sig.update((byte)3);
            sig.update((byte)(namedGroupId >> 8 & 0xFF));
            sig.update((byte)(namedGroupId & 0xFF));
            sig.update((byte)publicPoint.length);
            sig.update(publicPoint);
        }
    }
}

