/*
 * Decompiled with CFR 0.152.
 */
package org.jgroups.protocols.raft;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Map;
import java.util.Objects;
import java.util.function.ObjLongConsumer;
import org.fusesource.leveldbjni.JniDBFactory;
import org.iq80.leveldb.DB;
import org.iq80.leveldb.DBIterator;
import org.iq80.leveldb.Options;
import org.iq80.leveldb.WriteBatch;
import org.iq80.leveldb.WriteOptions;
import org.jgroups.Address;
import org.jgroups.logging.LogFactory;
import org.jgroups.protocols.raft.Log;
import org.jgroups.protocols.raft.LogEntries;
import org.jgroups.protocols.raft.LogEntry;
import org.jgroups.raft.util.LongHelper;
import org.jgroups.util.Util;

public class LevelDBLog
implements Log {
    protected final org.jgroups.logging.Log log = LogFactory.getLog(this.getClass());
    private static final byte[] FIRSTAPPENDED = "FA".getBytes();
    private static final byte[] LASTAPPENDED = "LA".getBytes();
    private static final byte[] CURRENTTERM = "CT".getBytes();
    private static final byte[] COMMITINDEX = "CX".getBytes();
    private static final byte[] VOTEDFOR = "VF".getBytes();
    private static final byte[] SNAPSHOT = "SN".getBytes();
    private DB db;
    private File dbFileName;
    private long currentTerm;
    private Address votedFor;
    private long firstAppended;
    private long commitIndex;
    private long lastAppended;
    private final WriteOptions write_options = new WriteOptions();

    @Override
    public void init(String log_name, Map<String, String> args) throws Exception {
        Options options = new Options().createIfMissing(true);
        this.dbFileName = new File(log_name);
        this.db = JniDBFactory.factory.open(this.dbFileName, options);
        this.log.trace("opened %s", this.db);
        if (this.isANewRAFTLog()) {
            this.log.trace("log %s is new, must be initialized", this.dbFileName);
            this.initLogWithMetadata();
        } else {
            this.log.trace("log %s exists, does not have to be initialized", this.dbFileName);
            this.readMetadataFromLog();
        }
        this.checkForConsistency();
    }

    @Override
    public Log useFsync(boolean f) {
        this.write_options.sync(f);
        return this;
    }

    @Override
    public boolean useFsync() {
        return this.write_options.sync();
    }

    @Override
    public void close() throws IOException {
        this.log.trace("closing DB: %s", this.db);
        Util.close((Closeable)this.db);
        this.votedFor = null;
        this.firstAppended = 0L;
        this.lastAppended = 0L;
        this.commitIndex = 0L;
        this.currentTerm = 0L;
    }

    @Override
    public void delete() throws IOException {
        Util.close((Closeable)this);
        this.log.trace("deleting DB directory: %s", this.dbFileName);
        JniDBFactory.factory.destroy(this.dbFileName, new Options());
    }

    @Override
    public long firstAppended() {
        return this.firstAppended;
    }

    @Override
    public long commitIndex() {
        return this.commitIndex;
    }

    @Override
    public long lastAppended() {
        return this.lastAppended;
    }

    @Override
    public long currentTerm() {
        return this.currentTerm;
    }

    @Override
    public Address votedFor() {
        return this.votedFor;
    }

    @Override
    public Log commitIndex(long new_index) {
        if (new_index == this.commitIndex) {
            return this;
        }
        this.log.trace("Updating commit index: %d", new_index);
        this.db.put(COMMITINDEX, LongHelper.fromLongToByteArray(new_index));
        this.commitIndex = new_index;
        return this;
    }

    @Override
    public Log currentTerm(long new_term) {
        if (new_term == this.currentTerm) {
            return this;
        }
        this.log.trace("Updating current term: %d", new_term);
        this.db.put(CURRENTTERM, LongHelper.fromLongToByteArray(new_term));
        this.currentTerm = new_term;
        return this;
    }

    @Override
    public Log votedFor(Address member) {
        if (Objects.equals(member, this.votedFor)) {
            return this;
        }
        try {
            this.log.debug("Updating voted for: %s", member);
            this.db.put(VOTEDFOR, Util.objectToByteBuffer(member));
            this.votedFor = member;
        }
        catch (Exception exception) {
            // empty catch block
        }
        return this;
    }

    @Override
    public void setSnapshot(ByteBuffer sn) {
        byte[] snapshot;
        if (sn.isDirect()) {
            snapshot = Util.bufferToArray(sn);
        } else if (sn.arrayOffset() > 0 || sn.capacity() != sn.remaining()) {
            int len = sn.remaining();
            snapshot = new byte[len];
            System.arraycopy(sn.array(), sn.arrayOffset(), snapshot, 0, len);
        } else {
            snapshot = sn.array();
        }
        this.db.put(SNAPSHOT, snapshot);
    }

    @Override
    public ByteBuffer getSnapshot() {
        byte[] snapshot = this.db.get(SNAPSHOT);
        return snapshot != null ? ByteBuffer.wrap(snapshot) : null;
    }

    @Override
    public long append(long index, LogEntries entries) {
        this.log.trace("Appending %d entries", entries.size());
        long new_last_appended = -1L;
        try (WriteBatch batch = this.db.createWriteBatch();){
            for (LogEntry entry : entries) {
                this.appendEntry(index, entry, batch);
                new_last_appended = index++;
                this.updateCurrentTerm(entry.term, batch);
            }
            if (new_last_appended >= 0L) {
                this.updateLastAppended(new_last_appended, batch);
            }
            this.log.trace("Flushing batch to DB: %s", batch);
            this.db.write(batch, this.write_options);
        }
        catch (Exception exception) {
            // empty catch block
        }
        return this.lastAppended;
    }

    @Override
    public LogEntry get(long index) {
        byte[] entryBytes = this.db.get(LongHelper.fromLongToByteArray(index));
        try {
            return entryBytes != null ? (LogEntry)Util.streamableFromByteBuffer(LogEntry.class, entryBytes) : null;
        }
        catch (Exception ex) {
            throw new RuntimeException(String.format("getting log entry at index %d failed", index), ex);
        }
    }

    @Override
    public void forEach(ObjLongConsumer<LogEntry> function, long start_index, long end_index) {
        start_index = Math.max(start_index, Math.max(this.firstAppended, 1L));
        end_index = Math.min(end_index, this.lastAppended);
        DBIterator it = this.db.iterator();
        it.seek(LongHelper.fromLongToByteArray(start_index));
        for (long i = start_index; i <= end_index && it.hasNext(); ++i) {
            Map.Entry e = (Map.Entry)it.next();
            try {
                LogEntry l = (LogEntry)Util.streamableFromByteBuffer(LogEntry.class, (byte[])e.getValue());
                function.accept(l, i);
                continue;
            }
            catch (Exception ex) {
                throw new RuntimeException("failed deserializing LogRecord " + i, ex);
            }
        }
    }

    @Override
    public void forEach(ObjLongConsumer<LogEntry> function) {
        this.forEach(function, Math.max(1L, this.firstAppended), this.lastAppended);
    }

    @Override
    public long sizeInBytes() {
        long size = 0L;
        long start_index = Math.max(this.firstAppended, 1L);
        DBIterator it = this.db.iterator();
        it.seek(LongHelper.fromLongToByteArray(start_index));
        for (long i = start_index; i <= this.lastAppended && it.hasNext(); ++i) {
            Map.Entry e = (Map.Entry)it.next();
            byte[] v = (byte[])e.getValue();
            size += v != null ? (long)v.length : 0L;
        }
        return size;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void truncate(long index_exclusive) {
        if (index_exclusive < this.firstAppended) {
            return;
        }
        if (index_exclusive > this.commitIndex) {
            this.log.warn("upto_index (%d) is higher than commit-index (%d); only truncating up to %d", index_exclusive, this.commitIndex, this.commitIndex);
            index_exclusive = this.commitIndex;
        }
        WriteBatch batch = null;
        try {
            batch = this.db.createWriteBatch();
            for (long index = this.firstAppended; index < index_exclusive; ++index) {
                batch.delete(LongHelper.fromLongToByteArray(index));
            }
            batch.put(FIRSTAPPENDED, LongHelper.fromLongToByteArray(index_exclusive));
            if (this.lastAppended < index_exclusive) {
                this.lastAppended = index_exclusive;
                batch.put(LASTAPPENDED, LongHelper.fromLongToByteArray(index_exclusive));
            }
            this.db.write(batch, this.write_options);
            this.firstAppended = index_exclusive;
        }
        finally {
            Util.close((Closeable)batch);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void reinitializeTo(long index, LogEntry le) throws Exception {
        WriteBatch batch = null;
        try {
            batch = this.db.createWriteBatch();
            for (long i = this.firstAppended; i <= this.lastAppended; ++i) {
                batch.delete(LongHelper.fromLongToByteArray(i));
            }
            this.appendEntry(index, le, batch);
            byte[] idx = LongHelper.fromLongToByteArray(index);
            batch.put(FIRSTAPPENDED, idx);
            batch.put(COMMITINDEX, idx);
            batch.put(LASTAPPENDED, idx);
            batch.put(CURRENTTERM, LongHelper.fromLongToByteArray(le.term()));
            this.commitIndex = this.lastAppended = index;
            this.firstAppended = this.lastAppended;
            this.currentTerm = le.term();
            this.db.write(batch, this.write_options);
        }
        finally {
            Util.close((Closeable)batch);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void deleteAllEntriesStartingFrom(long start_index) {
        if (start_index < this.firstAppended || start_index > this.lastAppended) {
            return;
        }
        WriteBatch batch = null;
        try {
            batch = this.db.createWriteBatch();
            for (long index = start_index; index <= this.lastAppended; ++index) {
                batch.delete(LongHelper.fromLongToByteArray(index));
            }
            LogEntry last = this.get(start_index - 1L);
            if (last == null) {
                this.updateCurrentTerm(0L, batch);
            } else {
                this.updateCurrentTerm(last.term, batch);
            }
            this.updateLastAppended(start_index - 1L, batch);
            if (this.commitIndex > this.lastAppended) {
                this.commitIndex(this.lastAppended);
            }
            this.db.write(batch, this.write_options);
        }
        finally {
            Util.close((Closeable)batch);
        }
    }

    public byte[] print(byte[] bytes) {
        return this.db.get(bytes);
    }

    public void printMetadata() throws Exception {
        this.log.info("-----------------");
        this.log.info("RAFT Log Metadata");
        this.log.info("-----------------");
        byte[] firstAppendedBytes = this.db.get(FIRSTAPPENDED);
        this.log.info("First Appended: %d", LongHelper.fromByteArrayToLong(firstAppendedBytes));
        byte[] lastAppendedBytes = this.db.get(LASTAPPENDED);
        this.log.info("Last Appended: %d", LongHelper.fromByteArrayToLong(lastAppendedBytes));
        byte[] currentTermBytes = this.db.get(CURRENTTERM);
        this.log.info("Current Term: %d", LongHelper.fromByteArrayToLong(currentTermBytes));
        byte[] commitIndexBytes = this.db.get(COMMITINDEX);
        this.log.info("Commit Index: %d", LongHelper.fromByteArrayToLong(commitIndexBytes));
        Address votedForTmp = (Address)Util.objectFromByteBuffer(this.db.get(VOTEDFOR));
        this.log.info("Voted for: %s", votedForTmp);
    }

    public String toString() {
        return String.format("first=%d, commit=%d, last-appended=%d, term=%d (size=%d)", this.firstAppended, this.commitIndex, this.lastAppended, this.currentTerm, this.size());
    }

    private void appendEntry(long index, LogEntry entry, WriteBatch batch) throws Exception {
        this.log.trace("Appending entry %d: %s", index, entry);
        batch.put(LongHelper.fromLongToByteArray(index), Util.streamableToByteBuffer(entry));
    }

    private void updateCurrentTerm(long new_term, WriteBatch batch) {
        if (new_term == this.currentTerm) {
            return;
        }
        this.log.trace("Updating currentTerm: %d", new_term);
        batch.put(CURRENTTERM, LongHelper.fromLongToByteArray(new_term));
        this.currentTerm = new_term;
    }

    private void updateLastAppended(long new_last_appended, WriteBatch batch) {
        if (new_last_appended == this.lastAppended) {
            return;
        }
        this.log.trace("Updating lastAppended: %d", new_last_appended);
        batch.put(LASTAPPENDED, LongHelper.fromLongToByteArray(new_last_appended));
        this.lastAppended = new_last_appended;
    }

    private boolean isANewRAFTLog() {
        return this.db.get(FIRSTAPPENDED) == null;
    }

    private void initLogWithMetadata() {
        this.log.debug("Initializing log with empty Metadata");
        WriteBatch batch = this.db.createWriteBatch();
        try {
            batch.put(FIRSTAPPENDED, LongHelper.fromLongToByteArray(0L));
            batch.put(LASTAPPENDED, LongHelper.fromLongToByteArray(0L));
            batch.put(CURRENTTERM, LongHelper.fromLongToByteArray(0L));
            batch.put(COMMITINDEX, LongHelper.fromLongToByteArray(0L));
            this.db.write(batch, this.write_options);
        }
        catch (Exception ex) {
            ex.printStackTrace();
        }
        finally {
            try {
                batch.close();
            }
            catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    private void readMetadataFromLog() throws Exception {
        this.firstAppended = LongHelper.fromByteArrayToLong(this.db.get(FIRSTAPPENDED));
        this.lastAppended = LongHelper.fromByteArrayToLong(this.db.get(LASTAPPENDED));
        this.currentTerm = LongHelper.fromByteArrayToLong(this.db.get(CURRENTTERM));
        this.commitIndex = LongHelper.fromByteArrayToLong(this.db.get(COMMITINDEX));
        this.votedFor = (Address)Util.objectFromByteBuffer(this.db.get(VOTEDFOR));
        this.log.debug("read metadata from log: firstAppended=%d, lastAppended=%d, currentTerm=%d, commitIndex=%d, votedFor=%s", this.firstAppended, this.lastAppended, this.currentTerm, this.commitIndex, this.votedFor);
    }

    private void checkForConsistency() throws Exception {
        long loggedFirstAppended = LongHelper.fromByteArrayToLong(this.db.get(FIRSTAPPENDED));
        this.log.trace("FirstAppended in DB is: %d", loggedFirstAppended);
        long loggedLastAppended = LongHelper.fromByteArrayToLong(this.db.get(LASTAPPENDED));
        this.log.trace("LastAppended in DB is: %d", loggedLastAppended);
        long loggedCurrentTerm = LongHelper.fromByteArrayToLong(this.db.get(CURRENTTERM));
        this.log.trace("CurrentTerm in DB is: %d", loggedCurrentTerm);
        long loggedCommitIndex = LongHelper.fromByteArrayToLong(this.db.get(COMMITINDEX));
        this.log.trace("CommitIndex in DB is: %d", loggedCommitIndex);
        Address loggedVotedForAddress = (Address)Util.objectFromByteBuffer(this.db.get(VOTEDFOR));
        this.log.trace("VotedFor in DB is: %s", loggedVotedForAddress);
        assert (this.firstAppended == loggedFirstAppended);
        assert (this.lastAppended == loggedLastAppended);
        assert (this.currentTerm == loggedCurrentTerm);
        assert (this.commitIndex == loggedCommitIndex);
        assert (this.votedFor == null || this.votedFor.equals(loggedVotedForAddress));
        LogEntry lastAppendedEntry = this.get(this.lastAppended);
        assert (lastAppendedEntry == null || lastAppendedEntry.term <= this.currentTerm);
        assert (this.firstAppended <= this.commitIndex) : String.format("first=%d, commit=%d", this.firstAppended, this.commitIndex);
        assert (this.commitIndex <= this.lastAppended) : String.format("commit=%d, last=%d", this.commitIndex, this.lastAppended);
    }
}

