/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.support.processor.idempotent;

import java.io.Closeable;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.camel.RuntimeCamelException;
import org.apache.camel.api.management.ManagedAttribute;
import org.apache.camel.api.management.ManagedOperation;
import org.apache.camel.api.management.ManagedResource;
import org.apache.camel.spi.Configurer;
import org.apache.camel.spi.IdempotentRepository;
import org.apache.camel.spi.Metadata;
import org.apache.camel.support.LRUCache;
import org.apache.camel.support.LRUCacheFactory;
import org.apache.camel.support.service.ServiceSupport;
import org.apache.camel.util.FileUtil;
import org.apache.camel.util.IOHelper;
import org.apache.camel.util.ObjectHelper;
import org.apache.camel.util.Scanner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Metadata(label="bean", description="A file based idempotent repository. Comes with 1st-level in-memory cache for fast check of the most frequently used keys.", annotations={"interfaceName=org.apache.camel.spi.IdempotentRepository"})
@Configurer(metadataOnly=true)
@ManagedResource(description="File based idempotent repository")
public class FileIdempotentRepository
extends ServiceSupport
implements IdempotentRepository {
    private static final Logger LOG = LoggerFactory.getLogger(FileIdempotentRepository.class);
    private static final String STORE_DELIMITER = "\n";
    private final AtomicBoolean init = new AtomicBoolean();
    private Map<String, Object> cache;
    private final Lock lock = new ReentrantLock();
    private final Lock cacheAndStoreLock = new ReentrantLock();
    @Metadata(description="The maximum size of the 1st-level in-memory cache", defaultValue="1000")
    private int cacheSize;
    @Metadata(description="File name of the repository (incl directory)", required=true)
    private File fileStore;
    @Metadata(description="The maximum file size for the file store in bytes. The default value is 32mb", defaultValue="32768000")
    private long maxFileStoreSize = 32768000L;
    @Metadata(description="Sets the number of oldest entries to drop from the file store when the maximum capacity is hit to reduce disk space to allow room for new entries.", defaultValue="1000")
    private long dropOldestFileStore = 1000L;

    public FileIdempotentRepository() {
    }

    public FileIdempotentRepository(File fileStore, Map<String, Object> cache) {
        this.fileStore = fileStore;
        this.cache = cache;
    }

    public static IdempotentRepository fileIdempotentRepository(File fileStore) {
        return FileIdempotentRepository.fileIdempotentRepository(fileStore, 1000);
    }

    public static IdempotentRepository fileIdempotentRepository(File fileStore, int cacheSize) {
        return FileIdempotentRepository.fileIdempotentRepository(fileStore, LRUCacheFactory.newLRUCache(cacheSize));
    }

    public static IdempotentRepository fileIdempotentRepository(File fileStore, int cacheSize, long maxFileStoreSize) {
        FileIdempotentRepository repository = new FileIdempotentRepository(fileStore, LRUCacheFactory.newLRUCache(cacheSize));
        repository.setMaxFileStoreSize(maxFileStoreSize);
        return repository;
    }

    public static IdempotentRepository fileIdempotentRepository(File store, Map<String, Object> cache) {
        return new FileIdempotentRepository(store, cache);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    @ManagedOperation(description="Adds the key to the store")
    public boolean add(String key) {
        this.cacheAndStoreLock.lock();
        try {
            if (this.cache.containsKey(key)) {
                boolean bl = false;
                return bl;
            }
            this.cache.put(key, key);
            boolean containsInFile = this.containsStore(key);
            if (containsInFile) {
                boolean bl = false;
                return bl;
            }
            this.appendToStore(key);
            if (this.maxFileStoreSize > 0L && this.fileStore.length() > this.maxFileStoreSize) {
                LOG.warn("Maximum capacity of file store: {} hit at {} bytes. Dropping {} oldest entries from the file store", new Object[]{this.fileStore, this.maxFileStoreSize, this.dropOldestFileStore});
                this.trunkStore();
            }
            boolean bl = true;
            return bl;
        }
        finally {
            this.cacheAndStoreLock.unlock();
        }
    }

    @Override
    @ManagedOperation(description="Does the store contain the given key")
    public boolean contains(String key) {
        this.cacheAndStoreLock.lock();
        try {
            boolean bl = this.cache.containsKey(key) || this.containsStore(key);
            return bl;
        }
        finally {
            this.cacheAndStoreLock.unlock();
        }
    }

    @Override
    @ManagedOperation(description="Remove the key from the store")
    public boolean remove(String key) {
        boolean answer;
        this.cacheAndStoreLock.lock();
        try {
            answer = this.cache.remove(key) != null;
            this.removeFromStore(key);
        }
        finally {
            this.cacheAndStoreLock.unlock();
        }
        return answer;
    }

    @Override
    public boolean confirm(String key) {
        return true;
    }

    @Override
    @ManagedOperation(description="Clear the store (danger this removes all entries)")
    public void clear() {
        this.cacheAndStoreLock.lock();
        try {
            this.cache.clear();
            Map<String, Object> map = this.cache;
            if (map instanceof LRUCache) {
                LRUCache lruCache = (LRUCache)map;
                lruCache.cleanUp();
            }
            this.clearStore();
        }
        finally {
            this.cacheAndStoreLock.unlock();
        }
    }

    public File getFileStore() {
        return this.fileStore;
    }

    public void setFileStore(File fileStore) {
        this.fileStore = fileStore;
    }

    @ManagedAttribute(description="The file path for the store")
    public String getFilePath() {
        return this.fileStore.getPath();
    }

    public Map<String, Object> getCache() {
        return this.cache;
    }

    public void setCache(Map<String, Object> cache) {
        this.cache = cache;
    }

    @ManagedAttribute(description="The maximum file size for the file store in bytes")
    public long getMaxFileStoreSize() {
        return this.maxFileStoreSize;
    }

    @ManagedAttribute(description="The maximum file size for the file store in bytes")
    public void setMaxFileStoreSize(long maxFileStoreSize) {
        this.maxFileStoreSize = maxFileStoreSize;
    }

    public long getDropOldestFileStore() {
        return this.dropOldestFileStore;
    }

    @ManagedAttribute(description="Number of oldest elements to drop from file store if maximum file size reached")
    public void setDropOldestFileStore(long dropOldestFileStore) {
        this.dropOldestFileStore = dropOldestFileStore;
    }

    @ManagedAttribute(description="The current 1st-level cache size (elements in cache)")
    public int getCacheSize() {
        if (this.cache != null) {
            return this.cache.size();
        }
        return 0;
    }

    public void setCacheSize(int size2) {
        this.cacheSize = size2;
    }

    @ManagedAttribute(description="The 1st-level maximum cache size")
    public int getMaxCacheSize() {
        return this.cacheSize;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @ManagedOperation(description="Reset and reloads the file store")
    public void reset() throws IOException {
        this.lock.lock();
        try {
            this.cacheAndStoreLock.lock();
            try {
                Map<String, Object> map = this.cache;
                if (map instanceof LRUCache) {
                    LRUCache lruCache = (LRUCache)map;
                    lruCache.cleanUp();
                }
                this.cache.clear();
                this.loadStore();
            }
            finally {
                this.cacheAndStoreLock.unlock();
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    protected boolean containsStore(String key) {
        if (this.fileStore == null) return false;
        if (!this.fileStore.exists()) {
            return false;
        }
        try (Scanner scanner = new Scanner(this.fileStore, null, STORE_DELIMITER);){
            String line;
            do {
                if (!scanner.hasNext()) return false;
            } while (!(line = scanner.next()).equals(key));
            boolean bl = true;
            return bl;
        }
        catch (IOException e) {
            throw RuntimeCamelException.wrapRuntimeCamelException(e);
        }
    }

    protected void appendToStore(String key) {
        LOG.debug("Appending: {} to idempotent filestore: {}", (Object)key, (Object)this.fileStore);
        FileOutputStream fos = null;
        try {
            File storeParentDirectory = this.fileStore.getParentFile();
            if (storeParentDirectory != null && !storeParentDirectory.exists()) {
                LOG.info("Parent directory of file store {} doesn't exist. Creating.", (Object)this.fileStore);
                if (this.fileStore.getParentFile().mkdirs()) {
                    LOG.info("Parent directory of filestore: {} successfully created.", (Object)this.fileStore);
                } else {
                    LOG.warn("Parent directory of filestore: {} cannot be created.", (Object)this.fileStore);
                }
            }
            if (!this.fileStore.exists()) {
                FileUtil.createNewFile(this.fileStore);
            }
            fos = new FileOutputStream(this.fileStore, true);
            fos.write(key.getBytes());
            fos.write(STORE_DELIMITER.getBytes());
        }
        catch (IOException e) {
            try {
                throw RuntimeCamelException.wrapRuntimeCamelException(e);
            }
            catch (Throwable throwable) {
                IOHelper.close(fos, "Appending to file idempotent repository", LOG);
                throw throwable;
            }
        }
        IOHelper.close((Closeable)fos, "Appending to file idempotent repository", LOG);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void removeFromStore(String key) {
        block19: {
            this.lock.lock();
            try {
                LOG.debug("Removing: {} from idempotent filestore: {}", (Object)key, (Object)this.fileStore);
                ArrayList<String> lines = new ArrayList<String>();
                boolean found = false;
                try (Scanner scanner = new Scanner(this.fileStore, null, STORE_DELIMITER);){
                    while (scanner.hasNext()) {
                        String line = scanner.next();
                        if (key.equals(line)) {
                            found = true;
                            continue;
                        }
                        lines.add(line);
                    }
                }
                catch (IOException e) {
                    throw RuntimeCamelException.wrapRuntimeCamelException(e);
                }
                if (!found) break block19;
                LOG.debug("Rewriting idempotent filestore: {} due to key: {} removed", (Object)this.fileStore, (Object)key);
                FileOutputStream fos = null;
                try {
                    fos = new FileOutputStream(this.fileStore);
                    for (String line : lines) {
                        fos.write(line.getBytes());
                        fos.write(STORE_DELIMITER.getBytes());
                    }
                }
                catch (IOException e) {
                    throw RuntimeCamelException.wrapRuntimeCamelException(e);
                }
                finally {
                    IOHelper.close((Closeable)fos, "Rewriting file idempotent repository", LOG);
                }
            }
            finally {
                this.lock.unlock();
            }
        }
    }

    protected void clearStore() {
        try {
            FileUtil.deleteFile(this.fileStore);
            FileUtil.createNewFile(this.fileStore);
        }
        catch (IOException e) {
            throw RuntimeCamelException.wrapRuntimeCamelException(e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void trunkStore() {
        block21: {
            this.lock.lock();
            try {
                if (this.fileStore == null || !this.fileStore.exists()) {
                    return;
                }
                LOG.debug("Trunking: {} oldest entries from idempotent filestore: {}", (Object)this.dropOldestFileStore, (Object)this.fileStore);
                ArrayList<String> lines = new ArrayList<String>();
                int count = 0;
                try (Scanner scanner = new Scanner(this.fileStore, null, STORE_DELIMITER);){
                    while (scanner.hasNext()) {
                        String line = scanner.next();
                        if ((long)(++count) <= this.dropOldestFileStore) continue;
                        lines.add(line);
                    }
                }
                catch (IOException e) {
                    throw RuntimeCamelException.wrapRuntimeCamelException(e);
                }
                if (!lines.isEmpty()) {
                    LOG.debug("Rewriting idempotent filestore: {} with {} entries:", (Object)this.fileStore, (Object)lines.size());
                    FileOutputStream fos = null;
                    try {
                        fos = new FileOutputStream(this.fileStore);
                        for (String line : lines) {
                            fos.write(line.getBytes());
                            fos.write(STORE_DELIMITER.getBytes());
                        }
                        break block21;
                    }
                    catch (IOException e) {
                        throw RuntimeCamelException.wrapRuntimeCamelException(e);
                    }
                    finally {
                        IOHelper.close((Closeable)fos, "Rewriting file idempotent repository", LOG);
                    }
                }
                LOG.debug("Clearing idempotent filestore: {}", (Object)this.fileStore);
                this.clearStore();
            }
            finally {
                this.lock.unlock();
            }
        }
    }

    protected void cleanup() {
        Map<String, Object> map = this.cache;
        if (map instanceof LRUCache) {
            LRUCache lruCache = (LRUCache)map;
            lruCache.cleanUp();
        }
    }

    protected void loadStore() throws IOException {
        if (!this.fileStore.exists()) {
            boolean created;
            boolean mkdirsResult;
            LOG.debug("Creating filestore: {}", (Object)this.fileStore);
            File parent = this.fileStore.getParentFile();
            if (parent != null && !parent.exists() && !(mkdirsResult = parent.mkdirs())) {
                LOG.warn("Cannot create the filestore directory at: {}", (Object)parent);
            }
            if (!(created = FileUtil.createNewFile(this.fileStore))) {
                throw new IOException("Cannot create filestore: " + this.fileStore);
            }
        }
        LOG.trace("Loading to 1st level cache from idempotent filestore: {}", (Object)this.fileStore);
        this.cache.clear();
        try (Scanner scanner = new Scanner(this.fileStore, null, STORE_DELIMITER);){
            while (scanner.hasNext()) {
                String line = scanner.next();
                this.cache.put(line, line);
            }
        }
        catch (IOException e) {
            throw RuntimeCamelException.wrapRuntimeCamelException(e);
        }
        LOG.debug("Loaded {} to the 1st level cache from idempotent filestore: {}", (Object)this.cache.size(), (Object)this.fileStore);
    }

    @Override
    protected void doStart() throws Exception {
        ObjectHelper.notNull(this.fileStore, "fileStore", this);
        if (this.cache == null) {
            this.cache = LRUCacheFactory.newLRUCache(this.cacheSize <= 0 ? 1000 : this.cacheSize);
        }
        if (this.init.compareAndSet(false, true)) {
            this.loadStore();
        }
    }

    @Override
    protected void doStop() throws Exception {
        Map<String, Object> map = this.cache;
        if (map instanceof LRUCache) {
            LRUCache lruCache = (LRUCache)map;
            lruCache.cleanUp();
        }
        this.cache.clear();
        this.init.set(false);
    }
}

