/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.data.redis.core;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.aop.RawTargetAccess;
import org.springframework.aop.framework.ProxyFactory;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.core.RedisCommand;
import org.springframework.lang.Nullable;
import org.springframework.transaction.support.ResourceHolderSupport;
import org.springframework.transaction.support.TransactionSynchronization;
import org.springframework.transaction.support.TransactionSynchronizationManager;
import org.springframework.util.Assert;
import org.springframework.util.ReflectionUtils;

public abstract class RedisConnectionUtils {
    private static final Log log = LogFactory.getLog(RedisConnectionUtils.class);

    public static RedisConnection bindConnection(RedisConnectionFactory factory) {
        return RedisConnectionUtils.doGetConnection(factory, true, true, false);
    }

    public static RedisConnection bindConnection(RedisConnectionFactory factory, boolean transactionSupport) {
        return RedisConnectionUtils.doGetConnection(factory, true, true, transactionSupport);
    }

    public static RedisConnection getConnection(RedisConnectionFactory factory) {
        return RedisConnectionUtils.getConnection(factory, false);
    }

    public static RedisConnection getConnection(RedisConnectionFactory factory, boolean transactionSupport) {
        return RedisConnectionUtils.doGetConnection(factory, true, false, transactionSupport);
    }

    public static RedisConnection doGetConnection(RedisConnectionFactory factory, boolean allowCreate, boolean bind, boolean transactionSupport) {
        boolean bindSynchronization;
        Assert.notNull((Object)factory, "No RedisConnectionFactory specified");
        RedisConnectionHolder conHolder = (RedisConnectionHolder)TransactionSynchronizationManager.getResource(factory);
        if (conHolder != null && (conHolder.hasConnection() || conHolder.isSynchronizedWithTransaction())) {
            conHolder.requested();
            if (!conHolder.hasConnection()) {
                log.debug("Fetching resumed Redis Connection from RedisConnectionFactory");
                conHolder.setConnection(RedisConnectionUtils.fetchConnection(factory));
            }
            return conHolder.getRequiredConnection();
        }
        if (!allowCreate) {
            throw new IllegalArgumentException("No connection found and allowCreate = false");
        }
        log.debug("Fetching Redis Connection from RedisConnectionFactory");
        RedisConnection connection = RedisConnectionUtils.fetchConnection(factory);
        boolean bl = bindSynchronization = TransactionSynchronizationManager.isActualTransactionActive() && transactionSupport;
        if (bind || bindSynchronization) {
            if (bindSynchronization && RedisConnectionUtils.isActualNonReadonlyTransactionActive()) {
                connection = RedisConnectionUtils.createConnectionSplittingProxy(connection, factory);
            }
            try {
                RedisConnectionHolder holderToUse = conHolder;
                if (holderToUse == null) {
                    holderToUse = new RedisConnectionHolder(connection);
                } else {
                    holderToUse.setConnection(connection);
                }
                holderToUse.requested();
                if (bindSynchronization) {
                    RedisConnectionUtils.potentiallyRegisterTransactionSynchronisation(holderToUse, factory);
                }
                if (holderToUse != conHolder) {
                    TransactionSynchronizationManager.bindResource(factory, holderToUse);
                }
            }
            catch (RuntimeException ex) {
                RedisConnectionUtils.releaseConnection(connection, factory);
                throw ex;
            }
            return connection;
        }
        return connection;
    }

    private static RedisConnection fetchConnection(RedisConnectionFactory factory) {
        return factory.getConnection();
    }

    private static void potentiallyRegisterTransactionSynchronisation(RedisConnectionHolder connectionHolder, RedisConnectionFactory factory) {
        if (!connectionHolder.isTransactionActive()) {
            connectionHolder.setTransactionActive(true);
            connectionHolder.setSynchronizedWithTransaction(true);
            connectionHolder.requested();
            RedisConnection connection = connectionHolder.getRequiredConnection();
            boolean readOnly = TransactionSynchronizationManager.isCurrentTransactionReadOnly();
            if (!readOnly) {
                connection.multi();
            }
            TransactionSynchronizationManager.registerSynchronization(new RedisTransactionSynchronizer(connectionHolder, connection, factory, readOnly));
        }
    }

    private static boolean isActualNonReadonlyTransactionActive() {
        return TransactionSynchronizationManager.isActualTransactionActive() && !TransactionSynchronizationManager.isCurrentTransactionReadOnly();
    }

    private static RedisConnection createConnectionSplittingProxy(RedisConnection connection, RedisConnectionFactory factory) {
        ProxyFactory proxyFactory = new ProxyFactory(connection);
        proxyFactory.addAdvice(new ConnectionSplittingInterceptor(factory));
        proxyFactory.addInterface(RedisConnectionProxy.class);
        return (RedisConnection)RedisConnection.class.cast(proxyFactory.getProxy());
    }

    public static void releaseConnection(@Nullable RedisConnection conn, RedisConnectionFactory factory) {
        if (conn == null) {
            return;
        }
        RedisConnectionHolder conHolder = (RedisConnectionHolder)TransactionSynchronizationManager.getResource(factory);
        if (conHolder != null) {
            if (conHolder.isTransactionActive()) {
                if (RedisConnectionUtils.connectionEquals(conHolder, conn)) {
                    if (log.isDebugEnabled()) {
                        log.debug("RedisConnection will be closed when transaction finished");
                    }
                    conHolder.released();
                }
                return;
            }
            RedisConnectionUtils.unbindConnection(factory);
            return;
        }
        RedisConnectionUtils.doCloseConnection(conn);
    }

    private static boolean connectionEquals(RedisConnectionHolder connectionHolder, RedisConnection passedInConnetion) {
        if (!connectionHolder.hasConnection()) {
            return false;
        }
        RedisConnection heldConnection = connectionHolder.getRequiredConnection();
        return heldConnection.equals(passedInConnetion) || RedisConnectionUtils.getTargetConnection(heldConnection).equals(passedInConnetion);
    }

    private static RedisConnection getTargetConnection(RedisConnection connection) {
        RedisConnection connectionToUse = connection;
        while (connectionToUse instanceof RedisConnectionProxy) {
            RedisConnectionProxy proxy = (RedisConnectionProxy)connectionToUse;
            connectionToUse = proxy.getTargetConnection();
        }
        return connectionToUse;
    }

    public static void unbindConnection(RedisConnectionFactory factory) {
        RedisConnectionHolder connectionHolder = (RedisConnectionHolder)TransactionSynchronizationManager.getResource(factory);
        if (connectionHolder == null) {
            return;
        }
        if (log.isDebugEnabled()) {
            log.debug("Unbinding Redis Connection");
        }
        if (connectionHolder.isTransactionActive()) {
            if (log.isDebugEnabled()) {
                log.debug("Redis Connection will be closed when outer transaction finished");
            }
        } else {
            RedisConnection connection = connectionHolder.getConnection();
            connectionHolder.released();
            if (!connectionHolder.isOpen()) {
                TransactionSynchronizationManager.unbindResourceIfPossible(factory);
                RedisConnectionUtils.doCloseConnection(connection);
            }
        }
    }

    public static boolean isConnectionTransactional(RedisConnection connection, RedisConnectionFactory connectionFactory) {
        Assert.notNull((Object)connectionFactory, "No RedisConnectionFactory specified");
        RedisConnectionHolder connectionHolder = (RedisConnectionHolder)TransactionSynchronizationManager.getResource(connectionFactory);
        return connectionHolder != null && RedisConnectionUtils.connectionEquals(connectionHolder, connection);
    }

    private static void doCloseConnection(@Nullable RedisConnection connection) {
        if (connection == null) {
            return;
        }
        if (log.isDebugEnabled()) {
            log.debug("Closing Redis Connection");
        }
        try {
            connection.close();
        }
        catch (DataAccessException ex) {
            log.debug("Could not close Redis Connection", ex);
        }
        catch (Throwable ex) {
            log.debug("Unexpected exception on closing Redis Connection", ex);
        }
    }

    private static class RedisConnectionHolder
    extends ResourceHolderSupport {
        @Nullable
        private RedisConnection connection;
        private boolean transactionActive = false;

        public RedisConnectionHolder(RedisConnection connection) {
            this.connection = connection;
        }

        protected boolean hasConnection() {
            return this.connection != null;
        }

        @Nullable
        public RedisConnection getConnection() {
            return this.connection;
        }

        public RedisConnection getRequiredConnection() {
            RedisConnection connection = this.getConnection();
            if (connection == null) {
                throw new IllegalStateException("No active RedisConnection");
            }
            return connection;
        }

        protected void setConnection(@Nullable RedisConnection connection) {
            this.connection = connection;
        }

        protected void setTransactionActive(boolean transactionActive) {
            this.transactionActive = transactionActive;
        }

        protected boolean isTransactionActive() {
            return this.transactionActive;
        }

        @Override
        public void released() {
            super.released();
            if (!this.isOpen()) {
                this.setConnection(null);
            }
        }

        @Override
        public void clear() {
            super.clear();
            this.transactionActive = false;
        }
    }

    private static class RedisTransactionSynchronizer
    implements TransactionSynchronization {
        private final RedisConnectionHolder connectionHolder;
        private final RedisConnection connection;
        private final RedisConnectionFactory factory;
        private final boolean readOnly;

        RedisTransactionSynchronizer(RedisConnectionHolder connectionHolder, RedisConnection connection, RedisConnectionFactory factory, boolean readOnly) {
            this.connectionHolder = connectionHolder;
            this.connection = connection;
            this.factory = factory;
            this.readOnly = readOnly;
        }

        /*
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        @Override
        public void afterCompletion(int status) {
            try {
                if (this.readOnly) return;
                switch (status) {
                    case 0: {
                        this.connection.exec();
                        return;
                    }
                    case 1: 
                    case 2: {
                        this.connection.discard();
                        return;
                    }
                }
                return;
            }
            finally {
                if (log.isDebugEnabled()) {
                    log.debug("Closing bound connection after transaction completed with " + status);
                }
                this.connectionHolder.setTransactionActive(false);
                RedisConnectionUtils.doCloseConnection(this.connection);
                TransactionSynchronizationManager.unbindResource(this.factory);
                this.connectionHolder.reset();
            }
        }
    }

    static class ConnectionSplittingInterceptor
    implements MethodInterceptor {
        private final RedisConnectionFactory factory;
        @Nullable
        private final Method commandInterfaceMethod;

        public ConnectionSplittingInterceptor(RedisConnectionFactory factory) {
            this.factory = factory;
            this.commandInterfaceMethod = null;
        }

        private ConnectionSplittingInterceptor(RedisConnectionFactory factory, Method commandInterfaceMethod) {
            this.factory = factory;
            this.commandInterfaceMethod = commandInterfaceMethod;
        }

        @Override
        public Object invoke(MethodInvocation invocation) throws Throwable {
            return this.intercept(invocation.getThis(), invocation.getMethod(), invocation.getArguments());
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public Object intercept(Object obj, Method method, Object[] args) throws Throwable {
            if (method.getName().equals("getTargetConnection")) {
                return obj;
            }
            Class<?> returnType = method.getReturnType();
            String returnTypeName = returnType.getSimpleName();
            if (returnType.isInterface() && returnType.getPackageName().equals("org.springframework.data.redis.connection") && returnTypeName.startsWith("Redis") && returnTypeName.endsWith("Commands")) {
                ProxyFactory proxyFactory = new ProxyFactory(ReflectionUtils.invokeMethod(method, obj));
                proxyFactory.addAdvice(new ConnectionSplittingInterceptor(this.factory, method));
                proxyFactory.addInterface(RedisConnectionProxy.class);
                proxyFactory.addInterface(returnType);
                return proxyFactory.getProxy();
            }
            RedisCommand commandToExecute = RedisCommand.failsafeCommandLookup(method.getName());
            if (this.isPotentiallyThreadBoundCommand(commandToExecute)) {
                if (log.isDebugEnabled()) {
                    log.debug(String.format("Invoke '%s' on bound connection", method.getName()));
                }
                return this.invoke(method, obj, args);
            }
            if (log.isDebugEnabled()) {
                log.debug(String.format("Invoke '%s' on unbound connection", method.getName()));
            }
            RedisConnection connection = this.factory.getConnection();
            Object target = connection;
            try {
                if (this.commandInterfaceMethod != null) {
                    target = ReflectionUtils.invokeMethod(this.commandInterfaceMethod, connection);
                }
                Object object = this.invoke(method, target, args);
                return object;
            }
            finally {
                if (!connection.isClosed()) {
                    RedisConnectionUtils.doCloseConnection(connection);
                }
            }
        }

        private Object invoke(Method method, Object target, Object[] args) throws Throwable {
            try {
                return method.invoke(target, args);
            }
            catch (InvocationTargetException ex) {
                throw ex.getCause();
            }
        }

        private boolean isPotentiallyThreadBoundCommand(RedisCommand command) {
            return RedisCommand.UNKNOWN.equals((Object)command) || !command.isReadonly();
        }
    }

    public static interface RedisConnectionProxy
    extends RedisConnection,
    RawTargetAccess {
        public RedisConnection getTargetConnection();
    }
}

