/*
 * Copyright (C) 2006-2024 Talend Inc. - www.talend.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 */
package com.talend.excel.xssf.event;

import org.apache.poi.ooxml.util.PackageHelper;
import org.apache.poi.openxml4j.opc.OPCPackage;
import org.apache.poi.poifs.crypt.Decryptor;
import org.apache.poi.poifs.crypt.EncryptionInfo;
import org.apache.poi.poifs.filesystem.POIFSFileSystem;
import org.apache.poi.ss.usermodel.DataFormatter;
import org.apache.poi.xssf.eventusermodel.ReadOnlySharedStringsTable;
import org.apache.poi.xssf.eventusermodel.XSSFReader;
import org.apache.poi.xssf.model.StylesTable;
import org.xml.sax.ContentHandler;
import org.xml.sax.InputSource;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.XMLReaderFactory;

import org.openxmlformats.schemas.spreadsheetml.x2006.main.CTWorkbook;
import org.openxmlformats.schemas.spreadsheetml.x2006.main.CTWorkbookPr;
import org.openxmlformats.schemas.spreadsheetml.x2006.main.WorkbookDocument;

import java.io.File;
import java.io.InputStream;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.FutureTask;

/**
 * created by wwang on 2012-9-27 Detailled comment
 */
public class ExcelReader implements Callable {

    private DataBufferCache cache;

    private Thread task;

    private FutureTask futureTask;

    private String fileURL = null;

    private String charset = "UTF-8";

    private String password = null;

    private java.io.InputStream is;

    private List<String> sheetNames = new ArrayList<>();

    private List<Integer> sheetPositions = new ArrayList<>();

    private List<Boolean> asRegexs = new ArrayList<>();

    private final Map<Integer, DateFormat> columnDateFormats = new HashMap<>();

    DefaultTalendSheetContentsHandler sheetContentsHandler = null;

    private boolean includePhoneticRuns;

    boolean isDate1904 = false;

    public ExcelReader() {
        cache = DataBufferCache.getInstance();
        futureTask = new FutureTask(this);
        task = new Thread(futureTask);
    }

    public void parse(String fileURL, String charset, String password) {
        this.fileURL = fileURL;
        this.charset = charset;
        this.password = password;
        task.start();
    }

    public void parse(java.io.InputStream is, String charset, String password) {
        this.is = is;
        this.charset = charset;
        this.password = password;
        task.start();
    }

    public boolean hasNext() {
        return cache.hasData();
    }

    public SheetRow next() {
        return cache.readData();
    }

    public void addSheetName(String name, Boolean asRegex) {
        this.sheetNames.add(name);
        this.asRegexs.add(asRegex);
    }

    public void addSheetName(int position, Boolean asRegex) {
        // Keep "asRegex" here for code generated
        if (!this.sheetPositions.contains(position)) {
            this.sheetPositions.add(position);
        }
    }

    public void addDateFormat(Integer columnIndex, DateFormat dateFormat) {
        this.columnDateFormats.put(columnIndex, dateFormat);
    }

    public void stopRead() {
        sheetContentsHandler.stop();
    }

    public boolean isIncludePhoneticRuns() {
        return includePhoneticRuns;
    }

    public void setIncludePhoneticRuns(boolean includePhoneticRuns) {
        this.includePhoneticRuns = includePhoneticRuns;
    }

    /**
     * handle the exception in task. FutureTask.get() is a block method waiting for the Task over and it can throw the
     * exception in Task(Callable,Thread,Runnable)
     *
     * @throws Exception
     */
    public void handleException() throws Exception {
        futureTask.get();
    }

    public boolean isDate1904() {
        return isDate1904;
    }

    public Object call() throws Exception {

        OPCPackage pkg = null;
        POIFSFileSystem fs = null;
        try {
            if (password != null) {
                if (fileURL != null) {
                    fs = new POIFSFileSystem(new File(fileURL));
                } else {
                    fs = new POIFSFileSystem(is);
                }
                Decryptor d = Decryptor.getInstance(new EncryptionInfo(fs));
                if (!d.verifyPassword(password)) {
                    throw new RuntimeException("Error: Cannot decrypt Excel file. Invalid password.");
                }
                pkg = OPCPackage.open(d.getDataStream(fs));
            } else {
                if (fileURL != null) {
                    pkg = OPCPackage.open(fileURL);
                } else {
                    pkg = PackageHelper.open(is);
                }
            }
            XSSFReader r = new XSSFReader(pkg);
            InputStream workbookXml = r.getWorkbookData();
            WorkbookDocument doc = WorkbookDocument.Factory.parse(workbookXml);
            CTWorkbook wb = doc.getWorkbook();
            CTWorkbookPr prefix = wb.getWorkbookPr();
            // TDI-50218 Only for NPE about some customer generated Excel which have WorkbookPr missing
            if (prefix != null) {
                isDate1904 = prefix.getDate1904();
            }
            StylesTable styles = r.getStylesTable();
            ReadOnlySharedStringsTable strings = new ReadOnlySharedStringsTable(pkg, includePhoneticRuns);
            sheetContentsHandler = new DefaultTalendSheetContentsHandler(cache);
            DataFormatter formatter = new DataFormatter();
            boolean formulasNotResults = false;

            XMLReader parser = XMLReaderFactory.createXMLReader();
            parser.setFeature("http://apache.org/xml/features/disallow-doctype-decl", true);
            // This may not be strictly required as DTDs shouldn't be allowed at all, per previous line.
            parser.setFeature("http://apache.org/xml/features/nonvalidating/load-external-dtd", false);
            parser.setFeature("http://xml.org/sax/features/external-general-entities", false);
            parser.setFeature("http://xml.org/sax/features/external-parameter-entities", false);
            ContentHandler handler = new TalendXSSFSheetXMLHandler(styles, strings, sheetContentsHandler, formatter,
                    formulasNotResults, columnDateFormats);
            parser.setContentHandler(handler);

            XSSFReader.SheetIterator sheets = (XSSFReader.SheetIterator) r.getSheetsData();
            // List<InputStream> iss = new ArrayList<InputStream>();
            LinkedHashMap<String, InputStream> issmap = new LinkedHashMap<>();
            List<String> allSheetNames = new ArrayList<>();
            int sheetPosition = 0;
            while (sheets.hasNext()) {
                InputStream sheet = sheets.next();
                String sheetName = sheets.getSheetName();
                if (allSheetNames.contains(sheetName)) {
                    sheet.close();
                    continue;
                } else {
                    allSheetNames.add(sheetName);
                }
                boolean match = false;
                if (!sheetNames.isEmpty()) {
                    for (int i = 0; i < sheetNames.size(); i++) {
                        if ((asRegexs.get(i) && sheetName.matches(sheetNames.get(i)))
                                || (!asRegexs.get(i) && sheetName.equalsIgnoreCase(sheetNames.get(i)))) {
                            match = true;
                            // iss.add(sheet);
                            issmap.put(sheetName, sheet);
                            break;
                        }
                    }
                }

                if (!sheetPositions.isEmpty()) {
                    if (sheetPositions.contains(sheetPosition)) {
                        match = true;
                        issmap.put(sheetName, sheet);
                        // remove from the list for index out of range check.
                        sheetPositions.remove(sheetPositions.indexOf(sheetPosition));
                    }
                }
                sheetPosition++;
                if (!match) {
                    sheet.close();
                }
            }

            if (issmap.isEmpty()) {
                throw new RuntimeException("No match sheets");
            }

            // if the "sheetPositions" still not empty, means those positions out of range.
            if (!sheetPositions.isEmpty()) {
                throw new IllegalArgumentException(
                        "Sheet index " + sheetPositions + " is out of range (0.." + (sheetPosition - 1) + ")");
            }

            for (Map.Entry<String, InputStream> entry : issmap.entrySet()) {
                try (InputStream sheetStream = entry.getValue()) {
                    sheetContentsHandler.setCurrentSheetName(entry.getKey());
                    InputSource sheetSource = new InputSource(sheetStream);
                    sheetSource.setEncoding(charset);
                    parser.parse(sheetSource);
                }
            }
        } finally {
            if (pkg != null) {
                pkg.revert();
            }
            if (fs != null) {
                fs.close();
            }
            cache.notifyErrorOccurred();
        }
        return null;
    }
}
